
// Thu Feb 14 16:58:28 GMT 2002  by  nts

////////////////////////////////////////////////////////////////////////////////////
//                                                                                //
// class XMLRegionHandler : public DefaultHandler                                 //
//                                                                                //
// This class defines an interface to XML Region data as defined by the XML       //
// Schema namespace "http://www.cvg.cs.reading.ac.uk/ADVISOR" (the current name). //
//                                                                                //
// Similar in design to our ImageSource classes, some methods are pure virtual    //
// and the class should therefore not be instantiated directly.                   //
//                                                                                //
// The XMLRegionHandler class is derived from the XML SAX2 DefaultHandler class   //
// which is designed to ignore all requests.  We only redefine the methods that   //
// we need such that there is little overhead.                                    //
//                                                                                //
////////////////////////////////////////////////////////////////////////////////////

#include "XMLRegionHandler.h"

#include <util/XMLString.hpp>  // XMLString class with static character conversion methods
#include <sax2/Attributes.hpp>

#include "Region.h"
#include "RegionSet.h"
#include "text_output.h"

namespace ReadingPeopleTracker
{

//////////////////////////////////////////////////////////////////////////
//   Constructor and Destructor                                         //
//////////////////////////////////////////////////////////////////////////

XMLRegionHandler::XMLRegionHandler()
{
    // initialise variables to sensible values...
    frame_count = 0;
    parsing_level = 0;
    new_region = NULL;
    current_region_set = NULL;
    have_valid_region_set = false;
    region_set_has_been_queried = true;
}

XMLRegionHandler::~XMLRegionHandler()
{
    // nothing
}


//////////////////////////////////////////////////////////////////////////
//   Access function to query current region set                        //
//////////////////////////////////////////////////////////////////////////

RegionSet *XMLRegionHandler::get_current()
{
    // check whether the current_region_set is valid
    if (have_valid_region_set)
    {
	// remember that RegionSet has been queried
	region_set_has_been_queried = true;
	
	// return current complete set
	return current_region_set;
    }
    
    // we have no complete RegionSet yet!
    //   give an error message and return an empty new RegionSet:
    cerror << " XMLRegionHandler::get_current() : RegionSet queried before it was complete "
	   << endl;
    
    return new RegionSet;
}


//////////////////////////////////////////////////////////////////////////
//  Implementation of the SAX2 DocumentHandler interface                //
//////////////////////////////////////////////////////////////////////////

void XMLRegionHandler::startDocument()
{
     // initialise parsing level to 0 (toplevel)
     parsing_level = 0;
}

void XMLRegionHandler::endDocument()
{
    // signal that the RegionSet current_regions we have is now complete
    //   but is has not been queried using get_current() yet
    region_set_has_been_queried = false;
    have_valid_region_set = true;
    
    assert (parsing_level == 0);   // check for valid XML document structure
    
    // if the region set has not been collected via get_current() it will
    // have to be de-allocated. 
}

void XMLRegionHandler::startElement(const XMLCh *const uri,
				    const XMLCh *const localname,
				    const XMLCh *const qname,
				    const XERCES_CPP_NAMESPACE::Attributes &attributes)
{
    // There are currently 5 valid elements we recognise (level in brackets) :-
    //   MD_blob_frame (0), blob(1), info2d(2), info3d(2), occlusion(2).
    //   All data (coordinates etc) is given in attributes.
    //   see http://www.cvg.cs.reading.ac.uk/~nts/ADVISOR/motion_detector.xsd
    char *element_name = XMLString::transcode(localname);
    
    if (strcasecmp(element_name, "md_blob_frame") == 0)  // matched this element name?
    {
	// call specific function
	handle_frame(&attributes);
	
	// we have gone one level further into the document
	parsing_level++;  
	return;
    }
    
    if (strcasecmp(element_name, "blob") == 0)  // matched this element name?
    {
	// call specific function
	handle_blob(&attributes);
	
	// we have gone one level further into the document
	parsing_level++;  
	return;
    }
    
    if (strcasecmp(element_name, "info2d") == 0)  // matched this element name?
    {
	// call specific function
	handle_info2d(&attributes);
	
	// we have gone one level further into the document
	parsing_level++;  
	return;
    }
    
    if (strcasecmp(element_name, "info3d") == 0)  // matched this element name?
    {
	// call specific function
	handle_info3d(&attributes);
	
	// we have gone one level further into the document
	parsing_level++;  
	return;
    }
    
    if (strcasecmp(element_name, "occlusion") == 0)  // matched this element name?
    {
	// call specific function
	handle_occlusion(&attributes);
	
	// we have gone one level further into the document
	parsing_level++;  
	return;
    }

    // if we end up here, the element name was not matched
    //   give an error message and ignore the data
    cerror << " XMLRegionHandler::startElement(): Unknown XML element "
	   << element_name
	   << "  while parsing XML region data from MotionDetector "
	   << endl;
    
    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
    delete [] element_name;  
}


void XMLRegionHandler::handle_frame(const XERCES_CPP_NAMESPACE::Attributes *const attributes)
{
    // first, check parsing level (do not accept this element on any level!)
    if (parsing_level != 0)
    {
	// wrong level: give error message and ignore element
	cerror << " XMLRegionHandler::handle_frame(): Wrong element at this parsing level ("
	       << parsing_level << ") . "
	       << endl;
	return;
    }
    
    // de-allocate the previous RegionSet if we have one and it it has not been queried
    if ((region_set_has_been_queried == false) &&
	(have_valid_region_set) && (current_region_set != NULL))
	delete current_region_set;   // FIXME: check concurrency (should not be a problem now)
    
    // signal that we have no complete new RegionSet yet
    have_valid_region_set = false;
    
    // create new (empty) RegionSet to put in XML regions into
    current_region_set = new RegionSet;
    
    // and remember that it has not been queried from us yet
    region_set_has_been_queried = false;

    // check and process attributes...
    unsigned int count;
    char *variable_name;
    char *value_string;
    
    for (count = 0; count < attributes->getLength(); count++)
    {
	// for each attribute, get the variable name and its value (from variable="value")
	variable_name = XMLString::transcode(attributes->getLocalName(count));
	value_string = XMLString::transcode(attributes->getValue(count));
	
	if (strcasecmp(variable_name,"id") == 0)
	{
	    current_region_set->xml_frame_id = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	
	// ignore other recognised attributes for now...
	if ((strcasecmp(variable_name,"time_year") == 0) ||
	    (strcasecmp(variable_name,"time_month") == 0) ||
	    (strcasecmp(variable_name,"time_day") == 0) ||
	    (strcasecmp(variable_name,"time_hour") == 0) ||
	    (strcasecmp(variable_name,"time_min") == 0) ||
	    (strcasecmp(variable_name,"time_sec") == 0) ||
	    (strcasecmp(variable_name,"time_ms") == 0) ||
	    (strcasecmp(variable_name,"camera_id") == 0) ||
	    (strcasecmp(variable_name,"pc_name") == 0) ||
	    (strcasecmp(variable_name,"camera_sector") == 0) ||
	    (strcasecmp(variable_name,"camera_area") == 0) ||
	    (strcasecmp(variable_name,"num_blobs") == 0))
	{
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	
	if ((strcasecmp(variable_name, "xmlns") == 0)  ||
	    (strcasecmp(variable_name, "xsi") == 0) ||
	    (strcasecmp(variable_name, "schemaLocation") == 0)) 
	{
	    // ignore XML Schema namespace things...
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
    
	// attribute not recognised.  Give a warning and ignore
	cerror << " XMLRegionHandler::handle_frame(): Unknown attribute "
	       << variable_name << " ignored. "
	       << endl;

	// de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	delete [] variable_name;
	delete [] value_string;	    
    }

    frame_count++;
}

void XMLRegionHandler::handle_blob(const XERCES_CPP_NAMESPACE::Attributes *const attributes)
{
    // first, check parsing level (do not accept this element on any level!)
    if (parsing_level != 1)
    {
	// wrong level: give error message and ignore element
	cerror << " XMLRegionHandler::handle_blob(): Wrong element at this parsing level ("
	       << parsing_level << ") . "
	       << endl;
	return;
    }
    
    unsigned int count;
    char *variable_name;
    char *value_string;

    // create new Region to put in values from XML region data
    new_region = new Region;
    new_region->xml_blob_data_available = true;  // assume we will put sensible data in it...

    for (count = 0; count < attributes->getLength(); count++)
    {
	// for each attribute, get the variable name and its value (from variable="value")
	variable_name = XMLString::transcode(attributes->getLocalName(count));
	value_string = XMLString::transcode(attributes->getValue(count));

	if (strcasecmp(variable_name,"id") == 0)
	{
	    new_region->xml_blob_data.id = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"type") == 0)
	{
	    new_region->xml_blob_data.type =
			(blob_type_t) atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	
	// attribute not recognised.  Give a warning and ignore
	cerror << " XMLRegionHandler::handle_blob(): Unknown attribute "
	       << variable_name << " . "
	       << endl;

	// de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	delete [] variable_name;
	delete [] value_string;
    }
}

void XMLRegionHandler::handle_info2d(const XERCES_CPP_NAMESPACE::Attributes *const attributes)
{
    // first, check parsing level (do not accept this element on any level!)
    if (parsing_level != 2)
    {
	// wrong level: give error message and ignore element
	cerror << " XMLRegionHandler::handle_info2d(): Wrong element at this parsing level ("
	       << parsing_level << ") . "
	       << endl;
	return;
    }
    
    unsigned int count;
    char *variable_name;
    char *value_string;

    for (count = 0; count < attributes->getLength(); count++)
    {
	// for each attribute, get the variable name and its value (from variable="value")
	variable_name = XMLString::transcode(attributes->getLocalName(count));
	value_string = XMLString::transcode(attributes->getValue(count));

	if (strcasecmp(variable_name,"xmin") == 0)
	{
	    new_region->xml_blob_data.xmin = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"xmax") == 0)
	{
	    new_region->xml_blob_data.xmax = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"ymin") == 0)
	{
	    new_region->xml_blob_data.ymin = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"ymax") == 0)
	{
	    new_region->xml_blob_data.ymax = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"xcog") == 0)
	{
	    new_region->xml_blob_data.xcog = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"ycog") == 0)
	{
	    new_region->xml_blob_data.ycog = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"num_pix") == 0)
	{
	    new_region->xml_blob_data.num_pix = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	
	// attribute not recognised.  Give a warning and ignore
	cerror << " XMLRegionHandler::handle_info2d(): Unknown attribute "
	       << variable_name << " . "
	       << endl;

	// de-allocate memory allocated by XMLString::transcode(const XMLCh *)
	delete [] variable_name;
	delete [] value_string;
    }
}

void XMLRegionHandler::handle_info3d(const XERCES_CPP_NAMESPACE::Attributes *const attributes)
{
    // first, check parsing level (do not accept this element on any level!)
    if (parsing_level != 2)
    {
	// wrong level: give error message and ignore element
	cerror << " XMLRegionHandler::handle_info3d(): Wrong element at this parsing level ("
	       << parsing_level << ") . "
	       << endl;
	return;
    }
    
    unsigned int count;
    char *variable_name;
    char *value_string;
    
    for (count = 0; count < attributes->getLength(); count++)
    {
	// for each attribute, get the variable name and its value (from variable="value")
	variable_name = XMLString::transcode(attributes->getLocalName(count));
	value_string = XMLString::transcode(attributes->getValue(count));
	
	if (strcasecmp(variable_name,"x") == 0)
	{
	    new_region->xml_blob_data.world_xpos = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"y") == 0)
	{
	    new_region->xml_blob_data.world_ypos = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"z") == 0)
	{
	    new_region->xml_blob_data.world_zpos = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"width") == 0)
	{
	    new_region->xml_blob_data.world_width = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"height") == 0)
	{
	    new_region->xml_blob_data.world_height = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	
	// attribute not recognised.  Give a warning and ignore
	cerror << " XMLRegionHandler::handle_info3d(): Unknown attribute "
	       << variable_name << " . "
	       << endl;
	
	// de-allocate memory allocated by XMLString::transcode(const XMLCh *)
	delete [] variable_name;
	delete [] value_string;
    }
}

void XMLRegionHandler::handle_occlusion(const XERCES_CPP_NAMESPACE::Attributes *const attributes)
{
    // first, check parsing level (do not accept this element on any level!)
    if (parsing_level != 2)
    {
	// wrong level: give error message and ignore element
	cerror << " XMLRegionHandler::handle_occlusion(): Wrong element at this parsing level ("
	       << parsing_level << ") . "
	       << endl;
	return;
    }
    
    unsigned int count;
    char *variable_name;
    char *value_string;
    
    for (count = 0; count < attributes->getLength(); count++)
    {
	// for each attribute, get the variable name and its value (from variable="value")
	variable_name = XMLString::transcode(attributes->getLocalName(count));
	value_string = XMLString::transcode(attributes->getValue(count));
	
	if (strcasecmp(variable_name,"left") == 0)
	{
	    new_region->xml_blob_data.occluded_left = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"right") == 0)
	{
	    new_region->xml_blob_data.occluded_right = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"top") == 0)
	{
	    new_region->xml_blob_data.occluded_top = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}
	if (strcasecmp(variable_name,"bottom") == 0)
	{
	    new_region->xml_blob_data.occluded_bottom = atoi(value_string);  // no check
	    // de-allocate memory allocated by char *XMLString::transcode(const XMLCh *)
	    delete [] variable_name;
	    delete [] value_string;
	    
	    continue;
	}

	// attribute not recognised.  Give a warning and ignore
	cerror << " XMLRegionHandler::handle_occlusion(): Unknown attribute "
	       << variable_name << " . "
	       << endl;
	
	// de-allocate memory allocated by XMLString::transcode(const XMLCh *)
	delete [] variable_name;
	delete [] value_string;
    }
}

void XMLRegionHandler::endElement(const XMLCh *const uri, 
				  const XMLCh *const localname, 
				  const XMLCh *const qname)
{
    // data is constructed by startElement.  We just adjust the parsing_level.
    assert (parsing_level > 0);  //  otherwise: non-open element ended???
    
    if (parsing_level == 2)
    {
	// end of blob: new_region should be complete at this point. 
	// Add it to current_region_set:
	current_region_set->add(new_region);

	// set new_region to NULL so that we do not access it any more
	new_region = NULL;
    }
    
    if (parsing_level > 0)
	parsing_level--;
}



// these are also available but not needed since all data is in attributes
// void XMLRegionHandler::characters(const XMLCh *const chars, const unsigned int length);
// void XMLRegionHandler::ignorableWhitespace (const XMLCh *const chars, const unsigned int length);
// void XMLRegionHandler::processingInstruction (const XMLCh *const target, const XMLCh *const data); 


//////////////////////////////////////////////////////////////////////////
//  Implementation of the SAX2 ErrorHandler interface                   //
//////////////////////////////////////////////////////////////////////////

void XMLRegionHandler::warning(const SAXParseException &exception)
{
    char *message = XMLString::transcode(exception.getMessage());
    
    // write warning message and ignore
    cerror << " XMLRegionHandler::warning: "
 	   << message
	   << " (ignored) "
	   << endl;

    delete [] message;
}

void XMLRegionHandler::error(const SAXParseException &exception)
{
    char *message = XMLString::transcode(exception.getMessage());

    // write error message and ignore
    cerror << " XMLRegionHandler::error: "
	   << message
	   << " (ignored) "
	   << endl;

    delete [] message;
}

void XMLRegionHandler::fatalError(const SAXParseException &exception)
{
    char *message = XMLString::transcode(exception.getMessage());
    
    // write error message and ignore  // FIXME: can we?
    cerror << " XMLRegionHandler::fatalError: "
	   << message
	   << " (ignored) "
	   << endl;

    delete [] message;
}

} // namespace ReadingPeopleTracker
