///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  BufferedSlaveImageSource.h  defining a buffered XMLRegionSource class    //
//                                                                           //
//  A XMLRegionSource which waits for external XML data blocks using         //
//    pthread_cond_wait.  The data given is buffered in a FIFO buffer and    //
//    can be queried using get_next().  For this, it will be parsed using    //
//    XMLRegionSource methods.  The get_next() method returns a RegionSet *  //
//    and deletes the XML buffer.                                            //
//                                                                           //
//  Additionally, the class can fill up the FIFO buffer from a               //
//    NumberedFileSource, either automatically in get_next() or using our    //
//    fill_buffer() method.                                                  //
//                                                                           //
//  XML Schema namespace "http://www.cvg.cs.reading.ac.uk/ADVISOR".          //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Wed Feb 20 18:11:45 GMT 2002                                 //
//  Revision  : 0.0 of Wed Feb 20 18:11:45 GMT 2002                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef __BUFFERED_SLAVE_XML_REGION_SOURCE_H__
#define __BUFFERED_SLAVE_XML_REGION_SOURCE_H__

#include <cassert>
#include <pthread.h>     // for IEEE POSIX 1003.1c --- 1995 threads
#include <climits>       // for UINT_MAX

// XML libraries for SAX2 API...
#include <util/XMLUniDefs.hpp>

// our own includes
#include "XMLRegionSource.h"
#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

// forward declarations
class RegionSet;
class TrackedObjectSet;
class NumberedFileSource;

// number of `RegionSet's to buffer before refusing external requests to
//   handle_new_blob_data(...).  Equals to maximum number of `RegionSet's used and
//   provided by this class at any one time.
#define BSXRS_DEFAULT_BUFFER_SIZE 10

////////////////////////////////////////////////////////////////////////////////////
//                                                                                //
// class BufferedSlaveXMLRegionSource : public XMLRegionSource                    //
//                                                                                //
// This class defines an interface to XML Region data as defined by the XML       //
// Schema namespace "http://www.cvg.cs.reading.ac.uk/ADVISOR" (the current name). //
//                                                                                //
// An instance of this class will read and buffer XML data given through          //
// handle_new_blob_data(...) until it is queried by get_next().  get_next() will  //
// parse the XML data, return a RegionSet * and delete the XML buffer.            //
//                                                                                //
////////////////////////////////////////////////////////////////////////////////////

class BufferedSlaveXMLRegionSource : public XMLRegionSource
{
private:

    //////////////////////////////////////////////////////////////////////////
    //   Private member data                                                //
    //////////////////////////////////////////////////////////////////////////

    unsigned int buffer_entries; // actual number of images in buffer, including current
    unsigned int buffer_size;    // maximum of the above (given during construction)
 
    // dimensions of associated images --- needed for converting coordinates
    unsigned int xdim;
    unsigned int ydim;
    
    // arrays [0..buffer_size-1] of data: 
    unsigned char **xml_data_buffer;       // buffered XML data buffers...
    unsigned int *xml_data_size_in_buffer; // ... and their respective sizes
    
    // our master xml data source, if given: to fill up the buffer, either
    //   using our fill_buffer method or automatically in get_next()
    NumberedFileSource *master_xml_data_source;

    //////////////////////////////////////////////////////////////////////////
    //   Pthread data: mutex and condition variables for xml_data_buffer    //
    //////////////////////////////////////////////////////////////////////////

    pthread_mutex_t xml_data_buffer_modification; // write access to xml_data_buffer
    pthread_cond_t new_xml_data_in_buffer;        // whether new XML data has arrived in buffer
    
public:

    //////////////////////////////////////////////////////////////////////////
    //   Constructor and Destructor                                         //
    //////////////////////////////////////////////////////////////////////////

    BufferedSlaveXMLRegionSource(unsigned int the_xdim, unsigned int the_ydim,
				 NumberedFileSource *the_master_xml_data_source = NULL,
				 unsigned int the_buffer_size = BSXRS_DEFAULT_BUFFER_SIZE);
    ~BufferedSlaveXMLRegionSource();

    //////////////////////////////////////////////////////////////////////////
    //   Method to call when a new XML blob data arrives.                   //
    //   It will be parsed and buffered or ignored (if the buffer is full). //
    //////////////////////////////////////////////////////////////////////////

    void handle_new_blob_data(unsigned char *the_data, unsigned int the_data_size);

    ///////////////////////////////////////////////////////////////////////////
    //   Method to fill up buffer from master_xml_data_source                //
    ///////////////////////////////////////////////////////////////////////////

    unsigned int fill_buffer(unsigned int number_of_blocks_to_add = UINT_MAX);
    
    //////////////////////////////////////////////////////////////////////////
    //   Interface to get RegionSet for current and next frame              //
    //////////////////////////////////////////////////////////////////////////
    
    virtual RegionSet *get_current()
	{
	    if (current_region_set != NULL)  // have valid RegionSet?
		return current_region_set;
	    else
		return get_next();
	}

    virtual RegionSet *get_next();

    //////////////////////////////////////////////////////////////////////////
    //   Interface to get frame id of current RegionSet                     //
    //////////////////////////////////////////////////////////////////////////

    virtual frame_id_t get_frame_id();

    //////////////////////////////////////////////////////////////////////////
    //   Interface to get [max] number of available `RegionSet's in buffer  //
    //////////////////////////////////////////////////////////////////////////

    inline unsigned int get_buffer_entries() const
	{
	    return buffer_entries;
	}
    
    inline unsigned int get_buffer_size() const
	{
	    return buffer_size;
	}

    //////////////////////////////////////////////////////////////////////////
    //   Interface to query the type of source this XMLRegionSource is      //
    //////////////////////////////////////////////////////////////////////////

    virtual XMLRegionSourceType get_source_type() const
	{
	    return XRST_SLAVE;
	}
};

} // namespace ReadingPeopleTracker

#endif
