// collection of defines, types and helpers (you guessed?) for The Tracker
//  created by nts on Tue Feb  5 14:05:55 GMT 2002
//  from `#define's, `typedef's etc collected from all around The Code 

#ifndef __TRACKER_DEFINES_TYPES_AND_HELPERS_H__
#define __TRACKER_DEFINES_TYPES_AND_HELPERS_H__

#include <sys/types.h>   // for types like u_int32_t and u_int64_t
#include <limits.h>      // for UINT_MAX

#include "os_specific_things.h"  // for Linux, Solaris, Windows etc things...
#include "realno.h"
#include "Point2.h"

namespace ReadingPeopleTracker
{

// type to use for object ids
typedef u_int32_t object_id_t;

// type to use for camera ids
typedef u_int32_t camera_id_t;

// type to use for frame ids and maximum value of that type
typedef u_int32_t frame_id_t;   // wraparound in 27.2385 years at 5 fps --- OK.
#define FRAME_ID_T_MAX UINT_MAX

// type to use for frame time (milliseconds since the Epoch (01/01/1970) UTC so use 64 bits!)
typedef u_int64_t frame_time_t;

// maximum frame_id before wraparound should occur (NB match frame_id_t type defined above)
#define MAX_FRAME_ID UINT_MAX

// maximum number of frames (ie, consecutive frame id differences) assumed
#define MAX_EXPECTED_FRAME_SKIP 25

// the default frame time in milliseconds per frame
const frame_time_t default_frame_time_in_ms = 200;    // that's 5 fps

// maxium expected size of XML (Region) data block in memory (for memory allocation)
#define MAX_XML_BUFFER_SIZE 16384

// for ActiveModel
#define MAX_BUNCH_SIZE 128
// (was once 500, changed to 128) --- nts.

// type of TrackedObject
typedef enum { PERSON , GROUP , CAR , UNKNOWN } ObjectType;

// type of "blob" (region) as provided by external (INRIA) motion detector via XML
typedef enum
{
    BT_UNKNOWN = 0,            /**< blob type unknown */
    BT_OCCL_OBJECT = 1,        /**< occluded type, ex: person occluded by a scene object */
    BT_CROWD = 2,              /**< blob contains many many persons */
    BT_PERSON_GROUP = 3,       /**< may be a person, maybe a group */
    BT_GROUP = 4,              /**< blob is a group */
    BT_PERSON = 5,             /**< blob is a person */   
    BT_CONTX_OBJECT = 6,       /**< blob is a context object, ex: the door */
    BT_VEHICLE = 7,            /**< blob is a vehicle, ex: metro, car */
    BT_NOISE = 8,              /**< noise */
    BT_VALIDATION = 9          /**< blob unknown found in zone of TBL, for RATP */
} blob_type_t;

// data from XML file used in XMLRegionHandler (stored in Region class)
typedef struct 
{
    // attributes to "blob" element
    object_id_t id;
    blob_type_t type;
    
    // info2d
    unsigned int xmin;
    unsigned int xmax;
    unsigned int ymin;
    unsigned int ymax;
    unsigned int xcog;
    unsigned int ycog;
    unsigned int num_pix;

    // info3d
    realno world_xpos;
    realno world_ypos;
    realno world_zpos;
    double world_width;
    double world_height;

    // occlusion
    bool occluded_left;
    bool occluded_right;
    bool occluded_top;
    bool occluded_bottom;   
} xml_blob_data_t;

// constants for images
const unsigned char MARK       = 0xff;
const unsigned char CLEAR_MARK = 0x00;
const long NULLWIN = 0;

// for Stack class: number of pointers to allocate in the Stack
#define DEFAULT_STACKSIZE 65536

// valid image types
typedef enum { BASE, GREY8, RGB32, HSV32} ImageType;

// valid image addressing modes, for storage im memory and for addressing by our modules
//   IA_TOP_TO_BOTTOM: line 0 is the top line,  IA_BOTTOM_TO_TOP: line 0 is the bottom line
typedef enum { IA_TOP_TO_BOTTOM, IA_BOTTOM_TO_TOP } ImageAddressing;

// valid storage types for MovieStore class
typedef enum { MOVIESTORE_JPEG, MOVIESTORE_PNM } MovieStoreFormat;

// PNM image file types: (invalid), 8-bit grey, 24-bit RGB
typedef enum
{
    PNM_INVALID = 0,
    PNM_GREY_COMPRESSED = 5,
    PNM_COLOUR_COMPRESSED = 6
} PnmFileType;

// a colour pixel: RGB32 colour space
struct RGB32pixel
{
public:
    
// nts: fix for byte ordering on x86:
#ifdef LSB_FIRST
    unsigned char red;
    unsigned char green;
    unsigned char blue;
    unsigned char alpha;
#else
#ifdef MSB_FIRST
    unsigned char alpha;
    unsigned char blue;
    unsigned char green;
    unsigned char red;
#else
#error "Please define LSB_FIRST or MSB_FIRST in generik.mk"
#endif  // ifdef MSB_FIRST
#endif  // ifdef LSB_FIRST
};

// a colour pixel: HSV32 colour space
struct HSV32pixel
{   
#ifdef LSB_FIRST
    unsigned char H;
    unsigned char S;
    unsigned char V;
    unsigned char alpha;
#else
#ifdef MSB_FIRST
    unsigned char alpha;
    unsigned char V;
    unsigned char S;
    unsigned char H;
#else
#error "Please define LSB_FIRST or MSB_FIRST in generik.mk"
#endif  // ifdef LSB_FIRST
#endif  // ifdef MSB_FIRST
};

// ImageSourceType: where does an ImageSource get the images from?
typedef enum
{
    IST_UNKNOWN,              // unknown
    IST_SINGLE_FILE,          // one single source file for all (multiple) frames, eg MPEG
    IST_NUMBERED_FILES,       // numbered set of files, one per frame, eg numbered JPEG/JFIF files
    IST_SLAVE,                // waiting (with pthread_cond_wait) for images provided elsewhere,
                              // eg by Thales Wrapper
    IST_STREAM                // input from a stream, eg a pipe (eg PnmStream class)
} ImageSourceType;


// Analogue: XMLRegionSourceType: where does an XMLRegionSource get the XML data from?
typedef enum
{
    XRST_UNKNOWN,              // unknown
    XRST_SINGLE_FILE,          // one single source file for all (multiple) frames --- ADVISOR.Testbed1
    XRST_NUMBERED_FILES,       // numbered set of files, one per frame, eg numbered XML files
    XRST_SLAVE,                // waiting (with pthread_cond_wait) for data provided elsewhere,
                               // eg by Thales Wrapper / from a NumberedFileSource --- ADVISOR.Testbed2
    XRST_STREAM                // input from a stream, eg a pipe --- not implemented yet
} XMLRegionSourceType;


// Thales use hard-coded image dimensions: quarter area PAL
#define DEFAULT_IMAGE_WIDTH 768
#define DEFAULT_IMAGE_HEIGHT 576


// these drawing colours, defined as {R,G,B} are from /usr/X11/lib/X11/rgb.txt
#define NUM_DRAW_COLOURS 13
static const unsigned char draw_colour_table[NUM_DRAW_COLOURS][3] =
{
    {255, 0, 255},    //   0: magenta
    {255, 255, 0},    //   1: yellow
    {255, 255, 255},  //   2: white
    {0, 255, 0},      //   3: green
    {160, 32, 240},   //   4: purple
    {0, 0, 255},      //   5: blue
    {255, 0, 0},      //   6: red
    {0, 255, 255},    //   7: cyan
    {131, 111, 255},  //   8: SlateBlue1
    {255, 215, 0},    //   9: gold
    {127, 255, 0},    //  10: chartreuse
    {255, 140, 0},    //  11: DarkOrange
    {191, 239, 255}   //  12: LightBlue1
};

// needed in RGB32Image: ColourFilteringMethod using RGB colour space
typedef enum
{
// Simple norm methods
    CF_METHOD_1_NORM,   //  1-norm of (R,G,B)
    CF_METHOD_2_NORM,   //  2-norm of (R,G,B)
    CF_METHOD_MAX_NORM, //  max-norm of (R,G,B)

// HSV after D Travis
    CF_METHOD_H,        //  H of HSV after D Travis
    CF_METHOD_S,        //  S of HSV after D Travis
    CF_METHOD_V,        //  V of HSV after D Travis
    
// Chrominance and luminance according to CIE recommendations
    CF_METHOD_Y_601,    // Y  from YCbCr after CIE recommendation 601
    CF_METHOD_CB_601,   // Cb from YCbCr after CIE recommendation 601
    CF_METHOD_CR_601,   // Cr from YCbCr after CIE recommendation 601
    CF_METHOD_Y_709     // Y  from YCbCr after CIE recommendation 709
} ColourFilteringMethod;



// EdgeDetector things...
typedef enum
{
    EDGE_OCCLUDED,
    EDGE_BAD,
    EDGE_GOOD
} edge_status_t;

typedef enum
{
    STATISTICAL_EDGE_SEARCH,
    MAXIMUM_EDGE_SEARCH,
    NEAREST_EDGE_SEARCH
} edge_search_method_t;


// Results status
typedef enum
{
    RESULTS_UNDEFINED,
    RESULTS_COMPLETE,
    RESULTS_PROCESSED
} results_status_t;


// vertical histrogram for HumanFeatureTracker
typedef struct      // histogram of pixels projected vertically
{
    // all of these measures are in pixels using region image coordinates
    unsigned int width;                // width of region (pixels of the image)
    unsigned int height;               // height of region (pixels of the image)
    unsigned int xlo;                  // position of region in video image
    unsigned int xhi;                  // position of region in video image
    unsigned int ylo;                  // position of region in video image
    unsigned int yhi;                  // position of region in video image

//    unsigned int average;              // average value ("height") of inner part
    unsigned int min_value;            // minimum value > 0
    unsigned int max_value;            // maximum value

    unsigned int leftmost;             // leftmost significantly non-empty y row.
    unsigned int rightmost;            // rightmost significantly non-empty y row.

    unsigned int peak_pos[10];         // position (y centre) of 0 to 10 peaks
    unsigned int peak_max_pos[10];     // position of maximum in these peaks
    unsigned int peak_max_value[10];   // value at this position
//    unsigned int mean_peak_value[10];  // mean value of these peaks
    unsigned int peak_width[10];       // width of peak region
    unsigned int peak_height[10];      // relative height of peak region
} VerticalHistogram;

typedef struct HeadInfo
{
    // all of these measures are in pixels using region image coordinates
    unsigned int x;        // position of centre (ie COG of bbox)
    unsigned int y;        // position of centre (ie COG of bbox)
    unsigned int x_abs;    // x, absolute image coordinates 
    unsigned int y_abs;    // y, absolute image coordinates 
    unsigned int width;    // width, height of bbox in pixels
    unsigned int height;

    VerticalHistogram *head_histogram;  // vertical histogram of upper body part

    HeadInfo &operator= (HeadInfo &original);
} HeadInfo;


// This is used in HSV32Image: ColourDistanceMethod using HSV colour space
typedef enum
{
    CD_METHOD_V_WEIGHTED_HS   // distance in H/S circle, weighted by V
} ColourDistanceMethod;

// Some useful constants.  From /usr/include/math.h.  Only a few of them are actually needed.
#define M_E            2.7182818284590452354   /* e */
#define M_LOG2E        1.4426950408889634074   /* log_2 e */
#define M_LOG10E       0.43429448190325182765  /* log_10 e */
#define M_LN2          0.69314718055994530942  /* log_e 2 */
#define M_LN10         2.30258509299404568402  /* log_e 10 */
#define M_PI           3.14159265358979323846  /* pi */
#define M_PI_2         1.57079632679489661923  /* pi/2 */
#define M_PI_4         0.78539816339744830962  /* pi/4 */
#define M_1_PI         0.31830988618379067154  /* 1/pi */
#define M_2_PI         0.63661977236758134308  /* 2/pi */
#define M_2_SQRTPI     1.12837916709551257390  /* 2/sqrt(pi) */
#define M_SQRT2        1.41421356237309504880  /* sqrt(2) */
#define M_SQRT1_2      0.70710678118654752440  /* 1/sqrt(2) */

// some simple inline math functions that had been `define's in the past...

#ifdef WIN32
// Windows does not have fast floating point fmax, fmin, copysign and hypot
#define fmax max
#define fmin min

template <class type>
inline type copysign(const type &a, const type &b)
{
    if ((a > 0) ^ (b > 0))
              return -a;
      else
              return a;
}

template <class type>
inline const type &hypot(const type &a, const type &b)
{
    return sqrt (a*a + b*b);
}

#endif


#ifndef HAVE_MIN_MAX
// ISO C99 defines these and does not like them re-defined...

// lower case versions of some
template <class type>
inline const type &min(const type &a, const type &b)
{
    if (a < b)
	return a;

    return b;
}

template <class type>
inline const type &max(const type &a, const type &b)
{
    if (a > b)
	return a;
    
    return b;
}

#endif  // ifndef HAVE_MIN_MAX_ABS

template <class type>
inline const type abs(const type &a)
{
    if (a < 0)
	return -a;
    
    return a;
}


// upper case versions
#ifdef MIN
#undef MIN
#endif
template <class type>
inline type MIN(const type &a, const type &b)
{
    if (a < b)
	return a;

    return b;
}

#ifdef MAX
#undef MAX
#endif
template <class type>
inline type MAX(const type &a, const type &b)
{
    if (a > b)
	return a;
    
    return b;
}

template <class type>
inline type SGN(const type &a)
{
    if (a < 0)
	return -1;
   
    return 1;
}

template <class type>
inline type ZSGN(const type &a)
{
    if (a == 0)
	return 0;
    
    if (a < 0)
	return -1;
    
    return 1;
}

#ifdef ABS
#undef ABS
#endif
template <class type>
inline type ABS(const type &a)
{
    if (a < 0)
	return -a;
    
    return a;
}

template <class type>
inline type SQUARE(const type &a)
{
    return a * a;
}

template <class type>
inline void SWAP(type &a, type &b)
{
    type temp = a;
    a = b;
    b = temp;
}

} // namespace ReadingPeopleTracker

#endif
