///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  NumberedFiles.cc   Helper (base) class to read or write numbered files.  //
//                                                                           //
//  Derive a class from this one.  Give first filename (eg, video017.jpg)    //
//    as parameter to this base class's contructor.                          //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Mon Apr  8 09:35:38 BST 2002                                 //
//  Revision  : 0.0 of Mon Apr  8 09:35:38 BST 2002                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "NumberedFiles.h"

#include <string.h>  // for strncpy() etc
#include <ctype.h>   // for isdigit()
#include <stdlib.h>  // for atol(...)

#include "text_output.h"  // for cerror

namespace ReadingPeopleTracker
{

static const char *NumberedFiles_Revision = "@(#) NumberedFiles.cc, rev 0.0 of Mon Apr  8 09:35:38 BST 2002, Author Nils T Siebel, Copyright (c) 2002 The University of Reading";

NumberedFiles::NumberedFiles(char *first_filename)
{
    // go through first_filename, separating out base, number, extension
    signed int pos;
    
    strncpy(current_filename, first_filename, 256+16+64-1);
    current_filename[256+16+64-1] = '\0';     // avoid overrun
    
    // set up file name(s)...
    for (pos = strlen(current_filename)-1; pos >= 0; pos--)
	if (isdigit(current_filename[pos]))
	    break;
    
    if (pos < 0)
    {
	cerror << "NumberedFiles::NumberedFiles: To use numbered files, please give first filename, like input017.jpg " << endl;
	exit(1);
    }
    
    strncpy(filename_ext, &current_filename[pos+1], 63);
    filename_ext[63] = (char) 0x00;
    current_filename[pos+1] = 0;  // for atol
    
    for ( ; pos >= 0; pos--)
	if (! (isdigit(current_filename[pos])))
	    break;
    
    if (! (filename_num_digits = strlen(&current_filename[pos+1])))
    {
	cerror << "NumberedFiles::NumberedFiles: To use numbered files, please give first filename, like input023.jpg\n" << endl;
	exit(1);
    }
    
    strncpy(filename_base, current_filename, MIN(pos+1, 255));
    filename_base[MIN(pos+1, 255)] = 0;
    start_frame_number = atol(&current_filename[pos+1]);  // set start frame number.

    file_count = 0;             // set file count to 0 (no files handled so far).
    current_file = NULL;        // signal no file

    // re-create a valid first current_filename
    strncpy(current_filename, first_filename, 256+16+64-1);
    current_filename[256+16+64-1] = '\0';     // avoid overrun
}


NumberedFiles::~NumberedFiles()
{
    // nothing
}


//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  NumberedFiles::get_next_filename: generate and return next filename     //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

char *NumberedFiles::get_next_filename()
{
    // generate file number
    if (file_count == 0) // first frame
	current_frame_number = start_frame_number;
    else 
	current_frame_number++;
    
    // generate file name
    sprintf(current_filename, "%s%0*i%s", filename_base,
	    filename_num_digits,
	    current_frame_number,
	    filename_ext);
    
    // increase file count
    file_count++;
   
    // return new current filename
    return current_filename;
}

} // namespace ReadingPeopleTracker
