///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  NumberedFileSource.cc  A class to read numbered files into memory.       //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Tue Apr  9 19:00:06 BST 2002                                 //
//  Revision  : 0.0 of Tue Apr  9 19:00:06 BST 2002                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <cassert>
#include <cstdio>   // for FILE type

#include "NumberedFileSource.h"

namespace ReadingPeopleTracker
{

static const char *NumberedFileSource_Revision = "@(#) NumberedFileSource.cc, rev 0.0 of Tue Apr  9 19:00:06 BST 2002, Author Nils T Siebel, Copyright (c) 2002 The University of Reading";


///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  NumberedFileSource::get_next: get and return next data block.            //
//                                                                           //
//  Call this method to read the next (also the first) file from disk.       //
//    The file contents will be put as a "data block" into newly allocated   //
//    memory.  The size of this data block can be queried using              //
//    get_current_size.                                                      //
//                                                                           //
//  NB the caller has to take care that the data block is de-allocated       //
//                                                                           //
//  Returns a pointer to the newly allocate data block, NULL if unavailable. //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

unsigned char *NumberedFileSource::get_next()
{
    // generate next filename (also works OK in case of the first file)
    get_next_filename();         // using NumberedFiles method
    
    // open file and determine its size
    current_file = fopen (get_current_filename(), "rb");  // `b' may be important on non-UNIX

    // if the file does not exist, simply return NULL
    if (current_file == NULL)
	return NULL;   // signal no more data

    // obtain file size
    fseek (current_file, 0L, SEEK_END);
    long file_size = ftell (current_file);
    rewind (current_file);

    // allocate memory to hold file contents (data block)
    data_block = new unsigned char[file_size + 5];  // 5 more bytes to be safe
    
    // read data from file: request to read 3 more bytes to be safe
    data_block_size = fread((void *)data_block,
			    (size_t) 1, (size_t) (file_size + 3),
			    current_file);
    
    return data_block;
}

} // namespace ReadingPeopleTracker
