/***************************************************************
 * C - C++ Header
 *
 * File : 	ConfigurationManager.h
 *
 * Module :	ConfigurationManager
 *
 * Author : 	A M Baumberg (CoMIR)
 *
 * Creation Date : Mon Oct 21 12:39:06 1996 
 *
 * Comments : 	keeps track of configuration variables and initialises them
 *		using a parameter file or a graphical user interface (Motif)
 *
 ***************************************************************/


#ifndef __CONFIGURATION_MANAGER_H__
#define __CONFIGURATION_MANAGER_H__

#include <iostream>
using namespace std;

#ifndef NO_DISPLAY
#include <X11/X.h>          // for Pixmap type
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>  // for Widget type
#include <Xm/Xm.h>
#include <Xm/ToggleB.h>

#include "icons.h"  // for default icon to be used by create_interface
#endif   // #ifndef NO_DISPLAY

#include "realno.h"

namespace ReadingPeopleTracker
{

class ConfigurationManager;

// a container for a single configuration variable
class ConfigurationVariable
{
protected:
    
    enum Type {REAL, INTEGER, STRING, BOOLEAN};
    Type type;
    
    // if fixed is true then 
    // value should not be changed after initialisation
    bool fixed;	
    
    char *owner_class;	// the class that uses this configuration variable
 
    // whether the ConfigurationVariable was matched by an entry in the
    //   configuration file --- to find unmatched entries
    bool matched_in_file;
  
public:
    
    char *tag;	// a unique identifier
    char *help_string;
    
    ConfigurationVariable(char *theTag, Type theType, bool isFixed,
			  char *theOwner, char *help)
	{
	    tag = theTag; type = theType; fixed = isFixed;
	    owner_class = theOwner; help_string = help;
	    matched_in_file = false;
	}
    
    virtual bool read_value(istream&)
	{
	    return false;
	}
    
    virtual void output_value(ostream&)
	{
	    // nothing
	}
    virtual void full_printout(ostream&);
    
#ifndef NO_DISPLAY
    virtual
#endif   // #ifndef NO_DISPLAY
    void *create_widget(void *parent)
	{
	    return NULL;
	}
    
    friend class ConfigurationManager;
};

class RealConfigurationVariable : public ConfigurationVariable
{
protected:
    
    volatile realno &value;
    bool constrained;	// is there a min or max
    realno min; // optional minimum allowed value
    realno max; // optional maximum allowed value;
    
public:
    
    RealConfigurationVariable(char *theTag, bool isFixed,
		       char *theOwner, char *help, volatile realno &theValue)
	: ConfigurationVariable(theTag, ConfigurationVariable::REAL, 
			 isFixed, theOwner, help),
	value(theValue)
	{ constrained = false; }
    
    RealConfigurationVariable(char *theTag, bool isFixed,
		       char *theOwner, char *help, volatile realno &theValue,
		       realno theMin, realno theMax)
	: ConfigurationVariable(theTag, ConfigurationVariable::REAL, isFixed, theOwner, help),
	  value(theValue)
	{ 
	    min = theMin; max = theMax;
	    constrained = true;
	}
    
    bool read_value(istream& );
    volatile realno &get_value() { return value; }
    realno get_min() { return min; }
    realno get_max() { return max; }
    bool is_constrained() { return constrained; }
    
    void output_value(ostream& );
    void full_printout(ostream& );
    void *create_widget(void *widget);
    
    friend class ConfigurationManager;
    
};

class IntConfigurationVariable : public ConfigurationVariable
{
protected:
    
    volatile int &value;
    bool constrained;	// is there a min or max
    int min; // optional minimum allowed value
    int max; // optional maximum allowed value;
    
public:
    
    IntConfigurationVariable(char *theTag, bool isFixed,
		      char *theOwner, char *help, volatile int &theValue)
	: ConfigurationVariable(theTag, ConfigurationVariable::INTEGER, 
			 isFixed, theOwner, help),
	value(theValue)
	{ constrained = false; }
    
    IntConfigurationVariable(char *theTag, bool isFixed,
		      char *theOwner, char *help, volatile int &theValue,
		      int theMin, int theMax)
	: ConfigurationVariable(theTag, ConfigurationVariable::INTEGER, 
			 isFixed, theOwner, help),
	value(theValue)
	{ 
	    min = theMin; max = theMax;
	    constrained = true;
	}
    
    bool read_value(istream& );
    void output_value(ostream& );
    void full_printout(ostream& );
    void *create_widget(void *widget);
    volatile int &get_value() { return value; }
    int get_min() { return min; }
    int get_max() { return max; }
    bool is_constrained() { return constrained; }
    
    friend class ConfigurationManager;
    
};

class BoolConfigurationVariable: public ConfigurationVariable
{
protected:
    
    volatile bool &value;
    
public:
    
    BoolConfigurationVariable(char *theTag, bool isFixed,
		       char *theOwner, char *help, volatile bool &theValue)
	: ConfigurationVariable(theTag, ConfigurationVariable::BOOLEAN, 
			 isFixed, theOwner, help),
	value(theValue) {}
    
    bool read_value(istream& );
    volatile bool &get_value() { return value; }
    
    void output_value(ostream& );
    void full_printout(ostream& );
    
    void *create_widget(void *parent);
    
    friend class ConfigurationManager;
};

class StringConfigurationVariable: public ConfigurationVariable
{
protected:
    
    char *&value;
    bool is_filename;	  // is it a file ? (for GUI)
    char **valid_strings; // if (non-NULL) contains array of possibles
    
public:
    
    StringConfigurationVariable(char *theTag, bool isFixed,
				char *theOwner, char *help, char *&theValue)
	: ConfigurationVariable(theTag, ConfigurationVariable::STRING, 
				isFixed, theOwner, help),
	  value(theValue)
	{
	    is_filename = false;
	}
    
    bool read_value(istream &);
    void output_value(ostream &);
    void full_printout(ostream &);
    void *create_widget(void *parent);
    char *&get_value() { return value; }
    friend class ConfigurationManager;
};


class ConfigurationManager
{
private:
    
    int add_configuration_widgets(void *parent, bool allow_fixed,
				  bool constrained, 
				  ConfigurationVariable::Type type,
				  char **owner_list = NULL,
				  char **tag_list = NULL);
    
    void setup_flags();
    void *create_partial_interface(void*, bool, char **owners, char **tags);
    
    // the actual configuration varables (array of pointers to classes)
    ConfigurationVariable **table;

public:
    
    unsigned int no_variables;
    unsigned int max_variables;
    bool *flags;
    char **our_icon;  // in XPM format so *char[]
    
#ifndef NO_DISPLAY
    Pixmap our_icon_pixmap;
    Pixmap our_icon_shapemask;
#endif   // #ifndef NO_DISPLAY
   
    ConfigurationManager();
    ~ConfigurationManager();
    
    // typical usage ...
    // extern int myConfiguration;
    // int myConfiguration = Configuration.register_int(3, &myConfiguration);
    // int &myConfiguration2 = Configuration.register_int(2);
    
    
    // registering a variable returns a reference to the actual 
    // value (which may be changed later)
    
    volatile realno &register_real(char *tag,
			  realno default_value, volatile realno *holder = NULL,
			  bool fixed = true,
			  char *owner = NULL, char *help = NULL);
    
    volatile realno &register_real(char *tag,
			  realno default_value,
			  realno min, realno max, volatile realno* holder = NULL,
			  bool fixed = true,
			  char *owner = NULL, char *help = NULL);
    
    
    volatile int &register_int(char *tag,
		      int default_value, volatile int *holder = NULL,
		      bool fixed = true,
		      char *owner = NULL, char *help = NULL);
    
    volatile int &register_int(char *tag,
		      int default_value,
		      int min, int max,  volatile int *holder = NULL,
		      bool fixed = true,
		      char *owner = NULL, char *help = NULL);
    
    // most int are unsigned so at least wrap the methods...  // FIXME: change implementation
    volatile inline unsigned int &register_int(char *tag,
				      unsigned int default_value, volatile unsigned int *holder = NULL,
				      bool fixed = true,
				      char *owner = NULL, char *help = NULL)
	{
	    return (unsigned int &) register_int(tag, (int) default_value, (volatile int *) holder,
			    fixed, owner, help);
	}
    
    inline volatile unsigned int register_int(char *tag,
				     unsigned int default_value,
				     unsigned int min, unsigned int max,
				     volatile unsigned int *holder = NULL,
				     bool fixed = true,
				     char *owner = NULL, char *help = NULL)
	{
	    return register_int(tag, (int) default_value,
				(int) min, (int) max,
				(volatile int *) holder, fixed, owner, help);
	}
    
    volatile bool &register_bool(char *tag,
				 bool default_value, volatile bool *holder,
				 bool fixed = true,
				 char *owner = NULL, char *help = NULL);
    
    
    char *&register_string(char *tag,
			   char *default_value, char **holder = NULL,
			   bool fixed = true,
			   char *owner = NULL, char *help = NULL, 
			   char **valid_strings = NULL);
    
    // cast to unsigned char
    unsigned char *&register_string(char *tag,
				    char *default_value,
				    unsigned char **holder = NULL,
				    bool fixed = true,
				    char *owner = NULL,
				    char *help = NULL, 
				    char **valid_strings = NULL)
	{
	    return (unsigned char *&) register_string(tag, default_value,
			    (char **) holder, fixed,
			    owner, help, 
			    valid_strings);
	}
    
    
    char *&register_filename(char *tag,
				      char *default_value, char **holder = NULL,
			     char *owner = NULL, char *help = NULL);
    
    
    void set_value(char *tag, realno new_value);
    void set_value(char *tag, int new_value);
    void set_value(char *tag, char *new_value);
    
    void unmanage_variable(char *tag);
    
    void register_variable(ConfigurationVariable *var);
    
    // read in configuration variable values from a file or stream
    void parse_parameter_stream(istream &input_stream);
    void parse_parameter_file(char *param_file);
    
    // save parameters in parseable format
    void dump_parameters(ostream &output_stream);
    void dump_parameters(char *param_file);
    
    // display all the info to stdout
    void output_info(); 
    
    // set default icon for all windows.
    // this icon will be used by the create_interface function
    inline void set_icon(char **new_icon)
	{
	    our_icon = new_icon;
	}
    
    // create a Motif User Interface so people can change variables
    void create_interface(int argc, char **argv, char **owner_list = NULL,
			  char **tag_list = NULL, bool use_constants = true);
    
    // more static variables and helpers

#ifndef NO_DISPLAY

private:
    static Widget toplevel;
    
    static Widget GetTopShell(Widget w);

public:
    static void toggled(Widget widget, BoolConfigurationVariable *client_data,
			XmToggleButtonCallbackStruct *call_data);
    
    static void help_done(Widget dialog, XtPointer, XtPointer);
    
    static char *word_wrap_string(char *str);
    
    static void help_callback(Widget parent, void *help_text, 
			      XtPointer call_data);
    
    static void real_changed_callback(Widget widget, RealConfigurationVariable *var,
				      XmAnyCallbackStruct *call_data);
    
    static void real_scale_callback(Widget widget, RealConfigurationVariable *var,
				    XmScaleCallbackStruct *call_data);
    
    static void int_changed_callback(Widget widget, IntConfigurationVariable *var,
				     XmAnyCallbackStruct *call_data);
    
    static void int_scale_callback(Widget widget, IntConfigurationVariable *var,
				   XmScaleCallbackStruct *call_data);
    
    static void string_changed_callback(Widget w, StringConfigurationVariable *var,
					XmToggleButtonCallbackStruct *call_data);
    
    static XmString create_directory_string(char *full_filename);
    
    static void file_select_callback(Widget w, StringConfigurationVariable *var,
				     XmFileSelectionBoxCallbackStruct *call_data);
    
    static void get_file_callback(Widget w, StringConfigurationVariable *var,
				  XmPushButtonCallbackStruct *call_data);
    
    static void string_select_callback(Widget w, 
				       StringConfigurationVariable *var,
				       XmSelectionBoxCallbackStruct *call_data);
    
    static void get_string_callback(Widget w, StringConfigurationVariable *var,
				    XmPushButtonCallbackStruct *call_data);

    static void query_for_help(Widget widget, Widget topwidget, XtPointer call_data);

private:
    static void show_widget_cb(Widget, Widget shell, XtPointer);
    static void hide_widget_cb(Widget, Widget shell, XtPointer);
    static void destroy_shell_cb(Widget, Widget shell, XtPointer);
    static void do_save_as_cb(Widget w, ConfigurationManager *gm,  
                              XmFileSelectionBoxCallbackStruct *call_data);
    static void save_as_cb(Widget w, ConfigurationManager *gm, XtPointer);

    static void bye_bye(Widget w, XtPointer, XtPointer);

#endif   // #ifndef NO_DISPLAY

};

} // namespace ReadingPeopleTracker

#endif

