/*********************************************************************
 * C++ source
 *
 * File : 	ConfigurationManager.cc (formerly GlobalsManager.cc)
 *
 * Module :	ConfigurationManager (formerly GlobalsManager)
 *
 * Author : 	A M Baumberg (CoMIR)
 *
 * Creation Date : Mon Oct 21 13:33:42 1996 
 *
 * Changes : nts: Some changes in 2000--2001.
 *           nts: Jul 27, 2001: Fix for problem under Xt 6.0 when using
 *                libstdc++-libc6.2-2.so.3 . Do text search for "////" to find it.
 *           nts: Aug 2001: changed file and class name to more appropriate name.
 *           nts: Aug 2001: added support for setting icon for our window (XPM),
 *                changed colours (fallback) for all windows.
 *
 *********************************************************************/


#include "ConfigurationManager.h"

#include <iostream>
#include <fstream>
#include <cmath>
#include <sstream>
#include <cassert>

#include <cstdio> // for sscanf(...), sprintf(...)

#ifdef LINUX
#include <sys/prctl.h>
#include <sched.h>
#include <unistd.h> // for getpid()
#endif

#include <pthread.h>

#ifndef NO_DISPLAY

#include <cstdlib>  // for atof(...) etc
#include <csignal>  // for SIGHUP

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/cursorfont.h>

#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>
#include <Xm/PushB.h>
#include <Xm/MessageB.h>
#include <Xm/TextF.h>
#include <Xm/Label.h>
#include <Xm/Text.h>
#include <Xm/Scale.h>
#include <Xm/Form.h>
#include <Xm/Separator.h>
#include <Xm/CascadeB.h>
#include <Xm/FileSB.h>
#include <Xm/SelectioB.h>
#include <Xm/DialogS.h>

#include <X11/xpm.h>   // for colour icons in XPM format

#include "os_specific_things.h"  // for usleep()

namespace ReadingPeopleTracker
{
static int back_argc;
static char **back_argv;
static char **back_owners;
static char **back_tags;
ConfigurationManager *back_this;
static pid_t back_pid = getpid();
} // namespace ReadingPeopleTracker

#else
//#include "strcasecmp.h"
#include "os_specific_things.h"  // for usleep()
#endif   // #ifndef NO_DISPLAY

#include "text_output.h" // for cerror etc
#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

void ConfigurationVariable::full_printout(ostream &strm_out )
{
    strm_out << "Tag: " << tag << endl;
    
    if (fixed)
	strm_out << "constant " << endl;
    else
	strm_out << "variable " << endl;
    
    if (owner_class != NULL)
	strm_out << "object class: " << owner_class << endl;
    
    if (help_string != NULL)
	strm_out << "help info: " << help_string << endl;
}

bool RealConfigurationVariable::read_value(istream &strm_in)
{
    realno new_value;
    strm_in >> new_value;
    if (constrained)
    {
	if ((new_value > max) || (new_value < min))
	    return false;
    }
    value = new_value;
    return true;
}

void RealConfigurationVariable::output_value(ostream &strm_out)
{
    if (real_isnan(value)) strm_out << "Undefined" << endl;
    else strm_out << value << endl;
}

void RealConfigurationVariable::full_printout(ostream &strm_out )
{
    ConfigurationVariable::full_printout(strm_out);
    strm_out << "current value "; output_value(strm_out);
    if (constrained)
    {
	strm_out << "( min: " << min << " max: "
		 << max << ")";
    }
    strm_out << endl;
}


bool IntConfigurationVariable::read_value(istream &strm_in)
{
    int new_value;
    strm_in >> new_value;
    if (constrained)
    {
	if ((new_value > max) || (new_value < min))
	    return false;
    }
    value = new_value;
    return true;
}

void IntConfigurationVariable::output_value(ostream &strm_out)
{
    strm_out << value << endl;
}

void IntConfigurationVariable::full_printout(ostream &strm_out )
{
    ConfigurationVariable::full_printout(strm_out);
    strm_out << "current value " << value;
    if (constrained)
    {
	strm_out << "( min: " << min << " max: "
		 << max << ")";
    }
    strm_out << endl;
}

bool BoolConfigurationVariable::read_value(istream &strm_in)
{
//    strm_in >> (int&) value;

// size of a bool is undefined.. better to do it like this
    int temp_value;
    strm_in >> temp_value;
    value = temp_value == 0 ? false : true;
    return true;
}

void BoolConfigurationVariable::output_value(ostream &strm_out)
{
    strm_out << value << endl; 
}

void BoolConfigurationVariable::full_printout(ostream &strm_out)
{
    ConfigurationVariable::full_printout(strm_out);
    strm_out << "current value ";
    if (value == false)
	strm_out << "false" << endl;
    else
	strm_out << "true" << endl;
}

bool StringConfigurationVariable::read_value(istream &strm_in)
{
    static char buffer[255];
    strm_in >> buffer;
    char *new_str = new char[strlen(buffer)+1];
    strcpy(new_str, buffer);
    value = new_str;
    return true;
}

void StringConfigurationVariable::output_value(ostream &strm_out)
{
    if (value != NULL)
	strm_out << value << endl;
}

void StringConfigurationVariable::full_printout(ostream &strm_out )
{
    ConfigurationVariable::full_printout(strm_out);
    
    if (value != NULL)
	strm_out << "current value: " << value << endl;
    else
	strm_out << "not set to any value" << endl;
}


ConfigurationManager::ConfigurationManager()
{
    table = NULL;
    no_variables = 0;
    max_variables = 0;
    
    our_icon = NULL;  // means use default icon from icons.h
    
//   max_variables = 128;
//   no_variables = 0;
//   // set up array of pointers
//   table = new ConfigurationVariable*[max_variables];
}

ConfigurationManager::~ConfigurationManager()
{
    for (unsigned int i = 0; i < no_variables; i++)
	delete table[i];
    
    delete [] table;
}

void ConfigurationManager::register_variable(ConfigurationVariable *var)
{
    if (table == NULL)
    { 
	max_variables = 128;
	no_variables = 0;
	// set up array of pointers
	table = new ConfigurationVariable*[max_variables];
	flags = NULL;
    }  
    
    bool already_registered = false;
    
    for (unsigned int i = 0; i < no_variables; i++)
	if ((strlen(var->tag) == strlen(table[i]->tag)) &&
	    (strcasecmp(table[i]->tag, var->tag) == 0))
	    already_registered = true;
    
    
//if not already registered
    if (!already_registered)
    {
	if ((no_variables+1) < max_variables)
	    table[no_variables++] = var;
	else
	{
	    unsigned int max_variables2 = 2 * max_variables;
	    ConfigurationVariable **table2 = 
		new ConfigurationVariable*[max_variables2];
	    memcpy(table2, table, sizeof(table));
	    delete [] table;
	    table = table2;
	    max_variables = max_variables2;
	    register_variable(var);
	}
    }
    else
    {
	cerror << "ConfigurationManager::register_variable: "
	       << " Illegal multiple registration of variable "
	       << var->tag << " " << endl;
	exit(1);
    }
    
    
}


volatile realno &ConfigurationManager::register_real(char *tag, realno default_value, 
						     volatile realno *holder, bool fixed,
						     char *owner, char *help)
{
    if (holder == NULL) holder = new realno;
    *holder = default_value;
    
    RealConfigurationVariable *tmp = 
	new RealConfigurationVariable(tag, fixed, owner, help, *holder);
    
    register_variable(tmp);
    
    return tmp->value;
}

volatile realno &ConfigurationManager::register_real(char *tag,
						     realno default_value,
						     realno min,
						     realno max, volatile realno *holder,
						     bool fixed,
						     char *owner, char *help)
{
    if (holder == NULL) holder = new realno;
    *holder = default_value;
    
    RealConfigurationVariable *tmp = 
	new RealConfigurationVariable(tag, fixed, owner, help, *holder, min, max);
    
    register_variable(tmp);
    
    return tmp->value;
}


volatile int &ConfigurationManager::register_int(char *tag,
						 int default_value, volatile int *holder,
						 bool fixed,
						 char *owner, char *help)
{
    if (holder == NULL)
	holder = new int;
    
    *holder = default_value;
    
    IntConfigurationVariable *tmp = 
	new IntConfigurationVariable(tag, fixed, owner, help, *holder);
    
    register_variable(tmp);
    
    return tmp->value;
}

volatile int &ConfigurationManager::register_int(char *tag,
						 int default_value,
						 int min,
						 int max, volatile int *holder,
						 bool fixed,
						 char *owner, char *help)
{
    if (holder == NULL)
	holder = new int;
    
    *holder = default_value;
    
    IntConfigurationVariable *tmp = 
	new IntConfigurationVariable(tag, fixed, owner, help, *holder, min, max);
    
    register_variable(tmp);
    
    return tmp->value;
}



volatile bool &ConfigurationManager::register_bool(char *tag,
						   bool default_value, volatile bool *holder,
						   bool fixed,
						   char *owner, char *help)
{
    if (holder == NULL)
	holder = new bool;
    
    *holder = default_value;
    
    BoolConfigurationVariable *tmp = 
	new BoolConfigurationVariable(tag, fixed, owner, help, *holder);
    
    register_variable(tmp);
    
    return tmp->value;
}



char *&ConfigurationManager::register_string(char *tag,
					     char *default_value,  
					     char **holder, bool fixed,
					     char *owner, char *help,
					     char **valid_strings)
{
    if (holder == NULL)
	holder = new char*;
    
    *holder = default_value;
    
    StringConfigurationVariable *tmp = 
	new StringConfigurationVariable(tag, fixed, owner, help, *holder);
    tmp->valid_strings = valid_strings;
    
    register_variable(tmp);
    
    return tmp->value;
}


char *&ConfigurationManager::register_filename(char *tag,
					       char *default_value,  
					       char **holder,
					       char *owner, char *help)
{
    if (holder == NULL)
	holder = new char*;
    
    *holder = default_value;
    
    StringConfigurationVariable *tmp = 
	new StringConfigurationVariable(tag, true, owner, help, *holder);
    tmp->is_filename = true;
    
    register_variable(tmp);
    
    return tmp->value;
}


void ConfigurationManager::parse_parameter_file(char *param_file)
{
    if (param_file == NULL)
	return;
    
    ifstream strm_in(param_file);
    
    if (strm_in.bad())
    {
	cerror << " ConfigurationManager::parse_parameter_file: cannot open file "
	       << param_file << " " << endl;
	exit(1);
    }
    
    cdebug << " ConfigurationManager::parse_parameter_file";
    
    cdebug << endl;
    
    cdebug << " ConfigurationManager::parse_parameter_file: Parsing file "
	   << param_file << " ..." << endl;
    
    parse_parameter_stream(strm_in);
}


// read in configuration variable values from a file or stream
void ConfigurationManager::parse_parameter_stream(istream &input_stream)
{
    char tag_name[256];
    while (input_stream.peek() != EOF)
    {
	// skip out commented lines
	while (input_stream.peek() == '#')
	    while (((char)input_stream.peek() != '\n') && (input_stream.peek() != EOF))
		input_stream.get();
	
	if (input_stream.peek() == EOF)
	    break;
	else
	    if ((char)input_stream.peek() == '\n')
	    {
		input_stream.get();  // the \n
		continue;
	    }
	
	
	tag_name[0] = '\0';
	input_stream >> tag_name;
	
	unsigned int i;
	bool tag_match = false;
	for (i = 0; i < no_variables; i++)
	{
	    if ((strlen(tag_name) == strlen(table[i]->tag)) &&
		(strcasecmp(table[i]->tag, tag_name) == 0))
	    {
		tag_match = true;
		if (! table[i]->read_value(input_stream))
		{
		    cerror << " ConfigurationManager::parse_parameter_stream: Value out of range."
			   << endl;
		    table[i]->full_printout(cdebug);
		}
		table[i]->matched_in_file = true;
	    }
	}
	if (!tag_match)
	{
	    cerror << " ConfigurationManager::parse_parameter_stream: no match for parameter tag "
		   << tag_name << " .  Ignored. " << endl;
	    
	    //ignore & go to next line
	    while (((char)input_stream.peek() != '\n') && (input_stream.peek() != EOF))
		input_stream.get();
	    
	    if (input_stream.peek() == EOF)
		break;
	    else
		if ((char)input_stream.peek() == '\n')
		{
		    input_stream.get();  // the \n
		    continue;
		}
	}
    }
    
    // check for variables which were registered but not in the file
    unsigned int variable_number;
    unsigned int count_unmatched = 0;
    
    for (variable_number = 0; variable_number < no_variables; variable_number++)
    {
	// check each variable in table
	if (table[variable_number]->matched_in_file == false)
	{
	    cerror << " ConfigurationManager::parse_parameter_stream: Warning: variable "
		   << table[variable_number]->tag
		   << " not found in configuration file " << endl;
	    count_unmatched++;
	}
    }
    
#ifdef CONFIG_FILE_DEBUG
    // exit if variable was not specified in config file
    if (count_unmatched > 0)
    {
	cerror << " ConfigurationManager::parse_parameter_stream: Unmatched variables, exiting..."
	       << endl;
	exit(1);
    }
#endif
}


void ConfigurationManager::dump_parameters(char *param_file)
{
    ofstream strm_out(param_file);
    
    if (!strm_out)
    { 
	cerror << " ConfigurationManager::dump_parameters: cannot open parameter file "
	       << param_file << " " << endl;
	return;
    }
    
    dump_parameters(strm_out);
}


// save parameters in parseable format
void ConfigurationManager::dump_parameters(ostream &output_stream)
{
    output_stream << "#\n# Configuration Parameters\n#" << endl;
    
    for (unsigned int i = 0; i < no_variables; i++)
    {
	// don't print out infinity or empty strings [cannot be read in]
	if (((table[i]->type == ConfigurationVariable::REAL)
	     && (real_isnan(((RealConfigurationVariable*)table[i])->value)))
	    || ((table[i]->type == ConfigurationVariable::STRING)
		&& (((StringConfigurationVariable*) table[i])->value == NULL)))
	    continue;
	output_stream << table[i]->tag << "  ";
	table[i]->output_value(output_stream);
    }
}

void ConfigurationManager::output_info()
{
    cinfo << "Configuration variables" << endl;
    cinfo << "****************" << endl;
    for (unsigned int i = 0; i < no_variables; i++)
    {
	table[i]->full_printout(cout);
	cout << endl;
    }
}

void ConfigurationManager::set_value(char *tag, realno new_value)
{
    for (unsigned int i = 0; i < no_variables; i++)    
    {
	if ((strcmp(table[i]->tag, tag) == 0)
	    && (table[i]->type == ConfigurationVariable::REAL))
	    
	    ((RealConfigurationVariable*) table[i])->value = new_value;
    }
}

void ConfigurationManager::set_value(char *tag, int new_value)
{
    for (unsigned int i = 0; i < no_variables; i++)    
    {
	if (strcmp(table[i]->tag, tag) == 0)
	{
	    
	    switch (table[i]->type)
	    {
	    case ConfigurationVariable::INTEGER:
		((IntConfigurationVariable*) table[i])->value = new_value;
		break;
		
	    case ConfigurationVariable::BOOLEAN:
		((BoolConfigurationVariable*) table[i])->value = new_value;
		break;
	    default:
		cerror << " Wrong call to ConfigurationManager::set_value() "
		       << endl;
		exit(1);
	    }      
	}
    }
}

void ConfigurationManager::set_value(char *tag, char *new_value)
{
    for (unsigned int i = 0; i < no_variables; i++)    
    {
	if ((strcmp(table[i]->tag, tag) == 0)
	    &&  (table[i]->type == ConfigurationVariable::STRING))
	    ((StringConfigurationVariable*) table[i])->value = new_value;
    }
}

void ConfigurationManager::unmanage_variable(char *tag)
{
    ConfigurationVariable **new_table = new ConfigurationVariable*[max_variables];
    unsigned int j = 0;
    for (unsigned int i = 0; i < no_variables; i++)    
    {
	if (strcmp(table[i]->tag, tag) != 0)
	    new_table[j++] = table[i];
    }
    delete [] table;
    table = new_table;
    no_variables = j;
}


/************************ MOTIF INTERFACE ***************************/

#ifndef NO_DISPLAY
Widget ConfigurationManager::toplevel = NULL;

Widget ConfigurationManager::GetTopShell(Widget w)
{
    while ((w != NULL) && !(XtIsTopLevelShell(w)))
	w = XtParent(w);
    return w;
}


void ConfigurationManager::toggled(Widget widget, BoolConfigurationVariable *client_data,
				   XmToggleButtonCallbackStruct *call_data)
{
    client_data->get_value() = (bool) call_data->set;
}

void ConfigurationManager::help_done(Widget dialog, XtPointer, XtPointer)
{
    XtDestroyWidget(dialog);
}


char *ConfigurationManager::word_wrap_string(char *str)
{
    static const unsigned int MAX_LINES = 100;
    static const unsigned int MAX_LINE_LENGTH = 1000; // FIXME nts: fix to display full help
    
    size_t new_str_length = strlen(str) + (MAX_LINES * strlen("\n")) + 2;
    char *new_str = new char[new_str_length];
    
    stringbuf data_in;
    stringbuf data_out;
    data_in.pubsetbuf(str, strlen(str) + 2);
    data_out.pubsetbuf(new_str, new_str_length);
    
    char aline[MAX_LINE_LENGTH + 2];
    
    unsigned int line, column;
    
    line = 0;
    
    while ((data_in.in_avail() > 0) && (line < MAX_LINES))
    {
	aline[0] = '\0';
	column = data_in.sgetn(aline, MAX_LINE_LENGTH);
	
	if (column >= (MAX_LINE_LENGTH-1))
	{
	    while ((column > MAX_LINE_LENGTH/2) && (strchr("\n .:-;!?", aline[--column]) == 0))
		data_in.sputbackc(aline[column]);
	    
	    if (column <= MAX_LINE_LENGTH/2)  // very long word at the end of a line?
	    {
		// break it across the line
		assert (MAX_LINE_LENGTH > 6);
		data_in.sgetn(aline, (MAX_LINE_LENGTH/2 - (column + 2)));
		column = strlen(aline);
		aline[column] = '\\';
	    }
	    
	    aline[column++] = '\0';
	}
	data_out.sputn(aline, column);
	data_out.sputc('\n');
	
	line++;  // count lines as we move along
    }
    
    strncpy(new_str, data_out.str().c_str(), new_str_length - 1);
    new_str[new_str_length - 1] = '\0';
    
    return new_str;
}


void ConfigurationManager::help_callback(Widget parent, void *help_text, 
					 XtPointer call_data)
{
    Widget dialog;
    XmString text;
    Arg args[5];
    unsigned int n = 0;
    
    char *word_wrap_text = word_wrap_string((char *)help_text);
    text = XmStringCreateLtoR(word_wrap_text, XmFONTLIST_DEFAULT_TAG);
    delete [] word_wrap_text;
    
    XtSetArg(args[n], XmNmessageString, text); n++;
    XtSetArg(args[n], XmNautoUnmanage, False); n++;
    dialog = XmCreateInformationDialog(parent,"help",args,n);
    XmStringFree(text);
    
    XtUnmanageChild(XmMessageBoxGetChild(dialog,
					 XmDIALOG_CANCEL_BUTTON));
    XtSetSensitive(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON),
		   False);
    XtAddCallback(dialog, XmNokCallback, help_done, NULL);
    
    XtManageChild(dialog);
    XtPopup(XtParent (dialog), XtGrabNone);
}


void *BoolConfigurationVariable::create_widget(void *parent)
{
    Widget toggle = 
	XtVaCreateManagedWidget(tag, xmToggleButtonWidgetClass,
				(Widget) parent, XmNset, (Boolean) value, NULL);
    
    XtAddCallback(toggle, XmNvalueChangedCallback,
		  (XtCallbackProc) ConfigurationManager::toggled, this);
    
    XtAddCallback(toggle, XmNhelpCallback,
		  &ConfigurationManager::help_callback, help_string);

    return toggle;
}


void ConfigurationManager::real_changed_callback(Widget widget, RealConfigurationVariable *var,
						 XmAnyCallbackStruct *call_data)
{
    
    char *text = XmTextGetString(widget);
    float new_value;
    if (sscanf(text,"%f", &new_value) == 1)
	var->get_value() = atof(text);
    
    static char buffer[256];
    if (real_isnan(var->get_value()))
	sprintf(buffer,"undefined");
    else sprintf(buffer,"%f", var->get_value());
    XtRemoveCallback(widget, XmNactivateCallback, (XtCallbackProc)
		     real_changed_callback, var);
    
    XmTextSetString(widget, buffer);
    
    XtAddCallback(widget, XmNactivateCallback, (XtCallbackProc)
		  real_changed_callback, var);
    XtFree(text);
}

void ConfigurationManager::real_scale_callback(Widget widget, RealConfigurationVariable *var,
					       XmScaleCallbackStruct *call_data)
{
    short dpoints;
    XtVaGetValues(widget, XmNdecimalPoints, &dpoints, NULL);
    realno sfac = pow(10.0, dpoints);
    var->get_value() = ((realno) call_data->value) / sfac;
}


void *RealConfigurationVariable::create_widget(void *parent)
{
    if (!constrained)
    {
	Widget container = 
	    XtVaCreateManagedWidget("container", xmRowColumnWidgetClass,
				    (Widget) parent,
				    XmNorientation, XmHORIZONTAL,
				    NULL);
	Widget wfield = 
	    XtVaCreateManagedWidget("field", xmTextFieldWidgetClass, container,
				    XmNcolumns, 9, NULL);
	
	XtAddCallback(wfield, XmNactivateCallback, 
		      (XtCallbackProc) ConfigurationManager::real_changed_callback, this);
	
	XtAddCallback(wfield, XmNlosingFocusCallback,
		      (XtCallbackProc) ConfigurationManager::real_changed_callback, this);
	
	XtAddCallback(container, XmNhelpCallback, ConfigurationManager::help_callback,
		      help_string);

	ConfigurationManager::real_changed_callback(wfield, this, NULL);

	XtVaCreateManagedWidget(tag, xmLabelWidgetClass, container,
				NULL);
	return container;
    }
    else
    {
	realno fstep = (max - min);
	int ndigits = 0;
	while (fstep >= 1) { fstep /= 10; ndigits++; }
	while ((ndigits <= 0) && (fstep < 0.1))
	{ fstep *= 10; ndigits--; }
	
	// we would like 4 digits so scale numbers appropriately
	int dpoints = 4 - ndigits;
	realno sfac = pow(10.0, dpoints);
	int imax = (int) (max * sfac);
	int imin = (int) (min * sfac);
	int ivalue = (int) (value * sfac);
	
	Widget wscale = 
	    XtVaCreateManagedWidget(tag, xmScaleWidgetClass,(Widget)
				    parent, XtVaTypedArg, XmNtitleString,
				    XmRString, tag, strlen(tag),
				    XmNscaleWidth, 180,
				    XmNmaximum, imax,
				    XmNminimum, imin,
				    XmNdecimalPoints, dpoints,
				    XmNvalue, ivalue,
				    XmNshowValue, True, 
				    XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(wscale, XmNvalueChangedCallback, 
		      (XtCallbackProc) ConfigurationManager::real_scale_callback, this);
	XtAddCallback(wscale, XmNhelpCallback, ConfigurationManager::help_callback,
		      help_string);			       
	return wscale;
    }
    
}



void ConfigurationManager::int_changed_callback(Widget widget, IntConfigurationVariable *var,
						XmAnyCallbackStruct *call_data)
{
    
    char *text = XmTextGetString(widget);
    int new_value;
    if (sscanf(text,"%d", &new_value) == 1)
	var->get_value() = atoi(text);
    
    static char buffer[256];
    sprintf(buffer,"%d", var->get_value());
    XtRemoveCallback(widget, XmNactivateCallback, (XtCallbackProc)
		     int_changed_callback, var);
    
    XmTextSetString(widget, buffer);
    
    XtAddCallback(widget, XmNactivateCallback, (XtCallbackProc)
		  int_changed_callback, var);
    XtFree(text);
}

void ConfigurationManager::int_scale_callback(Widget widget, IntConfigurationVariable *var,
					      XmScaleCallbackStruct *call_data)
{
    var->get_value() = (int) call_data->value;
}


void *IntConfigurationVariable::create_widget(void *parent)
{
    if (!constrained)
    {
	Widget container = 
	    XtVaCreateManagedWidget("container", xmRowColumnWidgetClass,
				    (Widget) parent,
				    XmNorientation, XmHORIZONTAL,
				    NULL);
	Widget wfield = 
	    XtVaCreateManagedWidget("field", xmTextFieldWidgetClass, container,
				    XmNcolumns, 6, NULL);
	XtAddCallback(wfield, XmNactivateCallback, 
		      (XtCallbackProc) ConfigurationManager::int_changed_callback, this);
	XtAddCallback(wfield, XmNlosingFocusCallback,
		      (XtCallbackProc) ConfigurationManager::int_changed_callback, this);
	XtAddCallback(container, XmNhelpCallback, ConfigurationManager::help_callback,
		      help_string);
	ConfigurationManager::int_changed_callback(wfield, this, NULL);
	XtVaCreateManagedWidget(tag, xmLabelWidgetClass, container, NULL);
	return container;
    }
    else
    {
	Widget wscale = 
	    XtVaCreateManagedWidget(tag, xmScaleWidgetClass,(Widget)
				    parent, XtVaTypedArg, XmNtitleString,
				    XmRString, tag, strlen(tag),
				    XmNscaleWidth, 180,
				    XmNmaximum, max,
				    XmNminimum, min,
				    XmNvalue, value,
				    XmNshowValue, True, 
				    XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(wscale, XmNvalueChangedCallback, 
		      (XtCallbackProc) ConfigurationManager::int_scale_callback, this);
	XtAddCallback(wscale, XmNhelpCallback, ConfigurationManager::help_callback,
		      help_string);			       
	return wscale;
    }
    
}

void ConfigurationManager::string_changed_callback(Widget w, StringConfigurationVariable *var,
						   XmToggleButtonCallbackStruct *call_data)
{
    if (call_data->set == True)
    {
	// the selected string is the name of the widget !
	var->get_value() = XtName(w);	
    }
}

XmString ConfigurationManager::create_directory_string(char *full_filename)
{
    static char buffer[256];
    strncpy(buffer, full_filename, 255);
    char *pnt = strrchr(buffer, '/');
    if (pnt != NULL) pnt[1] = '\0';
    XmString motif_str = XmStringCreateLocalized(buffer);
    return motif_str;
}

void ConfigurationManager::file_select_callback(Widget w, StringConfigurationVariable *var,
						XmFileSelectionBoxCallbackStruct 
						*call_data)
{
    char *new_value = NULL;
    if (!XmStringGetLtoR(call_data->value, XmFONTLIST_DEFAULT_TAG,
			 &new_value)) return;
    
    var->get_value() = new_value;
    
    XtDestroyWidget(w);
}

void ConfigurationManager::get_file_callback(Widget w, StringConfigurationVariable *var,
					     XmPushButtonCallbackStruct *call_data)
{
    Widget dialog = 
	XmCreateFileSelectionDialog(GetTopShell(w), "_file", NULL, 0);
    XtAddCallback(dialog, XmNcancelCallback, (XtCallbackProc) help_done,
		  NULL);
    XtAddCallback(dialog, XmNhelpCallback, (XtCallbackProc)
		  help_callback, var->help_string);
    XtAddCallback(dialog, XmNokCallback, (XtCallbackProc)
		  file_select_callback, var);
    
    XmString dirname = create_directory_string(var->get_value());
    XmString dirspec = XmStringCreateLocalized(var->get_value());
    XmString tag_str = XmStringCreateLocalized(var->tag);
    XmString tmp_str = XmStringCreateLocalized("Selection for ");
    XmString select_str = XmStringConcat(tmp_str, tag_str);
    
    XtVaSetValues(dialog, XmNdirectory, dirname,
		  XmNdirSpec, dirspec, NULL);
    
    Widget flabel =
	XmFileSelectionBoxGetChild(dialog, XmDIALOG_SELECTION_LABEL);
    XtVaSetValues(flabel, XmNlabelString, select_str, NULL);
    
    XmStringFree(dirname);
    XmStringFree(dirspec);
    XmStringFree(tag_str);
    XmStringFree(tmp_str);
    XmStringFree(select_str);
    
    XtManageChild(dialog);
}


void ConfigurationManager::string_select_callback(Widget w, 
						  StringConfigurationVariable *var,
						  XmSelectionBoxCallbackStruct *call_data)
{
    
    char *new_value = NULL;
    if (!XmStringGetLtoR(call_data->value, XmFONTLIST_DEFAULT_TAG,
			 &new_value)) return;
    if (strlen(new_value) == 0) new_value = NULL;
    var->get_value() = new_value;
    
    XtDestroyWidget(w);
    
}
void ConfigurationManager::get_string_callback(Widget w, StringConfigurationVariable *var,
					       XmPushButtonCallbackStruct *call_data)
{
    Widget dialog =
	XmCreatePromptDialog(w, var->tag, NULL, 0);
    XmString tmp1 = XmStringCreateLocalized(var->tag);
    XmString tmp2 = XmStringCreateLocalized(var->get_value());
    XtVaSetValues(dialog, XmNselectionLabelString, tmp1, 
		  XmNtextString, tmp2, NULL);
    XtAddCallback(dialog, XmNcancelCallback, (XtCallbackProc) help_done,
		  NULL);
    XtAddCallback(dialog, XmNhelpCallback, (XtCallbackProc)
		  help_callback, var->help_string);
    XtAddCallback(dialog, XmNokCallback, (XtCallbackProc)
		  string_select_callback, var);
    XmStringFree(tmp1);
    XmStringFree(tmp2);
    XtManageChild(dialog);
    XtPopup(XtParent(dialog), XtGrabNone);
}

void *StringConfigurationVariable::create_widget(void *vparent)
{
    Widget parent = (Widget) vparent;
    if (is_filename)
    {
	Widget get_file = 
	    XtVaCreateManagedWidget(tag, xmPushButtonWidgetClass,
				    parent, NULL);
	XtAddCallback(get_file, XmNactivateCallback,
		      (XtCallbackProc) ConfigurationManager::get_file_callback, this);
	XtAddCallback(get_file, XmNhelpCallback,
		      (XtCallbackProc) ConfigurationManager::help_callback, help_string);
	return get_file;
    }
    
    if (valid_strings == NULL) 
    {
	Widget get_string = 
	    XtVaCreateManagedWidget(tag, xmPushButtonWidgetClass,
				    parent, NULL);
	XtAddCallback(get_string, XmNactivateCallback,
		      (XtCallbackProc) ConfigurationManager::get_string_callback, this);
	XtAddCallback(get_string, XmNhelpCallback,
		      (XtCallbackProc) ConfigurationManager::help_callback, help_string);
	return get_string;
    }
    
    
    XmString tag_str = XmStringCreateLocalized(tag);
    Widget menubar = 
	XmVaCreateSimpleMenuBar(parent, "_menubar", XmVaCASCADEBUTTON,
				tag_str, tag[0], NULL);
    XmStringFree(tag_str);
    
    Widget PullDown =
	XmVaCreateSimplePulldownMenu(menubar, "_pulldown", 0,
				     (XtCallbackProc) NULL, NULL);
    XtVaSetValues(PullDown, XmNradioBehavior, True,
		  XmNradioAlwaysOne, True, NULL);
//   Widget cascade = 
//     XtVaCreateManagedWidget(tag, xmCascadeButtonWidgetClass,
// 			    parent, XmNsubMenuId, PullDown, NULL);
    for (int i = 0; valid_strings[i] != NULL; i++)
    {
	Boolean is_set = False;
	if (strcmp(valid_strings[i], value) == 0)
	    is_set = True;
	Widget option_i = 
	    XtVaCreateManagedWidget(valid_strings[i],
				    xmToggleButtonWidgetClass,
				    PullDown, XmNset, is_set,
				    XmNhighlightOnEnter, True, NULL);
	XtAddCallback(option_i, XmNvalueChangedCallback,
		      (XtCallbackProc) ConfigurationManager::string_changed_callback,
		      this);
    }
    XtAddCallback(menubar, XmNhelpCallback,
		  ConfigurationManager::help_callback, help_string);
    XtManageChild(menubar);
    return menubar;
}


void ConfigurationManager::query_for_help(Widget widget, Widget topwidget,
					  XtPointer call_data)
{
    Cursor cursor;
    Display *display;
    Widget help_widget;
    XmAnyCallbackStruct *cbs, newcbs;
    XEvent event;
    
    cbs = (XmAnyCallbackStruct *) call_data;
    display = XtDisplay(topwidget);
    cursor = XCreateFontCursor(display, XC_hand2);
    
    help_widget = XmTrackingEvent(topwidget, cursor, True, &event);
    while (help_widget != NULL)
    {
	if (XtHasCallbacks(help_widget, XmNhelpCallback) == 
	    XtCallbackHasSome) 
	{
	    newcbs.reason = XmCR_HELP;
	    newcbs.event = &event;
	    XtCallCallbacks(help_widget, XmNhelpCallback,
			    (XtPointer) &newcbs);
	    help_widget = NULL;
	}
	else help_widget = XtParent(help_widget);
    }
    XFreeCursor(display, cursor);
}


void ConfigurationManager::show_widget_cb(Widget, Widget shell, XtPointer)
{
    
    XtRealizeWidget(shell);
    XMapRaised(XtDisplay(shell), XtWindow(shell));
    
}

void ConfigurationManager::hide_widget_cb(Widget, Widget shell, XtPointer)
{
    XtUnmapWidget(shell);
}

void ConfigurationManager::destroy_shell_cb(Widget, Widget shell, XtPointer)
{
    XtUnmapWidget(shell);
    XtDestroyWidget(shell);
}

void ConfigurationManager::do_save_as_cb(Widget w, ConfigurationManager *gm,  
					 XmFileSelectionBoxCallbackStruct *call_data)
{
    
    char *new_value = NULL;
    if (!XmStringGetLtoR(call_data->value, XmFONTLIST_DEFAULT_TAG,
			 &new_value))
	return;
    
    gm->dump_parameters(new_value);
    
    XmStringFree((XmString)new_value);
    
    XtDestroyWidget(w);
}

void ConfigurationManager::save_as_cb(Widget w, ConfigurationManager *gm, XtPointer)
{
    Widget dialog = 
	XmCreateFileSelectionDialog(GetTopShell(w), "_file", NULL, 0);
    
    XtAddCallback(dialog, XmNcancelCallback, (XtCallbackProc) help_done,
		  NULL);
    XtAddCallback(dialog, XmNokCallback, (XtCallbackProc)
		  do_save_as_cb, gm);
    
    XtManageChild(dialog);
}

void ConfigurationManager::bye_bye(Widget w, XtPointer, XtPointer)
{
#ifdef LINUX
    pthread_exit(NULL);
#endif
    
    exit(0);
}

#ifdef LINUX
//int child_entry_function (void* )
void *child_entry_function (void* )
{
    prctl(PR_SET_PDEATHSIG,SIGHUP); 
    back_this->create_interface(back_argc, back_argv,
				back_owners, back_tags, false);
    return NULL;
}
#endif
#endif   // #ifndef NO_DISPLAY

void ConfigurationManager::create_interface(int argc, char **argv, 
					    char **owners,
					    char **tags, bool use_constants )
{
#ifndef NO_DISPLAY
    XtAppContext app;
    static String fallbacks[] =
	{
	    "*Foreground:  NavyBlue",                     // that's #000080
	    "*BorderWidth: 0",
	    "*Background:  LightSteelBlue1",              // that's #cae1ff
	    "*XmToggleButton.selectColor:         gold",  // that's #ffd700
	    "*XmToggleButton.indicatorSize:       16",
	    "*XmToggleButtonGadget.selectColor:   gold",  // that's #ffd700
	    "*XmToggleButtonGadget.indicatorSize: 16",
	    "*fontList: 6x13",
	    "*XmText*fontList: -adobe-courier-medium-r-normal--12-*-*-*-m-70-iso8859-1",
	    NULL
	};
    
    volatile static bool have_done = 0;
    
    if (use_constants)
	back_pid = getpid();
    
    XtSetLanguageProc(NULL, NULL, NULL);
    
    toplevel = XtVaAppInitialize (&app,"GUI", NULL, 0, &argc, argv, fallbacks, NULL);
    
    // set icon...
    if (our_icon == NULL)         // not set?
	our_icon = DEFAULT_ICON;  // use default
    
    if (XpmCreatePixmapFromData(XtDisplay(toplevel),
				RootWindowOfScreen(XtScreen(toplevel)),
				our_icon,
				&our_icon_pixmap,
				&our_icon_shapemask,
				0)
	== XpmSuccess)
    {
	XtVaSetValues(toplevel, // XmNtitle, "Reading People Tracker",
		      XmNiconPixmap, our_icon_pixmap, NULL);
    }
    else
	cdebug << " ConfigurationManager: could not XpmCreatePixmapFromData " << endl;
    
    static char *main_list[] = {"ReadingPeopleTracker", NULL};
    
    Widget mainpanel = (Widget)
	create_partial_interface(toplevel, use_constants, main_list, tags);
    
    Widget extra_shell = 
	XtVaCreatePopupShell("Options", topLevelShellWidgetClass,
			     toplevel, NULL);
    
    // set our default icon...
    XtVaSetValues(extra_shell,
		  XmNiconPixmap, our_icon_pixmap, NULL);
    
    Widget extra_panel = (Widget)
	create_partial_interface(extra_shell, use_constants, owners, tags);
    
    XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass,
			    extra_panel, XmNorientation, XmHORIZONTAL, NULL);
    
    Widget more_buttons = 
	XtVaCreateManagedWidget("more_buttons", xmRowColumnWidgetClass,
				extra_panel,
				XmNorientation, XmHORIZONTAL,
				XmNpacking, XmPACK_COLUMN, NULL);
    Widget hide_button =
	XtVaCreateManagedWidget("Hide", xmPushButtonWidgetClass,
				more_buttons, NULL);
    XtAddCallback(hide_button, XmNactivateCallback, (XtCallbackProc)
		  hide_widget_cb, extra_shell);
    
    Widget helpb2 = 
	XtVaCreateManagedWidget("Help", xmPushButtonWidgetClass, more_buttons,
				NULL);
    XtAddCallback(helpb2, XmNactivateCallback, (XtCallbackProc)
		  query_for_help, extra_shell);
    
    Widget savebutton = 
	XtVaCreateManagedWidget("Save parameters",
				xmPushButtonWidgetClass, more_buttons,
				NULL);
    
    XtAddCallback(savebutton, XmNactivateCallback, (XtCallbackProc)
		  save_as_cb, this);
    
    XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass,
			    mainpanel, XmNorientation, XmHORIZONTAL, NULL);
    
    Widget buttons = 
	XtVaCreateManagedWidget("buttons", xmRowColumnWidgetClass,
				mainpanel,
				XmNorientation, XmHORIZONTAL,
				XmNpacking, XmPACK_COLUMN, NULL);
    
    Widget extras_button = 
	XtVaCreateManagedWidget("More options ...", xmPushButtonWidgetClass,
				buttons, NULL);
    XtAddCallback(extras_button, XmNactivateCallback, (XtCallbackProc)
		  show_widget_cb, extra_shell);
    
    Widget help_button = 
	XtVaCreateManagedWidget("Help", xmPushButtonWidgetClass, buttons,
				NULL);
    XtAddCallback(help_button, XmNactivateCallback, (XtCallbackProc)
		  query_for_help, toplevel);
    
    
    if (use_constants)
    {
	BoolConfigurationVariable var_done("GO", false, "", "", have_done);
	Widget end_button =
	    XtVaCreateManagedWidget("GO!", xmPushButtonWidgetClass, buttons,
				    NULL);
	XtAddCallback(end_button, XmNactivateCallback,
		      (XtCallbackProc) toggled, &var_done);
	Widget quit_button =
	    XtVaCreateManagedWidget("Cancel", xmPushButtonWidgetClass, buttons,
				    NULL);
	XtAddCallback(quit_button, XmNactivateCallback,
		      (XtCallbackProc) bye_bye, NULL);
    }
    else
    {
	Widget end_button =
	    XtVaCreateManagedWidget("Hide", xmPushButtonWidgetClass, buttons,
				    NULL);
	XtAddCallback(end_button, XmNactivateCallback,
		      (XtCallbackProc) destroy_shell_cb, toplevel);
	Widget quit_button =
	    XtVaCreateManagedWidget("Quit", xmPushButtonWidgetClass, buttons,
				    NULL);
	XtAddCallback(quit_button, XmNactivateCallback,
		      (XtCallbackProc) bye_bye, NULL);
    }
    
    XtRealizeWidget(toplevel);  
    XMapRaised(XtDisplay(toplevel), XtWindow(toplevel));
    
    if (!use_constants) 
    {
	have_done = true;
	XtAppMainLoop(app);
	exit(1);
    }
    
    while (!have_done)
    {
	XEvent next_event;
	XtAppNextEvent(app, &next_event);
	XtDispatchEvent(&next_event);
    }
    
//// nts: changed the following to circumvent some nasty Xt problems when
////   compiling with libXt.so.6.0 and libstdc++-libc6.2-2.so.3  ---  Jul 2001
    
    XtUnmanageChild(extra_shell);
    XtUnrealizeWidget(extra_shell);  //// unrealise only because destroy won't work
////    XtDestroyWidget(extra_shell);
    Display *the_display = XtDisplay(toplevel);
    XtUnrealizeWidget(toplevel);  //// unrealise only because destroy won't work
////    XtDestroyWidget(toplevel);
    
    XFlush(the_display);
    toplevel = NULL;
    if (use_constants) 
    {
	// now generate background interface
	delete [] flags; flags = NULL;
	back_argc = argc; back_argv = argv;
	back_this = this; back_owners = owners;
	back_tags = tags;
	have_done = false;
#ifdef LINUX
// use the POSIX 1003.1c compliant pthread library
	pthread_t child_thread;
	if ((pthread_create (&child_thread,
			     NULL,  // no attributes => defaults are used
			     &child_entry_function,
			     NULL)))
	{
	    cerror << " ConfigurationManager::create_interface: Error creating new thread "
		   << endl;
	    exit(1);
	};
#endif
	// Note: when optimising code, this loop must be non-empty
	while (have_done == false)
	{
	    cdebug << "ConfigurationManager::create_interface: waiting for clone to come up...\r";
	    usleep(50*1000);  // wait 50 msec
	}
        // clear to end of line...
        cdebug << "                                                                        \r"
	       << flush;
    }
    
//  XtAppMainLoop(app);
#endif   // ifndef NO_DISPLAY
}

#ifndef NO_DISPLAY

// create_partial_interface:
// creates a row-column widget containing all the desired
// buttons etc 
// owners : list of owner strings to match on 
// tags: list of tag strings to match on
// use_constants : match constants as well as variables
void *ConfigurationManager::create_partial_interface(void *parent,
						     bool use_constants,
						     char **owners,
						     char **tags )
{
    Widget rowcol = 
	XtVaCreateManagedWidget("rowcolumn",
				xmRowColumnWidgetClass,	   
				(Widget) parent, 
				XmNorientation, XmVERTICAL,
				NULL);
    Widget row1 =
	XtVaCreateManagedWidget("row1", xmRowColumnWidgetClass,
				rowcol,
				XmNorientation, XmHORIZONTAL,
				XmNpacking, XmPACK_COLUMN, NULL);
    int n1 = 
	add_configuration_widgets(row1, use_constants, false,
				  ConfigurationVariable::BOOLEAN,  owners, tags);
    int r1 = 1 + (n1 / 5);
    XtVaSetValues(row1, XmNnumColumns, r1, NULL);
    if (n1 == 0) 
    { XtUnmanageChild(row1); XtDestroyWidget(row1); }
    
    
    Widget row2 =
	XtVaCreateManagedWidget("row2", xmRowColumnWidgetClass,
				rowcol,
				XmNorientation, XmHORIZONTAL,
				XmNpacking, XmPACK_COLUMN, NULL);
    
    int n2 = 
	add_configuration_widgets(row2, use_constants, false, ConfigurationVariable::REAL,
				  owners, tags);
    
    int r2 = 1 + (n2 / 4);
    XtVaSetValues(row2, XmNnumColumns, r2, NULL);
    if (n2 == 0)
    {  XtUnmanageChild(row2); XtDestroyWidget(row2); }
    
    Widget row3 = XtVaCreateManagedWidget("row3", xmRowColumnWidgetClass,
					  rowcol,
					  XmNorientation, XmHORIZONTAL,
					  XmNpacking, XmPACK_COLUMN, NULL);
    
    int n3 = 
	add_configuration_widgets(row3, use_constants, false, ConfigurationVariable::INTEGER,
				  owners, tags);
    int r3 = 1 + (n3 / 4);
    XtVaSetValues(row3, XmNnumColumns, r3, NULL);
    if (n3 == 0)
    {  XtUnmanageChild(row3); XtDestroyWidget(row3); }
    
    
    Widget row4 = XtVaCreateManagedWidget("row4", xmRowColumnWidgetClass,
					  rowcol,
					  XmNorientation, XmHORIZONTAL,
					  XmNpacking, XmPACK_COLUMN, NULL);
    
    int n4 = add_configuration_widgets(row4, use_constants,
				       true, ConfigurationVariable::INTEGER,
				       owners, tags);
    n4 += add_configuration_widgets(row4, use_constants, true, ConfigurationVariable::REAL,
				    owners, tags);
    int r4 = 1 + (n4 / 5);
    XtVaSetValues(row4, XmNnumColumns, r4, NULL);
    if (n4 == 0)
    {  XtUnmanageChild(row4); XtDestroyWidget(row4); }
    
    Widget sep =
	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass,
				rowcol, XmNorientation, XmHORIZONTAL, NULL);
    
    Widget row5 = XtVaCreateManagedWidget("row5", xmRowColumnWidgetClass,
					  rowcol, 
					  XmNorientation, XmHORIZONTAL,
					  XmNpacking, XmPACK_COLUMN, NULL);
    int n5 = add_configuration_widgets(row5, use_constants, false,
				       ConfigurationVariable::STRING,
				       owners, tags);
    int r5 = 1 + (n5 / 5);
    XtVaSetValues(row5, XmNnumColumns, r5, NULL);
    if (n5 == 0)
    { 
	XtUnmanageChild(row5); XtDestroyWidget(row5); 
	XtUnmanageChild(sep); XtDestroyWidget(sep); 
    }
    
    return rowcol;
}
#endif   // #ifndef NO_DISPLAY

void ConfigurationManager::setup_flags()
{
    if (flags != NULL)
	delete [] flags;
    flags = new bool[no_variables];
    for (int i = 0; i < no_variables; i++)
	flags[i] = false;
}


int ConfigurationManager::add_configuration_widgets(void *parent, bool allow_fixed,
						    bool constrained, 
						    ConfigurationVariable::Type type,
						    char **owner_list,
						    char **tag_list)
{
    if (flags == NULL) setup_flags();
    
    unsigned int i;
    unsigned int nmatched = 0;
    for (i = 0; i < no_variables; i++)
    {
	if (flags[i]) continue;
	ConfigurationVariable *curr = table[i];
	if (!allow_fixed && curr->fixed) continue;
	if (curr->type != type) continue;
	if ((type == ConfigurationVariable::REAL) &&
	    (constrained !=
	     ((RealConfigurationVariable*)curr)->is_constrained()))
	    continue;
	if ((type == ConfigurationVariable::INTEGER) &&
	    (constrained != 
	     ((IntConfigurationVariable*)curr)->is_constrained()))
	    continue;
	
	if (owner_list != NULL)
	{
	    unsigned int j = 0;
	    bool found_match = false;
	    while ((owner_list[j] != NULL) && (!found_match))
	    {
		if (strcmp(owner_list[j], curr->owner_class) == 0)
		    found_match = true;
		else j++;
	    }
	    if (!found_match) continue;
	}
	
	if (tag_list != NULL)
	{
	    unsigned int k = 0;
	    bool found_match = false;
	    while ((tag_list[k] != NULL) && (!found_match))
	    {
		if (strcmp(tag_list[k], curr->tag) == 0)
		    found_match = true;
		else k++;
	    }
	    if (!found_match) continue;
	}
	
	// add this configuration variable
	nmatched++;
//  #ifndef NO_DISPLAY
//  	Widget myWidget = (Widget) curr->create_widget(parent);
//  #endif   // #ifndef NO_DISPLAY
	// flag so that we don't match this one again
	flags[i] = true;
    }
    return nmatched;
}					
     
} // namespace ReadingPeopleTracker
