///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  TrackedObjectSet.cc                                                      //
//                                                                           //
//  List to hold `TrackedObject's                                            //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Tue Oct 23 12:04:12 BST 2001                                 //
//  Revision  : 1.5 of Tue Jun 18 16:47:21 BST 2002                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "Image.h"
#include "TrackedObjectSet.h"
#include "Tracking.h"
#include "Camera.h"
#include "Inputs.h"

namespace ReadingPeopleTracker
{

static const char *TrackedObjectSet_Revision = "@(#) TrackedObjectSet.cc, rev 1.5 of Tue Jun 18 16:47:21 BST 2002, Author Nils T Siebel, Copyright (c) 2001--2002 The University of Reading";

TrackedObject *TrackedObjectSet::add_observation(Region *region,
						 frame_id_t current_frame_id,
						 frame_time_t current_frame_time)
{
    // for now: simply create a new TrackedObject for each Region
    // FIXME: should compare with old regions in set and merge so it will not grow
    
    // record that region is visible and frame that region was first and last visible
    region->is_visible = true;
    region->frame_first_detected =
	region->frame_last_detected = current_frame_id;
    region->time_first_detected =
	region->time_last_detected = current_frame_time;
    
    return add(new TrackedObject(region));
}


TrackedObject *TrackedObjectSet::add_observation(Profile *profile,
						 frame_id_t current_frame_id,
						 frame_time_t current_frame_time)
{
    // for now: simply create a new TrackedObject for each Profile
    // FIXME: should compare with old profiles in set and merge so it will not grow
    
    // record that profile is visible and frame that profile was first and last visible
    profile->is_visible = true;
    profile->frame_first_detected =
	profile->frame_last_detected = current_frame_id;
    profile->time_first_detected =
	profile->time_last_detected = current_frame_time;
    
    return add(new TrackedObject(profile));
}

TrackedObject *TrackedObjectSet::add_observation(HumanFeatures *features,
						 frame_id_t current_frame_id,
						 frame_time_t current_frame_time)
{
    // for now: simply create a new TrackedObject for each HumanFeatures
    
    // record that features are visible and frame that features were first and last visible
    features->is_visible = true;
    features->frame_first_detected =
	features->frame_last_detected = current_frame_id;
    features->time_first_detected =
	features->time_last_detected = current_frame_time;
    
    return add(new TrackedObject(features));
}

void TrackedObjectSet::add_observations(RegionSet *region_set,
					frame_id_t current_frame_id,
					frame_time_t current_frame_time)
{
    // for now: simply add each observerd region separately
    // FIXME: should split/merge here!  use Calibration!
    
    ListNode<Region> *curr;
    
    for (curr = region_set->first; curr != NULL; curr = curr->next)
	add_observation(curr->dat,
			current_frame_id,
			current_frame_time);
}


void TrackedObjectSet::add_observations(ProfileSet *profile_set,
					frame_id_t current_frame_id,
					frame_time_t current_frame_time)
{
    // for now: simply add each observerd profile separately
    // FIXME: should compare with all observed regions here!
    
    ListNode<Profile> *curr;
    
    for (curr = profile_set->first; curr != NULL; curr = curr->next)
	add_observation(curr->dat,
			current_frame_id,
			current_frame_time);
}

void TrackedObjectSet::add_observations(HumanFeatureSet *features_set,
					frame_id_t current_frame_id,
					frame_time_t current_frame_time)
{
    // for now: simply add each observerd HumanFeatures separately
    // FIXME: should compare with other tracking data
    
    ListNode<HumanFeatures> *curr;
    
    for (curr = features_set->first; curr != NULL; curr = curr->next)
	add_observation(curr->dat,
			current_frame_id,
			current_frame_time);
}

// check all object `id's.  If they are invalid (0), create new id.
void TrackedObjectSet::check_and_create_ids()
{
    for (start(); current_ok(); forward())
	get_current()->check_and_create_id();
}


// remove (destroy) old objects from set
void TrackedObjectSet::remove_old_tracks(frame_id_t current_frame_id,
					 frame_id_t max_age)
{
    TrackedObject *object;
    
    for (start(); current_ok(); )  // (increment done below)
    {
	object = get_current();
	
	// for each TrackedObject: see if all measurements are older than max_age
	frame_id_t frames_not_detected = FRAME_ID_T_MAX;
	bool object_incorporated_into_background = false;
	
	// FIXME: heavily suboptimal
	for (object->regions->start(); object->regions->current_ok();
	     object->regions->forward())
	{
	    frames_not_detected = min(frames_not_detected,
				      current_frame_id - object->regions->get_current()->frame_last_detected);

	    if (object->regions->get_current()->incorporated_into_background)
		object_incorporated_into_background = true;
	}
	
	for (object->profiles->start(); object->profiles->current_ok();
	     object->profiles->forward())
	    frames_not_detected = min(frames_not_detected,
				      current_frame_id - object->profiles->get_current()->frame_last_detected);
	
	for (object->features->start(); object->features->current_ok();
	     object->features->forward())
	    frames_not_detected = min(frames_not_detected,
				      current_frame_id - object->features->get_current()->frame_last_detected);	
	
	// use age of object to determine whether it is visible
	if (frames_not_detected == 0)
	    object->is_visible = true;
	else
	    object->is_visible = false;
	
	// if too old, destroy whole TrackedObject
	if ((frames_not_detected > max_age) &&
	    (object_incorporated_into_background == false))
	{
	    // too old, destroy()
	    ListNode<TrackedObject> *next_node = current->next;
	    destroy(current);
	    current = next_node;
	}
	else
	{
	    // also, if no data available, destroy
	    if ((get_current()->regions->no_items == 0) &&
		(get_current()->profiles->no_items == 0) &&
		(get_current()->features->no_items == 0))
	    {
		ListNode<TrackedObject> *next_node = current->next;
		destroy(current);
		current = next_node;
	    }
	    else
		forward();
	}
    }
}


// drawing functions: into given Image
void TrackedObjectSet::draw_all_in_image (Image *canvas, frame_id_t min_age)
{
    for (start(); current_ok(); forward())
    {
	// for each TrackedObject: draw
	TrackedObject *object = get_current();
	
	if (object->is_visible == false)
	    continue;
	
	canvas->set_draw_colour(draw_colour_table[object->id % NUM_DRAW_COLOURS][0],
				draw_colour_table[object->id % NUM_DRAW_COLOURS][1],
				draw_colour_table[object->id % NUM_DRAW_COLOURS][2]);
	
	// draw profiles
	object->profiles->draw_in_image(canvas, min_age);

	// draw regions, setting min_age to a maximum of 2
	object->regions->draw_boxes_in_image(canvas, min(min_age, (frame_id_t)2));
    }
}

} // namespace ReadingPeopleTracker
