///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  TrackedObject.h                                                          //
//                                                                           //
//  Storage class for tracked objects (person, group, car, other) holding    //
//    all data from all trackers                                             //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Tue Oct 23 14:26:53 BST 2001                                 //
//  Revision  : 0.0 of Tue Oct 23 14:26:53 BST 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
//  Changes:                                                                 //
//    nts: rev 0.1: initial working revision   The Future                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef __TRACKED_OBJECT_H__
#define __TRACKED_OBJECT_H__

#include "RegionSet.h"
#include "HumanFeatureSet.h"
#include "ProfileSet.h"
#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

class Calibration;

class TrackedObject
{
public:
    
    ObjectType type;

    // id of object/track as a whole, can contain a number of objects with different ids
    object_id_t id;
    
    // included objects and hints (for instance, people contained in a GROUP)
    RegionSet *regions;
    HumanFeatureSet *features;
    ProfileSet *profiles;
    
    Calibration *calibration;
    
    void merge(TrackedObject *other_object);   // ids?

public:

    bool is_visible;   // whether object visible in the current frame
    bool is_static;    // whether object is static in the current frame
    frame_id_t frames_static;       // number of frames object is static
    
    // constructors: Region, Profile (single person), HumanFeatures
    TrackedObject(Region *region, ObjectType obj_type = UNKNOWN);
    TrackedObject(Profile *profile, ObjectType obj_type = PERSON);
    TrackedObject(HumanFeatures *feature, ObjectType obj_type = PERSON);

    // constructor for TrackedObject &List<TrackedObject>::add() in TrackedObjectSet
    TrackedObject ()
	{
	    allocate_datasets();
	    reset_variables();
	};
   
    // destructor
    ~TrackedObject ()
	{
	    if (regions != NULL)  delete regions;
	    if (features != NULL)  delete features;
	    if (profiles != NULL)  delete profiles;
	};

    // = operator
    TrackedObject &operator= (TrackedObject &original);

    object_id_t check_and_create_id();

private:

    void allocate_datasets()  //  allocate data structures (on construction)
	{
	    // allocate data sets
	    regions = new RegionSet;
	    features = new HumanFeatureSet;
	    profiles = new ProfileSet;
	};

    void reset_variables()  // set member variables to default values (on construction)
	{
	    // set variables to default or most likely values...
	    is_visible = true;
	    is_static = false;
	    frames_static = 0;
	    id = 0;
	};

};

// these are needed to declare a TrackedObjectSet
ostream& operator<< (ostream &out, const TrackedObject &obj);
istream& operator>> (istream &in, TrackedObject &obj);  

} // namespace ReadingPeopleTracker

#endif
