//
// Results.h  Storage class for tracking results.
//
// results from tracking will be added by the individual trackers
//

#ifndef __RESULTS_H__
#define __RESULTS_H__

// NB: under WIN32, please use the free ``pthreads for win32'' library,
//     see http://sources.redhat.com/pthreads-win32/
#include <pthread.h>  // for IEEE POSIX 1003.1c --- 1995 threads

#include "TrackedObjectSet.h"
#include "tracker_defines_types_and_helpers.h"  // for results_status_t, frame_id_t etc
#include "OcclusionImage.h"
//#include "OcclusionHandler.h"

namespace ReadingPeopleTracker
{

class Results
{
    friend ostream &operator<< (ostream &out, const Results &results);

private:

    TrackedObjectSet *tracked_objects;
    Image *motion_image;
    Image *background_image;
    Image *difference_image;
    Image *thresholded_difference_image;
    Image *filtered_difference_image;
    
    OcclusionImage *occlusion_image;   // 2D occlusion reasoning
//    WorldOcclusionHandler *occlusion_handler;   // 3D occlusion reasoning

public:
    
    Results()
	{
	    tracked_objects = new TrackedObjectSet;
	    motion_image = NULL;
	    background_image = NULL;
	    difference_image = NULL;
	    thresholded_difference_image = NULL;
	    filtered_difference_image = NULL;
	    occlusion_image = NULL;
//	    occlusion_handler = NULL;
	    
	    frame_id_of_results = FRAME_ID_T_MAX;  // signal no results for current frame (yet)
	    status = RESULTS_UNDEFINED;
	    
	    pthread_cond_init(&status_has_changed, NULL);
	    pthread_mutex_init(&status_modification, NULL);
	}
    
    ~Results()
	{
	    if (tracked_objects != NULL)  delete tracked_objects;
	    if (occlusion_image != NULL)  delete occlusion_image;
//     if (occlusion_handler != NULL)  delete occlusion_handler;  
	    pthread_cond_destroy(&status_has_changed);
	    pthread_mutex_destroy(&status_modification);

	    // NB we do not own the images so we do not delete them
	}

    //
    //  Information about results: used by PeopleTracker and Camera/Tracking/children
    //

    // status of results: undefined, complete (tracked), processed (written out etc)
    results_status_t status;
    
    // condition variable to indicate status changes, eg availability of new tracking data
    pthread_cond_t status_has_changed;

    // mutex for write access to status variable
    pthread_mutex_t status_modification;

    // frame id corresponding to the results
    //   --- set by Camera class when the results generated by Tracking/children are complete
    frame_id_t frame_id_of_results;

    //
    //  Accessors and Modifiers
    //
    inline TrackedObjectSet *get_tracked_objects() const
	{
	    return tracked_objects;
	}

    inline void set_tracked_objects(TrackedObjectSet *new_tracked_objects)
	{
	    tracked_objects = new_tracked_objects;
	}

    inline Image *get_motion_image() const
	{
	    return motion_image;
	}
    inline void set_motion_image(Image *new_image)
	{
	    motion_image = new_image;
	}

    inline Image *get_background_image() const
	{
	    return background_image;
	}
    inline void set_background_image(Image *new_image)
	{
	    background_image = new_image;
	}

    inline Image *get_difference_image() const
	{
	    return difference_image;
	}
    inline void set_difference_image(Image *new_image)
	{
	    difference_image = new_image;
	}

    inline Image *get_thresholded_difference_image() const
	{
	    return thresholded_difference_image;
	}
    inline void set_thresholded_difference_image(Image *new_image)
	{
	    thresholded_difference_image = new_image;
	}

    inline Image *get_filtered_difference_image() const
	{
	    return filtered_difference_image;
	}
    inline void set_filtered_difference_image(Image *new_image)
	{
	    filtered_difference_image = new_image;
	}

    inline OcclusionImage *get_occlusion_image()
	{
	    return occlusion_image;    
	}
    inline void set_occlusion_image(OcclusionImage *in)
	{
	    occlusion_image = in;
	}
    
//      inline WorldOcclusionHandler *get_occlusion_handler()
//    	{
//    	    return occlusion_handler;
//    	}
//      inline void set_occlusion_handler(WorldOcclusionHandler *in)
//    	{
//    	    occlusion_handler = in;
//    	}


    // operator=
    Results &operator= (Results &original);
    

};

// operator<<  ---  write out tracking results in XML
ostream &operator<< (ostream &out, const Results &results);

} // namespace ReadingPeopleTracker

#endif
