//
// Results.cc  Storage class for tracking results.
//
// results from tracking will be added by the individual trackers
//
// .cc file created from .h  nts  Tue Apr 16 16:29:32 BST 2002
//

#include <iostream>

#include "Results.h"

namespace ReadingPeopleTracker
{

// operator<<
//   write out results in XML format to given ostream
ostream &operator<< (ostream &out, const Results &results)
{
    // for now, only write out visible `Profile's
    unsigned int object_number;
    unsigned int profile_number;
    unsigned int visible_profile_count;
    
    TrackedObject *object;
    Profile *profile;
    
    // go through all tracked objects:
    for (object_number = 0; object_number < results.tracked_objects->no_items; object_number++)
    {
	// object = tracked_objects[object_number];
	object = results.tracked_objects->operator[](object_number);
	
	// see how many `Profile's there are.  If there are none, ignore object
	if (object->profiles->no_items == 0)
	    continue;
	
	visible_profile_count = 0;

	for (profile_number = 0; profile_number < object->profiles->no_items; profile_number++)
	{
	    // profile = object->profiles[profile_number];
	    profile = object->profiles->operator[](profile_number);
	    
	    // if profile is not visible, continue
	    if (profile->is_visible == false)
		continue;

	    // found a visible profile.  write out XML description.
	    visible_profile_count++;
	    
	    // if first profile, write "mobile" (tracked object) header
	    if (visible_profile_count == 1)
	    {
		out << "    <mobile id=\"" << object->id
		    // get start time from 1st visible profile  // FIXME: should be min...
		    << "\" start_time=\"" << profile->time_first_detected
		    << "\"" << endl;
	    }
	    
	    // write out profile data
	    out << "      <track id=\"" << object->id
		<< "\" type=\"" << BT_PERSON
		<< "\"" << endl;
	    
	    // info2d: bounding box description (no element members)
	    out << "        <info2d xmin=\"" << profile->xlo
		<< "\" xmax=\"" << profile->xhi
		<< "\" ymin=\"" << profile->ylo
		<< "\" ymax=\"" << profile->yhi
		<< "\" xcog=\"" << real_to_int(profile->origin.x)
		<< "\" ycog=\"" << real_to_int(profile->origin.y)
		<< "\" />" << endl;   // close element
	    
	    // info3d: position of feet on the ground plane (no element members)
	    // FIXME: to be implemented (use from blob input or something)
	    out << "        <info3d x=\"" << 0
		<< "\" y=\"" << 0
		<< "\" z=\"" << 0    // FIXME: not nice.  However, our calibration is ground plane now
		<< "\" width=\"" << 0
		<< "\" height=\"" << 0
		<< "\" />" << endl;   // close element

	    // occlusion: FIXME: to be implemented (use from blob input or something)
	    out << "        <occlusion left=\"" << 0
		<< "\" right=\"" << 0
		<< "\" bottom=\"" << 0
		<< "\" top=\"" << 0
		<< "\" />" << endl;   // close element
	    
	    // close track element
	    out << "      </track>" << endl;
	    
	// if we have found some visible profiles, close "mobile" element
	if (visible_profile_count > 0)
	    out << "    </mobile>" << endl;
	}
    }
    
    return out;
}

// operator=
Results &Results::operator= (Results &original)
{
    if (tracked_objects == NULL)  // should not happen but better check
	tracked_objects = new TrackedObjectSet;
    
    *tracked_objects = *original.tracked_objects;  // using its operator=
    
    // de-allocate current images if they exist
    if (motion_image != NULL)
        delete motion_image;
    if (background_image != NULL)
        delete background_image;
    if (difference_image != NULL)
        delete difference_image;
    if (thresholded_difference_image != NULL)
        delete thresholded_difference_image;
    if (filtered_difference_image != NULL)
        delete filtered_difference_image;
    
    // copy images over if they exist, else set to NULL
    if (original.motion_image != NULL)
	motion_image = original.motion_image->copy();
    else
	motion_image = NULL;
    
    if (original.background_image != NULL)
	background_image = original.background_image->copy();
    else
	background_image = NULL;
    
    if (original.difference_image != NULL)
	difference_image = original.difference_image->copy();
    else
	difference_image = NULL;
    
    if (original.thresholded_difference_image != NULL)
	thresholded_difference_image = original.thresholded_difference_image->copy();
    else
	thresholded_difference_image= NULL;
    
    if (original.filtered_difference_image != NULL)
	filtered_difference_image = original.filtered_difference_image->copy();
    else
	filtered_difference_image = NULL;
    
    // copy status and frame_id_of_results as we have copied the data
    status = original.status;
    frame_id_of_results = original.frame_id_of_results;
    
    // signal to any waiting thread that our status might be changed
    //   This is important since our data HAS indeed changed.
    //   The waiting thread is supposed to check status and
    //   frame_id_of_results to see whether the data is actually usable
    //
    //   this requires exclusive write access to status
    pthread_mutex_lock(&status_modification);
    //
    pthread_cond_signal(&status_has_changed);
    //
    pthread_mutex_unlock(&status_modification);
   
    return *this;
}

} // namespace ReadingPeopleTracker
