///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  RegionTracker.cc                                                         //
//                                                                           //
//  This tracking class tracks regions from frame to frame                   //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Mon Oct 22 11:21:21 BST 2001                                 //
//  Revision  : 0.0 of Mon Oct 22 11:21:21 BST 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <cassert>

#include "RegionTracker.h"

#ifndef NO_DISPLAY
#ifdef DEBUG
#include "os_specific_things.h"  // for sleep()
#endif
#endif

#include "Tracking.h"
#include "MotionDetector.h"
#include "Inputs.h"
#include "text_output.h"
#include "Results.h"
#include "MultiBackgroundSource.h"

#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

static const char *RegionTracker_Revision = "@(#) RegionTracker.cc, rev 0.0 of Mon Oct 22 11:21:21 BST 2001, Author Nils T Siebel, Copyright (c) 2001 The University of Reading";


void RegionTracker::predict_old_objects(Results *results)
{
    cdebug << "RT: RegionTracker::predict_old_objects() " << endl;
    
    ListNode<TrackedObject> *curr_obj;
    frame_id_t frame_id = results->get_motion_image()->get_frame_id();
    
    // get regions from tracked object set
    for (curr_obj = results->get_tracked_objects()->first; curr_obj != NULL;
	 curr_obj = curr_obj->next)	
    {
	// predict all regions for this object
	ListNode<Region> *curr;
	
	for (curr = curr_obj->dat->regions->first; curr != NULL; curr = curr->next)
	{
	    // if object integrated into the background, do not change predictions
	    if (curr->dat->incorporated_into_background)
	    {
		curr->dat->source = PREDICTION;
		curr->dat->is_visible = true;
		// curr->dat->frame_last_detected = frame_id - 1;
		
		assert(curr_obj->dat->regions->no_items == 1);  // FIXME: other cases to be implemented
		curr_obj->dat->frames_static++;
		
		continue;
	    }
	    
	    // do not predict new objects (from MotionDetector)
	    if (curr->dat->frame_first_detected == frame_id)
	    {
		cdebug << "RT: RegionTracker::predict_old_objects(): new region? "
		       << "object id " << curr_obj->dat->id << " :- "
		       << endl << *curr->dat << endl;
		
		assert(curr_obj->dat->id == 0); // new measurements have not yet been assigned an id
		
		curr->dat->old_origin = curr->dat->origin;
		curr->dat->direction = Point2(0,0);
		assert (curr->dat->source == MEASUREMENT); // this should have been set earlier
		assert (curr->dat->is_visible == true); // this should have been set earlier
		
		cdebug << "RT: RegionTracker::predict_old_objects(): changed to :- "
		       << endl << *curr->dat << endl;
		continue;
	    }
	    
	    cdebug << "RT: RegionTracker::predict_old_objects(): old region "
		   << "object id " << curr_obj->dat->id << " :- "
		   << endl << *curr->dat << endl;
	    
	    // get new `direction' guess if there are enough meausurements
	    if (curr->dat->frame_last_detected + 1 == frame_id)
	    {
		// detected previous frame, make proper prediction
		cdebug << "RT: RegionTracker::predict_old_objects(): "
		       << "region detected in previous frame, adjusting direction "
		       << endl;
		
		assert(curr->dat->source == MEASUREMENT); // otherwise data is inconsistent
		
		// check whether the update of direction would give a good result
		if ((curr->dat->old_origin == curr->dat->origin) &&
		    (curr->dat->direction != Point2(0,0)))
		{
		    // probably no real measurement (what else?), just slow down prediction
//		    curr->dat->direction /= 2;
		}
		else
		{
		    // we have a valid (measured) new origin so update direction guess
		    curr->dat->direction = curr->dat->origin - curr->dat->old_origin;
		}
	    }
	    else
	    {
		cdebug << "RT: RegionTracker::predict_old_objects(): not seen in previous frame "
		       << frame_id - 1 << " but in " << curr->dat->frame_last_detected << endl;
		assert(curr->dat->source == PREDICTION); // otherwise data is inconsistent
	    }
	    
	    // do actual prediction: very simple constant velocity assumption
	    curr->dat->old_origin = curr->dat->origin;
	    curr->dat->origin += curr->dat->direction;		
	    
	    // mark region as predicted
	    curr->dat->source = PREDICTION;
	    curr->dat->is_visible = false;  // we do not know yet
	    
	    cdebug << "RT: RegionTracker::predict_old_objects(): "
		   << "predicted region in object " << curr_obj->dat->id
		   << ": direction = " << curr->dat->direction
		   << endl
		   << " now object looks like this: " << endl
		   << *curr->dat << endl;
	}
    }
}

void RegionTracker::track_old_objects(Inputs *inputs, Results *results)
{
    cdebug << "RT: RegionTracker::track_old_objects() " << endl;    
    
    // first, initialise all matched tags to false.  As we match predictions
    // to measurements in split_and_merge_regions() and match_regions_in_objects(),
    // these will be queried and set to false later.
    assert (false == 0); // otherwise we cannot use memset in the following way
    memset ((void *) &matched_tag, 0, sizeof matched_tag);
    
    // split and merge measurements and match these hypotheses to predictions
    // using a constructive approach
    split_and_merge_regions(results);
    
    // calculate a region difference measure to compare preditions with measurements
    calculate_region_differences(results);
    
    // match the measurements not split/merged to the predictions not matched
    match_regions_in_objects(inputs, results);
    
}


void RegionTracker::detect_new_objects(Inputs *inputs, Results *results)
{
    cdebug << "RT: RegionTracker::detect_new_objects() " << endl;
    
    // all new objects are already there and tracked (ie compared
    //  to predictions) --- at the same time as the old objects.
}

// tracking loop for each new image
void RegionTracker::process_frame(Inputs *inputs,
				  Results *results,
				  unsigned int max_objects)
{
    predict_old_objects(results);    
    track_old_objects(inputs, results);
    detect_new_objects(inputs, results);
}


//  realno track_distance(Region* reg1, Region* reg2)
//  {
//      Point2 diff = reg1->origin - reg2->origin;
//      realno w = (reg1->width + reg2->width) / 2;
//      realno h = (reg1->height + reg2->height) / 2;
//    
//      realno dw = fabs(reg1->width - reg2->width) / w;
//      realno dh = fabs(reg1->height - reg2->height) / h;
//      realno dl = diff.length() / fmax(w,h);
//    
//      if (dw > 1.0 || dh > 1.0 || dl > 1.0)
//  	return 1000000;
//    
//      return dw + dh + dl;
//  }

void RegionTracker::register_configuration_parameters()
{
    // FIXME: this is a replication of code in BaseTracker::BaseTracker(). Why is this necessary???
    max_objects = 
	configuration.register_int("MAX_OBJECTS", 32,
				   (int *) &max_objects,  false,
				   "RegionTracker", 
				   "The maximum number of regions to track");
    
    debug_level =
	configuration.register_int("DEBUG_LEVEL", 0,
				   (int *) &debug_level, false,
				   "RegionTracker",
				   "General debug level");
    
    // our things
    max_match_difference = 
	configuration.register_int("MAX_MATCH_DIFFERENCE", 70,
				   (int *) &max_match_difference, true,
				   "RegionTracker",
				   "Maximum pixel difference between regions to match");
    
    max_merge_difference = 
	configuration.register_real("MAX_MERGE_DIFFERENCE", 30.0,
				    &max_merge_difference, true,
				    "RegionTracker",
				    "Maximum pixel difference between regions to split a measured moving region into 2 in order to match predictions ");
    
    static_count_incorporate =
	configuration.register_int("STATIC_COUNT_INCORPORATE", 5,
				   (int *) &static_count_incorporate, true,
				   "RegionTracker",
				   "Incorporate static objects into the background after this many frames.");
}

RegionTracker::RegionTracker (Calibration *the_calibration,
			      MotionDetector *the_motion_detector,
			      char *config_filename)
{
    calibration = the_calibration;
    motion_detector = the_motion_detector;
    
    // register configuration parameters with configuration manager
    register_configuration_parameters();
    
    // read values from configuration file
    if (config_filename != NULL)
	configuration.parse_parameter_file(config_filename);
    else
	cerror << "RT: RegionTracker: Warning: No parameter file specified. "
	       << "using default parameter values. \n";
}

// split and merge measurements in order to create hypotheses matching the
//    predictions, using a constructive approach.
// newly created hypotheses are marked as source = SPLITTING and added to
//    our observations --- no existing regions are matched/changed yet.
void RegionTracker::split_and_merge_regions(Results *results)
{
    cdebug << "RT: RegionTracker::split_and_merge_regions() " << endl;
    cdebug << "     FIXME: only splitting into 2 for now " << endl;
    cdebug << "     FIXME: only splitting for now " << endl;
    
    // Try to match pairs of predicted regions predicted to be close to each
    //   other to measurements which could be containing both blobs.
    //
    // If matched, the measurement will be removed and `matched_tag's set.
    
    // assumes that matched_tag [a] has been set to false for all non-matched regions a
    
    TrackedObjectSet *objects = results->get_tracked_objects();
    
    int index1, index2, index3;  // indices into the TrackedObjectSet *objects
    
    Region *prediction1;   // matching...
    Region *prediction2;   //     ...
    Region *measurement;   // ...objects
    
    const int distance_epsilon = 20;  // maximum distance to consider regions close
    
    //
    // get all pairs of predictions that are close enough to be considered
    //
    for (index1 = 0; index1 + 1 < objects->no_items; index1++)
    {
	TrackedObject *obj1 = (*objects)[index1];
	
	// make sure obj1 has regions
	if (obj1->regions->no_items == 0)
	    continue;
	
	// don't try to match regions already matched
	if (matched_tag[index1])
	    continue;
	
	// get region prediction1
	prediction1 = obj1->regions->first->dat;  // FIXME: don't use first only
	
	// prediction1 should be a prediction
	if (prediction1->source != PREDICTION)
	    continue;	
	
	if (prediction1->incorporated_into_background)  // do not split to match these // FIXME: ?
	    continue;
	
	// in order to try all pairs of predictions: get a 2nd one
	for (index2 = index1 + 1; index2 < objects->no_items; index2++)
	{
	    TrackedObject *obj2 = (*objects)[index2];
	    
	    // make sure obj2 has regions
	    if (obj2->regions->no_items == 0)
		continue;
	    
	    // don't try to match regions already matched
	    if (matched_tag[index2])
		continue;
	    
	    // get region prediction2
	    prediction2 = obj2->regions->first->dat;  // FIXME: don't use first only
	    
	    // prediction2 should be a prediction
	    if (prediction2->source != PREDICTION)
		continue;
	    
	    if (prediction2->incorporated_into_background)  // do not split to match these // FIXME: ?
		continue;
	    
	    // don't try to combine regions for matching unless they are close...
	    int hdistance;
	    int vdistance;
	    
	    // calculate horizontal and vertical distance
	    region_distance(prediction1, prediction2, &hdistance, &vdistance);
	    
	    if (vdistance > distance_epsilon)
		continue;
	    
	    //
	    //  now we have two close predictions.  combine into one:
	    //
	    realno combined_xlo = min(prediction1->xlo, prediction2->xlo);
	    realno combined_xhi = max(prediction1->xhi, prediction2->xhi);
	    realno combined_ylo = min(prediction1->ylo, prediction2->ylo);
	    realno combined_yhi = max(prediction1->yhi, prediction2->yhi);
	    Region combined_region((int) combined_xlo, (int) combined_xhi,
				   (int) combined_ylo, (int) combined_yhi);
	    
	    // FIXME: what happens if static objects are matched?
	    // FIXME: check visibility?  could be ok without!
	    
	    //
	    //  compare combined regions to all measurements:
	    //
	    
	    for (index3 = 0; index3 < objects->no_items; index3++)
	    {
		TrackedObject *obj3 = (*objects)[index3];
		
		// make sure obj3 has regions
		if (obj3->regions->no_items == 0)
		    continue;
		
		// don't try to match regions already matched
		if (matched_tag[index3])
		    continue;
		
		// get region measurement
		measurement = obj3->regions->first->dat;  // FIXME: don't use first only
		
		// measurement should be a measurement
		if (measurement->source != MEASUREMENT)
		    continue;
		
		assert(obj3->regions->no_items == 1);   // assume "pure" new measurement
		assert(obj3->profiles->no_items == 0);  // anything else would make no sense
		
		// check the region difference between combined region and measurement
		const realno region_diff = region_difference(&combined_region, measurement);
		
		if (region_diff > max_merge_difference)
		    continue;  // to far apart => mismatch, try with next measurement
		
		//
		//  combined regions do match new measurement.
		//  now create "synthetic" measurements by splitting the large region into 2
		//
		cdebug << "RT: RegionTracker::split_and_merge_regions(): "
		       << "combined regions match new measurement! " << endl;
		
		cdebug << "--- prediction 1: object id " << obj1->id << " " << endl
		       << *obj1->regions->first->dat << endl;
		cdebug << "--- prediction 2: object id " << obj2->id << " " << endl
		       << *obj2->regions->first->dat << endl;
		cdebug << "--- measurement: object id " << obj3->id << "  " << endl
		       << *obj3->regions->first->dat << endl;
		// create new TrackedObject, copying the data from prediction1
		TrackedObject *new_obj1 = objects->add_a_copy(obj1);
//		new_obj1->id = 0;                                     // but not the id
		Region *new_reg1 = new_obj1->regions->first->dat;     // get the region
		
		// create new TrackedObject, copying the data from prediction1
		TrackedObject *new_obj2 = objects->add_a_copy(obj2);
//		new_obj2->id = 0;                                     // but not the id
		Region *new_reg2 = new_obj2->regions->first->dat;     // get the region
		
		// adjust positions of new_reg1 and new_reg2, assuming size has not changed
		// FIXME: maybe it has become smaller?  is that possible?
		
		// let edges "snap" to edges of measurement
		const int max_adjustment = 50;   // max pos. shift in pixels, outward only
		int delta_x_l, delta_x_r, delta_y_l, delta_y_r;  // delta x/y left/right
		
		// new_reg1: let edges "snap" to edges of measurement
		//  1 - shift horizontally
		delta_x_l = measurement->xlo - new_reg1->xlo;
		delta_x_r = measurement->xhi - new_reg1->xhi;
		
		if ((delta_x_l > -max_adjustment) &&    // limit outward movement only
		    (abs(delta_x_l) < abs(delta_x_r)))  // we are closer to the left
		{
		    // shift horizontally to match measurement->xlo
		    new_reg1->xlo += delta_x_l;
		    new_reg1->xhi += delta_x_l;
		    new_reg1->origin.x += delta_x_l;
		    
		    // update movement prediction
		    new_reg1->old_origin.x = new_reg1->origin.x - delta_x_l;
		    new_reg1->direction.x = delta_x_l;
		}
		else
		    if ((delta_x_r < max_adjustment) &&      // limit outward movement only 
			(abs(delta_x_l) >= abs(delta_x_r)))  // we are closer to the right
		    {
			// shift horizontally to match measurement->xhi
			new_reg1->xlo += delta_x_r;
			new_reg1->xhi += delta_x_r;
			new_reg1->origin.x += delta_x_r;
			
			// update movement prediction
			new_reg1->old_origin.x = new_reg1->origin.x - delta_x_r;
			new_reg1->direction.x = delta_x_r;
		    }
		
		//  2 - shift vertically
		delta_y_l = measurement->ylo - new_reg1->ylo;
		delta_y_r = measurement->yhi - new_reg1->yhi;
		
		if ((abs(delta_y_l) < max_adjustment) &&
		    (abs(delta_y_l) < abs(delta_y_r)))
		{
		    // shift vertically to match measurement->ylo
		    new_reg1->ylo += delta_y_l;
		    new_reg1->yhi += delta_y_l;
		    new_reg1->origin.y += delta_y_l;
		}
		else
		    if (abs(delta_y_r) < max_adjustment)
		    {
			// shift vertically to match measurement->yhi
			new_reg1->ylo += delta_y_r;
			new_reg1->yhi += delta_y_r;
			new_reg1->origin.y += delta_y_r;
			
			// update movement prediction
			new_reg1->old_origin.y = new_reg1->origin.y - delta_y_r;
			new_reg1->direction.y = delta_y_r;
		    }
		
		// new_reg2: let edges "snap" to edges of measurement
		//  1 - shift horizontally
		delta_x_l = measurement->xlo - new_reg2->xlo;
		delta_x_r = measurement->xhi - new_reg2->xhi;
		
		if ((delta_x_l > -max_adjustment) &&    // limit outward movement only
		    (abs(delta_x_l) < abs(delta_x_r)))  // we are closer to the left
		{
		    // shift horizontally to match measurement->xlo
		    new_reg2->xlo += delta_x_l;
		    new_reg2->xhi += delta_x_l;
		    new_reg2->origin.x += delta_x_l;
		    
		    // update movement prediction
		    new_reg2->old_origin.x = new_reg2->origin.x - delta_x_l;
		    new_reg2->direction.x = delta_x_l;
		}
		else
		    if ((delta_x_r < max_adjustment) &&      // limit outward movement only 
			(abs(delta_x_l) >= abs(delta_x_r)))  // we are closer to the right
		    {
			// shift horizontally to match measurement->xhi
			new_reg2->xlo += delta_x_r;
			new_reg2->xhi += delta_x_r;
			new_reg2->origin.x += delta_x_r;
			
			// update movement prediction
			new_reg2->old_origin.x = new_reg2->origin.x - delta_x_r;
			new_reg2->direction.x = delta_x_r;
		    }
		
		//  2 - shift vertically
		delta_y_l = measurement->ylo - new_reg2->ylo;
		delta_y_r = measurement->yhi - new_reg2->yhi;
		
		if ((abs(delta_y_l) < max_adjustment) &&
		    (abs(delta_y_l) < abs(delta_y_r)))
		{
		    // shift vertically to match measurement->ylo
		    new_reg2->ylo += delta_y_l;
		    new_reg2->yhi += delta_y_l;
		    new_reg2->origin.y += delta_y_l;
		}
		else
		    if (abs(delta_y_r) < max_adjustment)
		    {
			// shift vertically to match measurement->yhi
			new_reg2->ylo += delta_y_r;
			new_reg2->yhi += delta_y_r;
			new_reg2->origin.y += delta_y_r;
			
			// update movement prediction
			new_reg2->old_origin.y = new_reg2->origin.y - delta_y_r;
			new_reg2->direction.y = delta_y_r;
		    }
		
		// adjust region image if necessary: new_reg1
		if (new_reg1->region_img != NULL)
		{
		    delete new_reg1->region_img;
		    Image *motion_image = results->get_motion_image();
		    
  		    unsigned int xmin = max(0, new_reg1->xlo);
  		    unsigned int xmax = min(motion_image->get_width() - 1,
					    (unsigned int) new_reg1->xhi);
  		    unsigned int ymin = max(0, new_reg1->ylo);
  		    unsigned int ymax = min(motion_image->get_height() - 1,
					    (unsigned int) new_reg1->yhi);
		    
  		    new_reg1->region_img =
  			motion_image->extract_subimage(xmin,xmax,ymin,ymax);
		}
		
		// adjust region image if necessary: new_reg2
		if (new_reg2->region_img != NULL)
		{
		    delete new_reg2->region_img;
		    Image *motion_image =
			results->get_motion_image();
		    
  		    unsigned int xmin = max(0, new_reg2->xlo);
  		    unsigned int xmax = min(motion_image->get_width() - 1,
					    (unsigned int) new_reg2->xhi);
		    unsigned int ymin = max(0, new_reg2->ylo);
  		    unsigned int ymax = min(motion_image->get_height() - 1,
					    (unsigned int) new_reg2->yhi);
		    
  		    new_reg2->region_img =
  			motion_image->extract_subimage(xmin,xmax,ymin,ymax);
		}
		
		// adjust status
		new_reg1->frame_last_detected =
		    new_reg2->frame_last_detected =
		    measurement->frame_last_detected;             // probably now
		
		new_reg1->time_last_detected =
		    new_reg2->time_last_detected =
		    measurement->time_last_detected;             // probably now
		
		new_reg1->is_visible =
		    new_reg2->is_visible =
		    measurement->is_visible;                      // probably true
		
		new_reg1->source =
		    new_reg2->source = SPLITTING;
		
		// NB: also adjust the measurement's source so it might be removed later
		measurement->source = SPLITTING;
		
#ifdef DEBUG
		cdebug << "--- split 1:     " << endl << *new_obj1->regions->first->dat << endl;
		cdebug << "--- split 2:     " << endl << *new_obj2->regions->first->dat << endl;
		cdebug << "--- measurement: " << endl << *obj3->regions->first->dat << endl;
		
#ifndef NO_DISPLAY
		if (debug_level == 1)
		{
		    // draw predictions
		    cdebug << " drawing: " << flush;
// FIXME		    inputs->get_video_image()->display();   // ie clear...
		    linewidth(1);
		    Image::set_colour(obj1->id);
		    cdebug << "prediction1, " << flush;
		    prediction1->draw_box();
		    gflush();
		    sleep(2);
		    Image::set_colour(obj2->id);
		    cdebug << "prediction2, " << flush;
		    prediction2->draw_box();
		    gflush();
		    sleep(2);
		    
		    // draw measurement
		    Image::set_colour(0);
		    cdebug << "measurement, " << flush;
		    measurement->draw_box();
		    gflush();
		    sleep(2);
		    
		    // draw new (split) regions
		    linewidth(2);
		    Image::set_colour(0);
		    cdebug << "split1, " << flush;
		    new_reg1->draw_box();
		    gflush();
		    sleep(2);
		    cdebug << "split2. " << endl << flush;
		    new_reg2->draw_box();
		    gflush();
		    sleep(5);
		}
#endif   // ifndef NO_DISPLAY
#endif   // ifdef DEBUG
	    }
	}
    }
}

void RegionTracker::calculate_region_differences(Results *results)
{
    cdebug << "RT: RegionTracker::calculate_region_differences() " << endl;
    
    // calculate a difference measure for all pairs (r1,r2) of regions in the
    //   TrackedObjectSet (objects), r1 in obj1 and r2 in obj2 with
    //   obj1 being a PREDICTION and obj2 being a MEASUREMENT such that
    //   region_diff[index1][index2] will give a reasonable difference measure for
    //   r1 and r2, with obj1 = (*objects) [index1] and obj2 = (*objects) [index2].
    
    TrackedObjectSet *objects = results->get_tracked_objects();
    
    int index1, index2;  // indices into the TrackedObjectSet *objects
    
    //  calculate differences for all suitable pairs of objects/regions
    for (index1 = 0; index1 < objects->no_items; index1++)
    {
	TrackedObject *obj1 = (*objects)[index1];
	
	// make sure obj1 has regions
	if (obj1->regions->no_items == 0)
	    continue;
	
	// get region r1
	Region *r1 = obj1->regions->first->dat;  // FIXME: don't use first only
	
	// r1 should be a prediction
	if (r1->source != PREDICTION)
	    continue;	
	
	// match to all regions in set2
	for (index2 = 0; index2 < objects->no_items; index2++)
	{
	    // don't match object to itself
	    if (index1 == index2)
		continue;
	    
	    TrackedObject *obj2 = (*objects)[index2];
	    
	    // make sure obj2 has regions
	    if (obj2->regions->no_items == 0)
		continue;
	    
	    // get region r2
	    Region *r2 = obj2->regions->first->dat;  // FIXME: don't use first only
	    
	    // r2 should be a measurement or derived from splitting/merging a measurement
	    if ((r2->source != MEASUREMENT) && (r2->source != SPLITTING))
		continue;
	    
	    // now calculate difference measure using position, width and height
	    region_diff [index1] [index2] = region_difference(r1,r2);
	    
	    // add a penalty for static objects so they won't be matched too easily
	    //   (situation: objects moving close to a static one)
	    if (r1->incorporated_into_background)
		region_diff [index1] [index2] += 15;   // FIXME: how about a calculation?
	    
//	    cdebug << " calculated region_diff[" << index1 << "] [" << index2
//		 << "] is " << region_diff [index1] [index2] << endl;
	}
    }
}


// match the measurements not split/merged to the predictions not matched
void RegionTracker::match_regions_in_objects(Inputs *inputs, Results *results)
{
    cdebug << "RT: RegionTracker::match_regions_in_objects() " << endl;
    
    //
    // use the previously calculated region difference measures (in region_diff[])
    //   to find pairs (pred_region,meas_region) of regions in a pair
    //   (pred_object,meas_object) of objects in the TrackedObjectSet objects,
    //   pred_region in pred_object and meas_region in meas_object with
    //   pred_object being a PREDICTION and meas_object being a MEASUREMENT or
    //   created by SPLITTING, and have a look at region_diff[index1][index2]
    //   to see whether the regions match.
    //
    // tracked objects where the regions are matched are assumed to be the same,
    //   the predicted region will be replaced by the new region measurement
    
    // assumes that matched_tag [r] has been set to false for all non-matched regions r
    
    TrackedObjectSet *objects = results->get_tracked_objects();
    
    // pair tested for match: indices, objects and regions
    unsigned int pred_index;  // indices into the TrackedObjectSet *objects
    unsigned int meas_index;
    TrackedObject *pred_object;
    TrackedObject *meas_object;
    Region *pred_region;
    Region *meas_region;
    
    // matching pair: indices, objects and regions
    unsigned int matched_pred_index = 0;
    unsigned int matched_meas_index = 0;
    TrackedObject *matched_pred_object = NULL;
    TrackedObject *matched_meas_object = NULL;
    Region *matched_pred_region = NULL;
    Region *matched_meas_region = NULL;
    
    unsigned int num_matched;
    realno min_difference_so_far;
    bool matched_static;           // whether we matched (and should revive) a static object
    
    // find all matches, best matches first
    do
    {
	num_matched = 0;
	min_difference_so_far = 10000;
	matched_static = false;
	
	for (pred_index = 0; pred_index < objects->no_items; pred_index++)
	{
	    pred_object = (*objects)[pred_index];
	    
	    // make sure pred_object has regions
	    if (pred_object->regions->no_items == 0)
		continue;
	    
	    // don't match regions that have been matched already
	    if (matched_tag[pred_index])  // (could be set by split_and_merge_regions())
		continue;
	    
	    // get region pred_region
	    pred_region = pred_object->regions->first->dat;  // FIXME: don't use first only
	    
	    // pred_region should be a prediction
	    if (pred_region->source != PREDICTION)
		continue;
	    
	    // search all measurements meas_region for combinations (pred_region,meas_region) with minimum/best fit
	    for (meas_index = 0; meas_index < objects->no_items; meas_index++)
	    {
		// don't match object to itself
		if (pred_index == meas_index)
		    continue;
		
		meas_object = (*objects)[meas_index];
		
		// make sure meas_object has regions
		if (meas_object->regions->no_items == 0)
		    continue;
		
		// don't match to regions that have been matched already
		if (matched_tag[meas_index])
		    continue;
		
		// get region meas_region
		meas_region = meas_object->regions->first->dat; // FIXME: don't use first only
		
		// meas_region should be a measurement or derived from splitting/merging measurements
		if ((meas_region->source != MEASUREMENT) && (meas_region->source != SPLITTING))
		    continue;
		
		assert((meas_object->id == 0) || (meas_region->source == SPLITTING)); // check data consistency
		
		// get difference measure from table
		realno difference = region_diff [pred_index] [meas_index];
		
		// check whether difference is the best (minimum) difference so far
		if ((difference < max_match_difference) &&
		    (difference < min_difference_so_far))
		{
		    // match found (best match so far, more to compare...)
		    min_difference_so_far = difference;
		    
		    // remember matching pair...
		    matched_pred_index = pred_index;
		    matched_meas_index = meas_index;
		    matched_pred_object = pred_object;
		    matched_meas_object = meas_object;
		    matched_pred_region = pred_region;
		    matched_meas_region = meas_region;
		    
		    num_matched = 1;
		    matched_static = pred_object->is_static;
		}
  		else
  		{
#ifdef DEBUG
		    cdebug << "RT: RegionTracker::match_regions_in_objects(): "
			   << "ignored bad match dist " << difference
			   << " betw pred " << pred_object->id << " idx " << pred_index;
		    if (meas_region->source == MEASUREMENT)
			cdebug << " to meas ";
		    else
			cdebug << " to split ";
		    cdebug << meas_object->id << " idx " << meas_index << endl;
#endif
		}
	    }
	}
	
	//
	//  if we have found a match we replace the prediction by the measurement
	//
	if (num_matched > 0)  // found a match!
	{
#ifdef DEBUG
	    cdebug << "RT: RegionTracker::match_regions_in_objects(): "
		   << "matched pred " << matched_pred_object->id 
		   << " idx " << matched_pred_index;
	    if (matched_meas_region->source == MEASUREMENT)
		cdebug << " to meas ";
	    else
		cdebug << " to split ";
	    cdebug << matched_meas_object->id
		   << " idx " << matched_meas_index
		   << " at difference " << min_difference_so_far << ": " 
		   << endl << "---" << endl;
	    cdebug << *matched_pred_region << "---" << endl;
	    cdebug << *matched_meas_region << "---" << endl;
#endif	    
	    
	    // tag to remember match so we won't match it again
	    matched_tag [matched_pred_index] = true;
	    matched_tag [matched_meas_index] = true;
	    
	    bool object_is_static;
	    
	    // check for static regions and incorporate them
	    //   into the background
	    cdebug << "RT: RegionTracker::match_regions_in_objects(): "
		   << " pred id " << matched_pred_object->id
		   << " diff " << min_difference_so_far
		   << " Euklidean dist "
		   << hypot(matched_pred_region->old_origin.x - matched_meas_region->origin.x,
			    matched_pred_region->old_origin.y - matched_meas_region->origin.y)
		   << " hypoth off by "
		   << hypot(matched_meas_region->origin.x - matched_pred_region->origin.x,
			    matched_meas_region->origin.y - matched_pred_region->origin.y)
		   << ", matched_static == " << matched_static
		   << endl;
	    
	    // check whether we have detected a static object which is not marked as such
//  	    if (hypot(matched_pred_region->old_origin.x - matched_meas_region->origin.x,
//  		      matched_pred_region->old_origin.y - matched_meas_region->origin.y)
	    if (min_difference_so_far
		< fmin(6, fmax(3,matched_meas_region->width / 10)))  // close enough?
	    {
		// the region has not moved much -> is static!
		object_is_static = true;
		matched_pred_object->frames_static++;
		
		cdebug << "RT: RegionTracker::match_regions_in_objects(): "
		       << "found static object id " << matched_pred_object->id
		       << " time static " << matched_pred_object->frames_static
		       << endl;
	    }
	    else
	    {
		object_is_static = false;
		matched_pred_object->frames_static = 0;  // reset
		
		// object is moving.  check whether it has been static an is in background
		if (matched_pred_region->incorporated_into_background)  // started moving again!
		{
		    // remove revived profile from background if MultiBackgroundSource is used
		    if (motion_detector->get_multi_background() != NULL)
		    {
			motion_detector->get_multi_background()
			    -> remove_region(matched_pred_region);
			
			cdebug << "RT: RegionTracker::match_regions_in_objects(): "
			       << "one static object removed from background "
			       << endl;
		    }
		    matched_pred_region->incorporated_into_background = false;
		}
	    }
	    
	    //
	    //  now correct matched prediction
	    //
	    
	    //  copy over new measurements (position, size) to old match
	    //  so we can destroy the new one and use the old	    
	    
	    
	    // use Observation::operator= and do not copy over everything in Region...
	    // NB do not copy over Region::old_origin or Region::incorporated_into_background
	    //    and not frame_first_detected or frame_last_detected
	    
	    // remember some values before copying the rest...
	    frame_id_t old_frame_first_detected =
		matched_pred_region->frame_first_detected;

	    frame_time_t old_time_first_detected =
		matched_pred_region->time_first_detected;

    	    matched_pred_region->Observation::operator=(*matched_meas_region);
	    
	    // set values not set correctly by the operator
	    matched_pred_region->frame_first_detected = old_frame_first_detected;
	    matched_pred_region->time_first_detected = old_time_first_detected;

	    matched_pred_object->is_visible = matched_meas_object->is_visible;   // ie true
	    
	    if (object_is_static)
	    {
		// make no prediction next time to distinguish static and slowly moving objects
		matched_pred_region->old_origin = matched_pred_region->origin;
		matched_pred_region->direction = Point2(0,0);
	    }	    
	    
	    // region image...
	    if (matched_pred_region->region_img != NULL)
		delete matched_pred_region->region_img;
	    
	    matched_pred_region->region_img = matched_meas_region->region_img;
	    matched_meas_region->region_img = NULL;
	    
	    // boundary points...
	    if (matched_pred_region->region_boundary != NULL)
		delete matched_pred_region->region_boundary;
	    
	    matched_pred_region->region_boundary = matched_meas_region->region_boundary;
	    matched_meas_region->region_boundary = NULL;
	    
	    matched_pred_region->source = MEASUREMENT;
	    
	    // keep associations with other type measurements by moving them over...
	    assert(matched_meas_object->regions->no_items == 1); // FIXME: other cases to be implemented
	    matched_pred_object->profiles  -> concat(matched_meas_object->profiles);
	    matched_pred_object->features  -> concat(matched_meas_object->features);  
	    
	    if (object_is_static)
	    {
		//
		// check MultiBackground features if available
		//
		if ((motion_detector->get_multi_background() != NULL) &&
		    (matched_pred_object->frames_static > static_count_incorporate) &&
		    (! matched_pred_region->incorporated_into_background))
		{
		    // temporarily incorporate region into background
		    
		    // in order to add the static object into the background we need
		    //   to create a Region with an Image of the area
		    
		    Image *video_image = inputs->get_video_image();
		    
		    int xmin = max (0, matched_pred_region->xlo - 3);
		    int xmax = min ((int) video_image->get_width() - 1,
				    matched_pred_region->xhi + 3);
		    int ymin = max (0, matched_pred_region->ylo - 3);
		    int ymax = min ((int) video_image->get_height() - 1,
				    matched_pred_region->yhi + 3);
		    
		    Image *static_image =
			video_image->extract_subimage(xmin,xmax,ymin,ymax);
		    
		    Region *static_region = 
			new Region(xmin,xmax,ymin,ymax,static_image);
		    
		    // as static objects in MultiBackgroundSource are identified by
		    //   the origin location, set it to the same value
		    static_region->origin = matched_pred_region->origin;
		    
		    matched_pred_object->frames_static--; // sorry. for better readability elsewhere
		    
		    motion_detector->get_multi_background()->add_region(static_region); // uses List::add()
		
		    cdebug << "RT: RegionTracker::match_regions_in_objects(): "
			   << "static object incorporated into background :- "
			   << endl
			   << *matched_pred_object;
		    
		    matched_pred_region->incorporated_into_background = true;
  		}
	    }
	    
	    //
	    // finally, mark new TrackedObject for deletion simply by deleting
	    //   the RegionSet.  NB don't delete TrackedObject or otherwise meas_index
	    //   becomes incorrect (index on difference table).
	    //
	    matched_meas_object->regions->destroy_all();  // FIXME: What if there is more than 1
	}
    }
    while (num_matched > 0);
    
    cdebug << endl;
}

// calculate a difference measure between two regions, considering positions as well as sizes
realno RegionTracker::region_difference(const Region *r1, const Region *r2)
{
    return (hypot((r1->origin.x - r2->origin.x), (r1->origin.y - r2->origin.y)) +
	    hypot(1.5*(r1->width - r2->width), (r1->height - r2->height))) / 3.216989200105089695361299;

// NB  3.216989200105089695361299 ``normalises'' as it is sqrt (1*1 + 1*1) + srqt (1.5*1.5 + 1*1)
//		 fabs(r1->height - r2->height)) / 2;	    
//  		(hypot(r1->origin.x - r2->origin.x, 5*(r1->origin.y - r2->origin.y)) +
//  		 hypot(r1->width - r2->width, 2*(r1->height - r2->height))) / 3;
}

// calculate a distance measure between two regions, considering positions only
realno RegionTracker::region_distance(const Region *r1,
				      const Region *r2,
				      int *res_hdist,
				      int *res_vdist)
{
    // calculate horizontal and vertical distances
    int hdistance, vdistance;
    
    if (((r2->xlo >= r1->xlo) && (r2->xlo <= r1->xhi)) ||
	((r1->xlo >= r2->xlo) && (r1->xlo <= r2->xhi)) ||
	((r2->xhi >= r1->xlo) && (r2->xhi <= r1->xhi)) ||
	((r1->xhi >= r2->xlo) && (r1->xhi <= r2->xhi)))
	hdistance = 0;
    else
	hdistance = min(abs((r1->xhi)-(r2->xlo)), abs((r1->xlo)-(r2->xhi)));
    
    if (((r2->ylo >= r1->ylo) && (r2->ylo <= r1->yhi)) ||
	((r1->ylo >= r2->ylo) && (r1->ylo <= r2->yhi)) ||
	((r2->yhi >= r1->ylo) && (r2->yhi <= r1->yhi)) ||
	((r1->yhi >= r2->ylo) && (r1->yhi <= r2->yhi)))
	vdistance = 0;
    else
	vdistance = min(abs((r1->yhi)-(r2->ylo)), abs((r1->ylo)-(r2->yhi)));
    
    if (res_hdist != NULL)  // somebody wants this value?
	*res_hdist = hdistance;
    
    if (res_vdist != NULL)  // somebody wants this value?
	*res_vdist = vdistance;
    
    return hypot((realno)hdistance, (realno)vdistance);
}

// This should be called after all other trackers to correct tracks
//   (try matching against results from ActiveShapeTracker) and clean away
//   tracks or hypotheses which could not be matched
void RegionTracker::post_process_frame(Inputs *inputs,
				       Results *results)
{
    TrackedObjectSet *objects = results->get_tracked_objects();
    int index1;   // index into the TrackedObjectSet *objects
    frame_id_t current_frame_id = results->get_motion_image()->get_frame_id();
    
    //
    // 1 - look in all objects for non-matched regions where a profile was tracked
    //
    for (index1 = 0; index1 < objects->no_items; index1++)
    {
	TrackedObject *obj1 = (*objects)[index1];
	
	// make sure obj1 has regions
	if (obj1->regions->no_items == 0)
	    continue;
	
	assert (obj1->regions->no_items == 1); // FIXME: other cases to be implemented
	
	// make sure obj1 has exactly one profile
	if (obj1->profiles->no_items != 1)          // FIXME (TODO): handle case >1
	    continue;
	
	Region *region = obj1->regions->first->dat;
	Profile *profile = obj1->profiles->first->dat;
	
	// do not match region if it has been matched to or is a measurement
	if (region->source == MEASUREMENT)
	    continue;
	
	// match to a profile only if it is a measurement
	if (profile->source != MEASUREMENT)
	    continue;
	
	cdebug << "RT: RegionTracker::correct_and_cleanup_tracks(): correcting unmatched region... "
	       << endl;
	
	// check whether the remainder of the region could make another person  // FIXME: check height as well as width!!!
	if ((region->width > 1.7 * profile->width) &&
	    (max(profile->xlo - region->xlo,
		 region->xhi - profile->xhi) > 0.85 * profile->width))
	{
	    // assume these are two people  // FIXME: or three???
	    Region *remainder;
	    
	    // check whether the remainder is left or right of our profile
	    if (abs(profile->xlo - region->xlo) < abs(region->xhi - profile->xhi))
	    {
		// create remainder in right part of region
		remainder = new Region(max(max(profile->xhi, (int) region->origin.x),
					   (int) (region->xhi - 1.5 * profile->width)),
				       region->xhi,
				       profile->ylo, profile->yhi);  // FIXME: assuming same height
	    }
	    else
	    {
		// create remainder in left part of region
		remainder = new Region(region->xlo,
				       min(min(profile->xlo, (int) region->origin.x),
					   (int) (region->xlo + 1.5 * profile->width)),
				       profile->ylo, profile->yhi);  // FIXME: assuming same height
	    }
	    TrackedObject *r_object = objects->add_observation(remainder,
							       region->frame_last_detected,
							       region->time_last_detected);
	    
	    // correct status set by TrackedObjectSet::add_observation
	    remainder->source = region->source;  // ie MEASUREMENT
	    remainder->frame_first_detected = region->frame_first_detected;
	    remainder->time_first_detected = region->time_first_detected;
	    //leave at true for now     remainder->is_visible = false;
	}
	
	// now correct position, size and status of the region matched to the profile
	region->Observation::operator=(*profile);
	region->old_origin = region->origin;
	region->direction.x = region->direction.y = 0;
    }


    //	
    // 2 - clean away unmatched regions which were created by splitting
    //
    for (index1 = 0; index1 < objects->no_items; index1++)
    {
	TrackedObject *obj1 = (*objects)[index1];
	
	// make sure obj1 has regions
	if (obj1->regions->no_items == 0)
	    continue;
	
	assert (obj1->regions->no_items == 1);  // cannot handle more than one for now	
	
	if (obj1->regions->first->dat->source == SPLITTING)
	{
// FIXME?	    assert (obj1->profiles->no_items == 0);  // otherwise we're in trouble
	    
	    // synthetic object not matched: remove region so object will be cleaned up later
	    obj1->regions->destroy_all();
	}
    }

    //	
    // 3 - clean away new objects which are contained completely within a
    //     profile (noise in the motion image!)
    //
    for (index1 = 0; index1 < objects->no_items; index1++)
    {
	TrackedObject *obj1 = (*objects)[index1];

	// make sure obj1 has regions
	if (obj1->regions->no_items == 0)
	    continue;
	
	ListNode<Region> *reg1;
	bool reg1_was_destroyed;
	
	// for each region: check if it is new and within some profile
	for (reg1 = obj1->regions->first; reg1 != NULL; )  // increment done below
	{
	    // only check new regions
	    if (reg1->dat->frame_first_detected != current_frame_id)
	    {
		reg1 = reg1->next;
		continue;
	    }
	    
	    // do not check regions which are integrated into the background
	    assert(reg1->dat->incorporated_into_background == false);  // otherwise something is wrong

	    reg1_was_destroyed = false;

	    ListNode<TrackedObject> *obj2;
	    
	    // now check all profiles in all objects
	    for (obj2 = objects->first; (obj2 != NULL) && (reg1_was_destroyed == false);
		 obj2 = obj2->next)
	    {
		ListNode<Profile> *prf2;
		
		for (prf2 = obj2->dat->profiles->first; prf2 != NULL; prf2 = prf2->next)
		{
		    // if profile prf2->dat contains our region reg1->dat, destroy region

		    // only check against visible regions
		    if (prf2->dat->source != MEASUREMENT)
			continue;
	   
		    assert(prf2->dat->frame_last_detected == current_frame_id);  // otherwise something is wrong
		    
		    // now check bounding boxes
		    if ((prf2->dat->xlo >= reg1->dat->xlo) || 
			(prf2->dat->xhi <= reg1->dat->xhi) || 
			(prf2->dat->ylo >= reg1->dat->ylo) || 
			(prf2->dat->yhi <= reg1->dat->yhi))
			continue;		    

		    // the region is completely within the profile, so destroy it
		    ListNode<Region> *next_reg1 = reg1->next;
		    obj1->regions->destroy(reg1);
		    reg1 = next_reg1;
		    
		    reg1_was_destroyed = true;
		}
	    }

	    if (reg1_was_destroyed == false)
		reg1 = reg1->next;	
	}
    }
}

} // namespace ReadingPeopleTracker
