///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  PeopleTracker.h                                                          //
//                                                                           //
//  This class handles and schedules trackers and outputs tracking results   //
//                                                                           //
//  It instantiates Camera objects which in turn hold everything associated  //
//   with the respective camera: Inputs, Calibration, the actual Tracking    //
//   class which generates and stores results, a Configuration class etc.    //
//                                                                           //
//  The PeopleTracker lets each Camera class start a thread.  These wait     //
//   for the PeopleTracker to supply the input (images, XML motion data).    //
//   After tracking, they signal us that they have new results.  The         //
//   PeopleTracker then extracts the results from the Camera and fuses data  //
//   from all cameras in order to write the person/object tracks out in XML  //
//   format.                                                                 //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Thu Jan 10 17:20:32 GMT 2002                                 //
//  Revision  : 0.0 of Thu Jan 10 17:20:32 GMT 2002                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
//  Changes:                                                                 //
//   nts: rev 1.0: initial working revision    The Future                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <iostream>
#include <cassert>
#include <limits.h>
#include <ctime>               // for time()

#ifdef _BSD_SOURCE
#include <unistd.h>   // for gethostname()
#endif

// NB: under WIN32, please use the free ``pthreads for win32'' library,
//     see http://sources.redhat.com/pthreads-win32/
#include <pthread.h>            // for IEEE POSIX 1003.1c --- 1995 threads

#include "timestamp_c.h"        // defines TS_TIMESTAMP
#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

class ConfigurationManager;
class Camera;

// Maximum number of cameras per PeopleTracker.  Use 4 within the ADVISOR project.
#define MAX_NUM_CAMERAS 4

class PeopleTracker
{
protected:
    ConfigurationManager *configuration;
    
    Camera *cameras [MAX_NUM_CAMERAS];            // pointers to Camera classes
    unsigned int number_of_cameras;               // number of Camera classes instantiated
    
    char *configuration_filename_base;
    
    // POSIX threading stuff
    pthread_t thread_id;  // our own unique thread id: this will usually be an unsigned long int
    pthread_t camera_thread_id [MAX_NUM_CAMERAS];  // thread ids of individual cameras
    
    // this is the threaded method waiting for data and doing all the processing
    void *do_processing(void *unused);      // NB the method will never return
    
    // Windows requires some overhead here ...
    static void *start_processing(void *people_tracker_object_ptr) // NB the method will never return
	{
	    return ((PeopleTracker *) people_tracker_object_ptr) -> do_processing(NULL);
	}
    
    int state;  // < 0 if there is an error, == 0 otherwise.  Use get_state() to query
                // whether there is an error but note get_state() return values below.
    
    time_t start_time;  // time the PeopleTracker was started, in seconds since the Epoch.
    
    frame_id_t current_frame_id;   // frame id of current / processed results
    frame_id_t previous_frame_id;  // frame id of previous frame / processed results
    
    frame_id_t frame_count;        // simple frame count: #frames completely processed (starts at 0)
    
    static pthread_mutex_t write_access_to_max_id_used_so_far;
    static object_id_t max_id_used_so_far;
    
    // configuration parameters
    char *camera_1_name;
    char *camera_2_name;
    char *camera_3_name;
    char *camera_4_name;
    volatile bool camera_1_enabled;
    volatile bool camera_2_enabled;
    volatile bool camera_3_enabled;
    volatile bool camera_4_enabled;
    volatile bool quiet_mode;
    volatile bool write_results_in_XML;                  // whether to write out tracking results in XML
    char *xml_tracking_results_filename;        // the file to write it to
    ostream *xml_tracking_results_ostream_ptr;  // ostream of the above file, or cinfo
    
#ifdef _BSD_SOURCE
    // host name if we have it (SVr4, 4.4BSD, POSIX 1003.1-2001)
    char hostname[HOST_NAME_MAX + 2];
#endif
    
public:
    PeopleTracker(char *toplevel_config_filename);  // check get_state() to detect failure
    
    ~PeopleTracker();
    
    // set up cameras according to configuration/parameter file
    void setup_cameras();
    
    // static member function to create a new, globally unique id
    static object_id_t create_new_id()
	{
	    pthread_mutex_lock(&write_access_to_max_id_used_so_far);
	    //
	    object_id_t new_id = ++max_id_used_so_far;
	    //
	    pthread_mutex_unlock(&write_access_to_max_id_used_so_far);
	    
	    return new_id;
	}
    
    int get_state();  // return number of active cameras if no error, state, < 0, otherwise
    
    // start a thread which does all processing as data arrives.  returns thread id
    pthread_t start_thread();
    
    int add_cameras (unsigned int num_cameras_to_add,
		     char *camera_names[]);
    
    int remove_cameras (unsigned int num_cameras_to_remove,
			char *camera_names[]);
    
    void handle_new_video_image (unsigned int camera_number,
				 unsigned char *image_data, TS_TIMESTAMP *timestamp);
    
    void handle_new_background_image (unsigned int camera_number,
				      unsigned char *image_data, TS_TIMESTAMP *timestamp);
    
    void handle_new_motion_image (unsigned int camera_number,
				  unsigned char *image_data, TS_TIMESTAMP *timestamp);
    
    void handle_new_blob_data (unsigned int camera_number,
			       unsigned char *xml_data, unsigned int data_size);
    
    
protected:
    // make sure no-one inadvertedly uses copy constructor or operator=
    PeopleTracker (PeopleTracker &original)
	{
	    bool this_is_recommended = false;
	    assert (this_is_recommended == true);   // do not use the copy constructor!
	}
    
    PeopleTracker &operator= (PeopleTracker &original)
	{
	    bool this_is_recommended = false;
	    assert (this_is_recommended == true);   // do not use the copy constructor
	    return *this;
	}
    
private:
    // private helpers
    
    //
    //  Method to write out XML data to a given ostream
    //
    void write_results_in_XML_to_stream(ostream &out);
    
    void register_configuration_parameters();
};

} // namespace ReadingPeopleTracker
