//////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  PeopleTracker.cc  ---  The master People Tracker class, used in main()   //
//                                                                           //
//  This class handles inputs, trackers, data and outputs tracking results   //
//                                                                           //
//  It instantiates Camera objects which in turn hold everything associated  //
//   with the respective camera: Inputs, Calibration, the actual Tracking    //
//   class which generates and stores results, a Configuration class etc.    //
//                                                                           //
//  The PeopleTracker class starts a thread for each Camera class and waits  //
//   for these to signal that they have new results.  The results are        //
//   then extracted and fused by the PeopleTracker and written out in XML    //
//   format.                                                                 //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Thu Jan 10 17:20:32 GMT 2002                                 //
//  Revision  : 0.0 of Thu Jan 10 17:20:32 GMT 2002                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <cstring>  // for strlen etc

#ifdef HAVE_SSTREAM
#include <sstream>
#else
// use strstream for compatability with older versions
#ifdef WIN32
#include <strstrea.h> // DOS uses 8.3 filenames
#else
#include <strstream>  // other systems call it by the real name
#endif
#endif

#ifndef WIN32
#include <unistd.h>   // for gethostname(...)
#endif

#include <fstream>

#include "PeopleTracker.h"
#include "TrackedObject.h"
#include "Camera.h"
#include "text_output.h"   // which defines the ostreams cinfo, cerror etc
#include "Inputs.h"
#include "BufferedSlaveImageSource.h"
#include "BufferedSlaveXMLRegionSource.h"
#include "Results.h"
#include "ConfigurationManager.h"

namespace ReadingPeopleTracker
{

static const char *PeopleTracker_Revision = "@(#) PeopleTracker.cc, rev 0.0 of Thu Jan 10 17:20:32 GMT 2002, Author Nils T Siebel, Copyright (c) 2002 The University of Reading";

// definition and initialisation of static member variables
object_id_t PeopleTracker::max_id_used_so_far = 0;
pthread_mutex_t PeopleTracker::write_access_to_max_id_used_so_far = PTHREAD_MUTEX_INITIALIZER;


///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  CONSTRUCTOR                                                              //
//      PeopleTracker::PeopleTracker(char *toplevel_config_filename)         //
//                                                                           //
//  toplevel_config_filename is the base for all configuration file names.   //
//  Other configuration files are generated by appending camera number (eg,  //
//  LUL-conf1 -> LUL-conf1-01 for the first camera) and initials of tracking //
//  modules (eg, LUL-conf1-01-rt for the RegionTracker of the first camera). //
//                                                                           //
//  The constructor sets a state variable to the number of active cameras    //
//  if there is no error during set-up.  Otherwise the state is set to a     //
//  value < 0.  Use get_state() to query this state.                         //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
PeopleTracker::PeopleTracker(char *toplevel_config_filename)
{
    start_time = time(NULL);
    
    unsigned int index;
    
    for (index = 0; index < MAX_NUM_CAMERAS; index++)
	cameras[index] = NULL;
    
    number_of_cameras = 0;
    
    if (toplevel_config_filename == NULL)
    {
	state = -10000;      // indicate fatal toplevel error
	return;
    }
    
    // setup configuration manager
    configuration = new ConfigurationManager;
    
    // register configuration variables with configuration manager
    register_configuration_parameters();
    
    // read parameter file data into their respective homes
    configuration->parse_parameter_file(toplevel_config_filename);
    
    // remember toplevel config filename as base for ALL configuration files
    configuration_filename_base = new char[strlen(toplevel_config_filename) + 2];
    sprintf(configuration_filename_base, "%s", toplevel_config_filename);
    
#ifdef _BSD_SOURCE
    // create hostname (for XML output)
    gethostname(hostname, sizeof(hostname));
    hostname[sizeof(hostname) - 1] = '\0';  // this is just a precaution if we have a long name
#endif
 

#ifndef NO_DISPLAY
    // create user interface
    if (quiet_mode == false)
	configuration->create_interface(0, NULL);
    else
    {
#ifdef USE_GL
#ifdef LINUX
	// tell Ygl not to mind us calling deflinestyle() in Profile
	noport();
	winopen("dummy tracker graphics window");
#endif
#endif
    }
#endif   // ifndef NO_DISPLAY
    
#ifdef DEBUG
    if (write_results_in_XML == false)
	cdebug << " >>>>   PeopleTracker::PeopleTracker: write_results_in_XML == "
	       << write_results_in_XML
	       << ", FIXME: FORCED to true <<<< " << endl;
    write_results_in_XML = true;   /////  FIXME: this is because the value gets lost
#endif
    
    if (write_results_in_XML)
    {   
	// open output file for XML output
	xml_tracking_results_ostream_ptr = NULL;
	
	if (xml_tracking_results_filename != NULL)
	    xml_tracking_results_ostream_ptr = (ostream*)
		new ofstream(xml_tracking_results_filename, ios::out);
	
	if ((xml_tracking_results_ostream_ptr == NULL) ||
	    (xml_tracking_results_ostream_ptr->bad()))
	{
	    if(xml_tracking_results_filename == NULL)
		xml_tracking_results_filename = "";
	    cerror << "PeopleTracker::PeopleTracker: could not open file "
		   << "" << xml_tracking_results_filename << " for XML output. "
		   << " Defaulting to cinfo. "
		   << endl;
	    xml_tracking_results_ostream_ptr = &cinfo;  // default to cinfo
	}
    }
    else
	xml_tracking_results_ostream_ptr = &cinfo;  // disabled, but still default to cinfo

    // set up cameras for PeopleTracker.
    //   this will start a thread for each configured Camera
    setup_cameras();    
    
    cinfo << " PeopleTracker object set up ok. " << endl;
    
    // set this to a number != the first frame id
    current_frame_id = previous_frame_id = FRAME_ID_T_MAX;
    
    frame_count = 0;
    state = 0;              //  indicate "no error"
    
    thread_id = 0;
}

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  DESTRUCTOR                                                               //
//      PeopleTracker::~PeopleTracker                                        //
//                                                                           //
//  Cancel our thread, then delete the cameras (which cancels their threads) //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
PeopleTracker::~PeopleTracker()
{
    // cancel the thread if we have one
    if ((thread_id != 0) &&              // we have a thread
	(pthread_self() != thread_id))   // and it is not us (just checking...)
	pthread_cancel(thread_id);
    
    // delete all camera classes
    delete [] cameras;
    
    // delete allocated memory
    if (configuration_filename_base != NULL)
	delete configuration_filename_base;
}


///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  int PeopleTracker::add_cameras(unsigned int num_cameras,                 //
//                                 char *camera_names[])                     //
//                                                                           //
//  Add a set of num_cameras cameras to the set of cameras known to the      //
//    tracker.  This will create a thread within each Camera, waiting for    //
//    the arrival of data to be processed.                                   //
//                                                                           //
//  The method returns < 0 on error, otherwise number of new cameras added   //
//                                                                           //
//  Within the ADVISOR system, this will only be called once.  However, the  //
//    method handles the possibility to add cameras at any time by calling   //
//    it again.                                                              //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
int PeopleTracker::add_cameras (unsigned int num_cameras_to_add,
				char *camera_names[])
{
    unsigned int count;
    unsigned int number_of_new_cameras = 0;
    
    // add each camera given to the list of cameras,
    // creating a thread for each camera class instantiated
    for (count = 0; count < num_cameras_to_add; count++)
    {
	// add camera to the list
	if (number_of_cameras >= MAX_NUM_CAMERAS)
	{
	    cerror << "PeopleTracker::add_cameras(): cannot add camera #" << count
		   << " named " << camera_names[count] << " "
		   << "to list of cameras because maximum of "
		   << MAX_NUM_CAMERAS << " cameras is already reached."
		   << endl;
	    
	    state -= (100 * (count+1));
	    
	    assert (state < 0);  // state has to indicate an error here
	    
	    return state;
	}
	
	// create filename for camera configuration by appending the
	//  identifier string to the main configuration filename base
	char *camera_config_filename = new char[strlen(configuration_filename_base) +
					       strlen(camera_names[count]) + 3];
	
	sprintf(camera_config_filename, "%s-%s",  // put a minus `-' in between names
		configuration_filename_base,
		camera_names[count]);
	
#ifdef NO_DISPLAY
	quiet_mode = true;  // disable all screen display and user input
#endif
	
	// try to instantiate camera class    
	try
	{
	    cameras[number_of_cameras] = new Camera(camera_config_filename,
						    quiet_mode);
	}
	catch(...)  // catch all exceptions
	{
	    cerror << "PeopleTracker::add_cameras(): "
		   << "Instantiating camera object failed for camera #" << count
		   << " named " << camera_names[count] << ". " << endl;
	    
	    continue;  // proceed with next camera
	}
	
	// try to start Camera thread which would wait for data and process it
	try
	{
	    camera_thread_id [number_of_cameras] = cameras[number_of_cameras]->start_thread();
	}
	catch(...)  // catch all exceptions
	{
	    cerror << "PeopleTracker::add_cameras(): "
		   << "Starting camera thread failed for camera #" << count
		   << " named " << camera_names[count] << ". " << endl;
	    
	    continue;  // proceed with next camera
	}
	
	//  Success!  new Camera created and waiting for data.
	cinfo << "PeopleTracker::add_cameras(): "
	      << "Camera " << camera_names[count] << " successfully added to system." << endl;
	
	number_of_cameras++;
	number_of_new_cameras++;
	delete [] camera_config_filename;
    }
    
    assert (number_of_cameras >= 0);
    assert (number_of_cameras <= MAX_NUM_CAMERAS);
    assert (number_of_new_cameras >= 0);
    
    return number_of_new_cameras;
}


///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  int PeopleTracker::get_state()  check whether there is an error          //
//                                                                           //
//  returns  the number of active cameras (>= 0) if no error can be detected //
//           < 0 (value of state variable, qv) if there is a known error     //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
int PeopleTracker::get_state()
{
    if (state == 0)   // everything OK?
    {
	// count number of active cameras (ie. existing and enabled `Camera's)
	unsigned int number_of_active_cameras = 0;
	unsigned int index;
	
	for (index = 0; index < number_of_cameras; index++)
	    if ((cameras[index] != NULL) && (cameras[index]->camera_enabled))
		number_of_active_cameras++;
	
	return number_of_active_cameras;
    }
    else
    {
	assert (state < 0);  // (state != 0) means error and this should be < 0
	return state;        // this will hold an error value < 0
    }
}

void PeopleTracker::handle_new_video_image (unsigned int camera_number,
					    unsigned char *image_data,
					    TS_TIMESTAMP *timestamp)
{
    assert(camera_number < MAX_NUM_CAMERAS);  // avoid array index out of bounds
    
    if (cameras[camera_number] == NULL)
    {
	// give a warning message and simply ignore the image
	cerror << " PeopleTracker: Warning: Received new video image for NULL Camera #"
	       << camera_number << "." << endl;
	
	cerror << " Now who de-allocates the image ? " << endl;  // FIXME: TBI
	return;
    }
    
    // simply forward image to camera->inputs
    assert(cameras[camera_number]->inputs->get_video_image_source()->get_source_type() == IST_SLAVE);  // check before type cast
    
    ((BufferedSlaveImageSource *) cameras[camera_number]->inputs->get_video_image_source())
	->handle_new_image(image_data, timestamp);
}

void PeopleTracker::handle_new_background_image (unsigned int camera_number,
						 unsigned char *image_data,
						 TS_TIMESTAMP *timestamp)
{
    assert(camera_number < MAX_NUM_CAMERAS);  // avoid array index out of bounds
    
    if (cameras[camera_number] == NULL)
    {
	// give a warning message and simply ignore the image
	cerror << " PeopleTracker: Warning: Received new background image for NULL Camera #"
	       << camera_number << "." << endl;
	
	cerror << " Now who de-allocates the image ? " << endl;  // FIXME: TBI

	return;
    }
    
    // simply forward image to camera->inputs
    assert(cameras[camera_number]->inputs->external_background_image_source->get_source_type() == IST_SLAVE);  // check before type cast
    
    ((BufferedSlaveImageSource *) cameras[camera_number]->inputs->external_background_image_source)
	->handle_new_image(image_data, timestamp);
}

void PeopleTracker::handle_new_motion_image (unsigned int camera_number,
					     unsigned char *image_data,
					     TS_TIMESTAMP *timestamp)
{
    assert(camera_number < MAX_NUM_CAMERAS);  // avoid array index out of bounds
    
    if (cameras[camera_number] == NULL)
    {
	// give a warning message and simply ignore the image
	cerror << " PeopleTracker: Warning: Received new motion image for NULL Camera #"
	       << camera_number << "." << endl;
	
	cerror << " Now who de-allocates the image ? " << endl;  // FIXME: TBI

	return;
    }
    
    // simply forward image to camera->inputs
    assert(cameras[camera_number]->inputs->external_motion_image_source->get_source_type() == IST_SLAVE);  // check before type cast
    
    ((BufferedSlaveImageSource *) cameras[camera_number]->inputs->external_motion_image_source)
	->handle_new_image(image_data, timestamp);
}

void PeopleTracker::handle_new_blob_data (unsigned int camera_number,
					  unsigned char *xml_data,
					  unsigned int data_size)
{
    assert(camera_number < MAX_NUM_CAMERAS);  // avoid array index out of bounds
    
    if (cameras[camera_number] == NULL)
    {
	// give a warning message and simply ignore the image
	cerror << " PeopleTracker: Warning: Received new blob data for NULL Camera #"
	       << camera_number << "." << endl;
	
	cerror << " Now who de-allocates the XML data ? " << endl;  // FIXME: TBI

	return;
    }
    
    // simply forward XML data block to camera->inputs
    assert(cameras[camera_number]->inputs->external_xml_region_source->get_source_type() == XRST_SLAVE);  // check before type cast
    
    // simply forward XML data to camera->inputs
    ((BufferedSlaveXMLRegionSource *) cameras[camera_number]->inputs->external_xml_region_source)
	->handle_new_blob_data(xml_data, data_size);
}

// start a thread which does all processing as data arrives.  returns thread id
pthread_t PeopleTracker::start_thread()
{
    if (thread_id != 0)
    {
	// we have been called before?!  check whether the thread had been cancelled...
	
	// in order to check that, we will try to cancel the thread ourselves:
	//   FIXME: can you think of another way of doing this w/out using another mutex?
	int return_value = pthread_cancel(thread_id);
	
	if (return_value == 0)
	{
	    // no error occurred => thread exists and it was running!
	    // issue an error message and start it again.
	    cerror << " PeopleTracker::start_thread() requested "
		   << "when thread was already running! "
		   << endl;
	}
    }
    
    // Windows requires some overhead here, see static member function start_processing
    int result = pthread_create(&thread_id, NULL, &start_processing, this);
    
    if (result != 0)
    {
	// an error occurred
	cerror << " PeopleTracker::start_thread(): could not start thread. "
	       << " error code " << result << ". "<< endl;
    }
    
    return thread_id;
}

void *PeopleTracker::do_processing(void *unused)
{
    // for each frame, wait until all cameras have new tracking data available
    //   then merge it and write it out in XML
    
    // first set cancellation parameters: we would like to enable cancelling of
    //   this thread but only after results for the current frame are generated
    int old_cancel_state;
    int old_cancel_type;
    
    // cancellation enabled but deferred until we call pthread_testcancel() ...
    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &old_cancel_state);
    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &old_cancel_type);
    
    unsigned int camera_number;
    bool have_valid_camera;

    while (true)   // loop forever: for each frame...
    {
	have_valid_camera = false;

	// see whether results for the target frame are already available
	for (camera_number = 0; camera_number < number_of_cameras; camera_number++)
	{
	    // check whether the camera is valid
	    if ((cameras[camera_number] == NULL) ||
		(cameras[camera_number]->camera_enabled == false))
		continue;
	    
	    have_valid_camera = true;
	    
	    // these are the results we are waiting for now: Camera::do_processing will copy 
	    // them from its results to previous_results once it has finished generating them
	    Results *results = cameras[camera_number]->previous_results;
	    
	    // We wait until each Camera has completed its Results.
	    //   Once complete, Camera will copy the results to previous_results
	    //   and signal their completeness to us.  While we process previous_results
	    //   Camera will start working on its results for the next frame.
	    if ((results->status != RESULTS_COMPLETE) ||
		(results->frame_id_of_results == previous_frame_id))
	    {
		// no new results yet: wait for signal that new results are available
		pthread_mutex_lock(&results->status_modification);  // avoid deadlocks
		//
		while ((results->status != RESULTS_COMPLETE) ||
		       (results->frame_id_of_results == previous_frame_id))
		{
		    pthread_cond_wait(&results->status_has_changed, &results->status_modification);
		}
		//
		pthread_mutex_unlock(&results->status_modification);
	    }
	    
	    // set frame id to that of the last valid camera
	    current_frame_id = results->frame_id_of_results;
	}

	// check whether no camera is valid (enabled)
	if (have_valid_camera == false)
	{
	    // no cameras available, exit PeopleTracker class
	    cinfo << " PeopleTracker::do_processing: all cameras disabled, exiting..." 
		  << endl;

	    thread_id = 0;    // signal no thread present	    
	    
	    pthread_exit(0);  // EXIT THREAD
	    
	    /* NOTREACHED */
	}

	// fuse data from all cameras
	// ..........
	
	// write out Results in XML if requested

	if (write_results_in_XML == false)
	    cdebug << " >>>>   PeopleTracker::do_processing: write_results_in_XML == "
		   << write_results_in_XML
		   << ", FIXME: set to true <<<< " << endl;
	write_results_in_XML = true;   /////  FIXME: this is because the value gets lost

	if (write_results_in_XML)
	{
	    write_results_in_XML_to_stream(*xml_tracking_results_ostream_ptr);
	}
	
	// other things are possible...
	// ..........
	
	// flag and signal (to Camera) that the Results have been processed by us
	for (camera_number = 0; camera_number < number_of_cameras; camera_number++)
	{
	    // check whether the camera is valid
	    if ((cameras[camera_number] == NULL) ||
		(cameras[camera_number]->camera_enabled == false))
		continue;
	    
	    // these are the results we have processed
	    Results *results = cameras[camera_number]->previous_results;
	    results->status = RESULTS_PROCESSED;
	    pthread_cond_signal(&results->status_has_changed);
	}   
	
	// create a cancelation point.
	pthread_testcancel();	
	
	previous_frame_id = current_frame_id;
	frame_count++;
	
//  	// check whether POSIX 1003.1b-1993 (aka POSIX.4) scheduling is avaliable...
//  #ifdef _POSIX_PRIORITY_SCHEDULING
//  	// yield CPU to Camera classes and other PeopleTracker classes...
//  	sched_yield();
//  #endif
    }
    
    /* NOTREACHED */
    
    return NULL;
}

int PeopleTracker::remove_cameras (unsigned int num_cameras_to_remove,
				   char *camera_names[])
{
    bool this_is_implemented = false;
    assert (this_is_implemented == true);  // not implemented yet (not needed for ADVISOR)
    
    return 0;
}


///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  void PeopleTracker::write_results_in_XML_to_stream(ostream &out)         //
//                                                                           //
//  write out tracking results in XML to a given ostrem                      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
void PeopleTracker::write_results_in_XML_to_stream(ostream &out)
{
    // 1 - write header
    // (opens frame)
    
    // FIXME: should get host name here.  cannot do that in a portable manner --- nts
    if (frame_count == 0)
	out << "<?xml version='1.0' encoding='ISO-8859-1' ?>" << endl;
    else
	out << endl;

    out << "<frame xmlns=\"http://www.cvg.cs.reading.ac.uk/ADVISOR/people\"" << endl
	<< "       xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"" << endl
	<< "             xsi:schemaLocation=\"http://www.cvg.cs.reading.ac.uk/ADVISOR" << endl
	<< "                                  http://www.cvg.cs.reading.ac.uk/~nts/ADVISOR/people_tracker-multi.xsd\""
	<< endl
	<< "       id =\"" << current_frame_id
#ifdef _BSD_SOURCE
	<< "\" pc_name = \"" << hostname << "\" "   // we have one under BSD compatible systems
#else
	<< "\" pc_name = \"unknown\" "
#endif
	<< "num_cameras =\"" << number_of_cameras
	<< "\">" << endl;
    
    unsigned int camera_number;
    
    for (camera_number = 0; camera_number < number_of_cameras; camera_number++)
    {
	// pointers to current camera and results to be written out
	Camera *camera = cameras[camera_number];
	// these are the results to be written out
	Results *results = camera->previous_results;
	
	// 2 - write camera header
	// (opens camera)
	out << "  <camera id =\"" << camera->camera_id
	    << "\" time =\"" << camera->inputs->get_frame_time_in_ms()
	    << "\">" << endl;
	
	// 3 - use Results::operator<< to write out tracking results
	// (opens and closes 0 to unbounded `mobile's)
	out << *results;
	
	// close camera
	out << "  </camera>" << endl;
    }
    
    // close frame
    out << "</frame>" << endl;
}

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  void PeopleTracker::setup_cameras()                                      //
//                                                                           //
//  set up cameras according to configuration/parameter file                 //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
void PeopleTracker::setup_cameras()
{
#ifndef NO_DISPLAY
#ifdef USE_GL
    // tell Ygl not to mind us calling deflinestyle()
    noport();
    winopen("dummy tracker graphics window");
#endif
#endif

    // go through list of camera parameters, create camera_name table with names
    unsigned int number_of_cameras = 0;
    char *camera_names [MAX_NUM_CAMERAS];         // pointers to Camera names
    
    if ((camera_1_enabled) && (camera_1_name != NULL))
	camera_names[number_of_cameras++] = (char *) camera_1_name;
    
    if ((camera_2_enabled) && (camera_2_name != NULL))
	camera_names[number_of_cameras++] = (char *) camera_2_name;
    
    if ((camera_3_enabled) && (camera_3_name != NULL))
	camera_names[number_of_cameras++] = (char *) camera_3_name;
    
    if ((camera_4_enabled) && (camera_4_name != NULL))
	camera_names[number_of_cameras++] = (char *) camera_4_name;
    
    // check whether there is at least one camera given
    if (number_of_cameras == 0)
    {
	cerror << "RPT: No cameras were specified for use! \n";
	exit(1);
    }
    
    // add cameras to PeopleTracker
    add_cameras(number_of_cameras, camera_names);
}    

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  void PeopleTracker::register_configuration_parameters()                  //
//                                                                           //
//  register our configuration data with configuration manager               //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
void PeopleTracker::register_configuration_parameters() 
{
    quiet_mode =
	configuration->register_bool("QUIET_MODE", false,
				     &quiet_mode, true,
				     "PeopleTracker",
				     "In quite mode, no images will be displayed and no user interface shown.");
    
    camera_1_enabled =
	configuration->register_bool("CAMERA_1_ENABLED", true,
				     &camera_1_enabled, false,
				     "PeopleTracker",
				     "Whether to enable the use of camera 1 (ie, the first camera).  Configuration file name for this camera will be ${TOPLEVEL_CONFIG_FILENAME} ## ${CAMERA_1_NAME}");
    
    camera_2_enabled =
	configuration->register_bool("CAMERA_2_ENABLED", false,
				     &camera_2_enabled, false,
				     "PeopleTracker",
				     "Whether to enable the use of camera 2 (ie, the second camera).  Configuration file name for this camera will be ${TOPLEVEL_CONFIG_FILENAME} ## ${CAMERA_2_NAME}");
    
    camera_3_enabled =
	configuration->register_bool("CAMERA_3_ENABLED", false,
				     &camera_3_enabled, false,
				     "PeopleTracker",
				     "Whether to enable the use of camera 3 (ie, the third camera).  Configuration file name for this camera will be ${TOPLEVEL_CONFIG_FILENAME} ## ${CAMERA_3_NAME}");
    
    camera_4_enabled =
	configuration->register_bool("CAMERA_4_ENABLED", false,
				     &camera_4_enabled, false,
				     "PeopleTracker",
				     "Whether to enable the use of camera 4 (ie, the fourth camera).  Configuration file name for this camera will be ${TOPLEVEL_CONFIG_FILENAME} ## ${CAMERA_4_NAME}");
    
    camera_1_name = 
	configuration->register_string("CAMERA_1_NAME", NULL,
				       &camera_1_name, true,
				       "PeopleTracker",
				       "Camera name for camera 1.  Used in camera filename.");
    
    camera_2_name = 
	configuration->register_string("CAMERA_2_NAME", NULL,
				       &camera_2_name, true,
				       "PeopleTracker",
				       "Camera name for camera 2.  Used in camera filename.");
    
    
    camera_3_name = 
	configuration->register_string("CAMERA_3_NAME", NULL,
				       &camera_3_name, true,
				       "PeopleTracker",
				       "Camera name for camera 3.  Used in camera filename.");
    
    
    camera_4_name = 
	configuration->register_string("CAMERA_4_NAME", NULL,
				       &camera_4_name, true,
				       "PeopleTracker",
				       "Camera name for camera 4.  Used in camera filename.");

    
    write_results_in_XML = configuration->register_bool("WRITE_RESULTS_IN_XML", true,
							&write_results_in_XML, true,
							"PeopleTracker",
							"Whether to write out tracking results in XML format");   
    xml_tracking_results_filename =
 	configuration->register_string("XML_TRACKING_RESULTS_FILENAME", NULL,
				       &xml_tracking_results_filename, true,
				       "PeopleTracker",
				       "Optional filename for outputting xml tracking data.");
    
}

} // namespace ReadingPeopleTracker
