/***************************************************************
 * C++ Header
 *
 * File : 	MotionDetector.h
 *
 * Module :	MotionDetector
 *
 * Author : 	A M Baumberg
 *
 * Creation Date : Mon Jun 13 11:25:41 1994 
 *
 * Comments : 	A class to detect moving objects for subsequent 
 *		identification/tracking.
 *
 ***************************************************************/

#ifndef __MOTION_DETECTOR_H__
#define __MOTION_DETECTOR_H__

#include <cassert>

#include "BaseTracker.h"
#include "tracker_defines_types_and_helpers.h"  // for frame_id_t

namespace ReadingPeopleTracker
{

// forward declarations
class Inputs;
class Results;
class RegionSet;
class TrackedObjectSet;
class Image;
class Grey8Image;
class ImageSource;
class PipeSource;
class MultiBackgroundSource;

class MotionDetector : public BaseTracker
{
    friend class Tracking;
    
private:
    unsigned int background_run_length;
    
    ImageSource *input_source;

    PipeSource *foreground;
    PipeSource *background;
    MultiBackgroundSource *multi_background;  // NULL if not used
    PipeSource *difference;
    PipeSource *thresholded_difference;
    PipeSource *blurred_input;
    PipeSource *blurred_background;
   
    // ColourFilter sources
    PipeSource *filtered_input;
    PipeSource *filtered_background;
    PipeSource *filtered_difference;

    // HSV32 sources
    PipeSource *HSV_input;
    PipeSource *HSV_background;
    PipeSource *HSV_difference;

    // images which are constant over time
    Image *initial_background_image;
    // additional mask, read from a file, to mask out time displays etc
    //   value CLEAR_MARK (==0x00) means ignore pixel for mo/detec,
    //   value MARK (==0xff) means do detect motion at this pixel.
    Grey8Image *additional_motion_mask;

    // inverted version of the above, for occlusion reasoning (where CLEAR_MARK
    //   means no object at a given pixel)
    Grey8Image *inverted_additional_motion_mask;

    
    // file names
    char *additional_motion_mask_filename;
    char *initial_background_image_filename;

private:
    Grey8Image *motion_mask;

    bool use_precalculated_difference;
    // true if we should always difference whole image (not pixels on the fly)

    unsigned int sample_skip; 	// for spatial image subsampling
    frame_id_t run_length; 	// run length (life time) of background for median filtering
    
    realno diff_threshold; 	// threshold for image subtraction    
    
    realno max_region_ratio;
    realno min_region_ratio;
    
    realno camera_moved_threshold;
    
    unsigned int total_image_size;		  // number of pixels in subsampled image
    
    realno merge_threshold;

    unsigned int resample_shift;
    bool do_background_updating;
    unsigned int background_update_skip;
    bool merge_regions;
    bool median_filter_motion_image;
    bool do_blur;
    bool do_motion_image_dilation;
    unsigned int gap_size;
    bool prefilter_difference;
    bool use_multi_background_source;
    
    //region tracker vars to pass be passed on
    realno MIN_HEIGHT_TO_WIDTH;
    realno MAX_HEIGHT_TO_WIDTH;
    bool NO_BLOB_FILTER;
  
private:    
    // register configuration variables with the configuration manager
    void register_configuration_variables();

    // update background image from get_video_image() in inputs, put into results
    void update_background_image(Inputs *inputs, Results *results);

    // generate motion image, difference image etc from the images in inputs
    //   and put them into results
    void generate_motion_image(Inputs *inputs, Results *results);

    // extract new regions from results->get_motion_image() and put them
    //  into results->get_tracked_objects()
    void extract_new_regions(Inputs *inputs, Results *results);

    // helper: filter out regions with incorrect height / width ratio
    void height_width_ratio_filter_regions(RegionSet *res);

public:
    MotionDetector(Inputs *inputs, char *config_filename);

    // this method does all the processing
    void process_frame(Inputs *inputs, Results *results,
		       unsigned int max_objects = 32);
    
    // post processing: clean up tracks etc in results
    void post_process_frame(Inputs *inputs,
			    Results *results)
	{
	    // nothing
	}

private:

    inline void detect_new_objects(Inputs *inputs, Results *results)
	{
	    // two steps to detect new objects:
	    generate_motion_image(inputs, results);
	    extract_new_regions(inputs, results);
	}
    
    // these do nothing here because we only do detection, no tracking.
    void predict_old_objects(Results *previous_results)
	{
	    // nothing
	}
    void track_old_objects(Inputs *inputs, Results *results)
	{
	    // nothing
	}

    ConfigurationManager *configuration_manager;
    
    // draw tracked_objects into motion mask to mask them out from detection
    void mask_out_old_objects(TrackedObjectSet *objects);

    // do not use a copy contructor.  nothing good could come from it.
    MotionDetector (const MotionDetector &other)
	{
	    bool this_is_recommended = false;
	    assert (this_is_recommended == true);  // do not use the copy contructor
	}

public:
    inline PipeSource *get_motion_image_source() const
	{
	    return foreground;
	}

    inline PipeSource *get_background_image_source() const
	{
	    return background;
	}
    
    inline PipeSource *get_difference_image_source() const
	{
	    return difference;
	}

    
    // the following will return NULL if no multi_background is available
    inline MultiBackgroundSource *get_multi_background() const
	{
	    return multi_background;
	}

    inline Image *get_initial_background_image() const
	{
	    return initial_background_image;
	}

    inline Grey8Image *get_additional_motion_mask() const
	{
	    return additional_motion_mask;
	}

    inline Grey8Image *get_inverted_additional_motion_mask() const
	{
	    return inverted_additional_motion_mask;
	}

};

} // namespace ReadingPeopleTracker

#endif
