/*********************************************************************
 * C++ source
 *
 * File : 	MotionDetector.cc
 *
 * Module :	MotionDetector
 *
 * Author : 	A M Baumberg
 *
 * Creation Date : Mon Jun 13 13:02:58 1994 
 *
 * Changes : nts: added support for external motion image source  Nov 2000
 *           nts: Many, major changes.  New concept for creation of motion
 *                image, colour filtering techniques, pre-calculated difference
 *                image, dilation for motion image...   Dec 2000--Apr 2001
 *
 *********************************************************************/


// include files

#ifndef NO_DISPLAY
#ifdef DEBUG
#include "os_specific_things.h"  // for sleep()
#endif   // ifdef DEBUG
#endif   // ifndef NO_DISPLAY

#include "MotionDetector.h"

#include "Region.h"
#include "RegionSet.h"
#include "Profile.h"
#include "ProfileSet.h"
#include "PnmSource.h"
#ifdef HSV
#include "HSV32Image.h"    // for HSV filtering using lookup table
#endif
#include "PipeSource.h"    // PipeSource class and image filters

#include "Tracking.h"
#include "Inputs.h"
#include "Results.h"
#include "text_output.h"

namespace ReadingPeopleTracker
{

// see HSV32Image.{h,cc}
// #ifdef HSV
// extern void setup_HSV32_to_RGB32_lookup_table();
// #endif

//  not implemented yet.
//  #ifdef DEBUG
//  #ifndef NO_DISPLAY
//
//  #include "MovieStore.h"   // for outputting demos
//
//  MovieStore *motion_movie = NULL;
//  MovieStore *difference_movie = NULL;
//
//  EnvStringParameter motion_movie_name("MOTION_MOVIE_NAME");
//  EnvStringParameter difference_movie_name("DIFFERENCE_MOVIE_NAME");
//
//  Grey8Image *motion_image = NULL;
//  RGB32Image *difference_image = NULL;
//
//  #endif // ifndef NO_DISPLAY
//  #endif // ifdef DEBUG


void MotionDetector::register_configuration_variables()
{
    // set default values...
    initial_background_image = NULL;
    additional_motion_mask = NULL;
    inverted_additional_motion_mask = NULL;
    
    camera_moved_threshold = 
	configuration_manager->register_real("CAMERA_MOVED_THRESH", 0.5,
					     0.0, 1.0,
					     &camera_moved_threshold,
					     true, "MotionDetector", 
					     "If the number of moving pixels over the total number of\
 pixels is greater than this threshold, assume the camera has moved.");
    
    resample_shift =
	configuration_manager->register_int("MOTION_RESAMPLE_SHIFT", 2,
					    (int *) &resample_shift,
					    true, "MotionDetector",
					    "Resample image dimensions by 2^n in motion detection");
    
    do_background_updating =
	configuration_manager->register_bool("UPDATE_BACKGROUND",
					     true, &do_background_updating,
					     false, "MotionDetector",
					     "Update background, using a temporal median filter ?");
    
    background_update_skip =
	configuration_manager->register_int("BACKGROUND_UPDATE_SKIP", 4,
					    (int *) &background_update_skip, true,
					    "MotionDetector",
					    "Temporal resampling for background filtering.  1 = no skipping");
    
    diff_threshold = 
	configuration_manager->register_real("DETECT_DIFF_THRESH", 0.05,
					     0.0, 1.0, &diff_threshold, 
					     false, "MotionDetector",
					     "Differencing threshold for object detection");
    
    background_run_length=
	configuration_manager->register_int("BACKGROUND_RUN_LENGTH", 50,
					    (int *) &background_run_length, true,
					    "MotionDetector",
					    "Median window length (in frames) for background update");
    
    max_region_ratio =
	configuration_manager->register_real("MAX_REGION_SIZE", 0.5,
					     &max_region_ratio,
					     false, "MotionDetector",
					     "Maximum size of a motion blob over size of image");
    
    min_region_ratio = 
	configuration_manager->register_real("MIN_REGION_SIZE", 0.001,
					     &min_region_ratio,
					     false, "MotionDetector",
					     "Minimum size of a motion blob over size of image");
    
    merge_threshold = 
	configuration_manager->register_real("REGION_MERGE_THRESH", 0.5,
					     &merge_threshold,
					     false, "MotionDetector",
					     "Distance threshold for blob merging relative to blob sizes");
    
    merge_regions =
	configuration_manager->register_bool("DO_MERGE_REGIONS",
					     true, &merge_regions,
					     true, "MotionDetector",
					     "Merge motion regions ?");
    
    median_filter_motion_image =
	configuration_manager->register_bool("MEDIAN_FILTER_MOTION_IMAGE",
					     true, &median_filter_motion_image,
					     true, "MotionDetector",
					     "Do (spatial) median filtering on motion image (slow) ?");
    
    do_blur =
	configuration_manager->register_bool("BLUR_MOTION",
					     false, &do_blur,
					     true, "MotionDetector",
					     "Blur images in motion detection (slow) ?");
    
    do_motion_image_dilation = 
	configuration_manager->register_bool("MOTION_IMAGE_DILATION",
					     false, &do_motion_image_dilation,
					     false, "MotionDetector",
					     "Use dilation to improve motion image quality (slow) ?");
    
    gap_size = 
	configuration_manager->register_int("GAP_SIZE", 2,
					    (int *) &gap_size, false, "MotionDetector",
					    "The dilation size in pixels when filtering binary motion \
image.  The effect is to join blobs that are this many pixels apart.");
    
    use_precalculated_difference = 
	configuration_manager->register_bool("PRECALCULATE_DIFFERENCE_IMAGE",
					     true, &use_precalculated_difference,
					     false, "MotionDetector",
					     "Always calculate whole difference image for foreground \
edge detector");
    
    additional_motion_mask_filename =
	configuration_manager->register_string("ADDITIONAL_MOTION_MASK_FILENAME",
					       NULL, &additional_motion_mask_filename,
					       false, "MotionDetector",
					       "An optional PGM file with an additional constant \
motion mask, for example for masking out digits from a time overlay.");
    
    initial_background_image_filename =
	configuration_manager->register_string("INITIAL_BACKGROUND_IMAGE_FILENAME",
					       NULL, &initial_background_image_filename,
					       false, "MotionDetector",
					       "Filename for the initial background image.");
    
    prefilter_difference =
	configuration_manager->register_bool("PREFILTER_DIFFERENCE",
					     true, &prefilter_difference,
					     false, "MotionDetector",
					     "ColourFilterSource images before differencing ?");
    
    MAX_HEIGHT_TO_WIDTH =
	configuration_manager->register_real("MAX_HEIGHT_TO_WIDTH", 4.0,
					     &MAX_HEIGHT_TO_WIDTH,
					     false, "MotionDetector",
					     "If shapes are filtered, this is the \
maximum aspect ratio allowed");
    
    MIN_HEIGHT_TO_WIDTH =
	configuration_manager->register_real("MIN_HEIGHT_TO_WIDTH", 0.8, 
					     &MIN_HEIGHT_TO_WIDTH,
					     false, "MotionDetector",
					     "If shapes are filtered, this is the \
minimum aspect ratio allowed");
    
    NO_BLOB_FILTER =
	configuration_manager->register_bool("NO_BLOB_FILTER", false, 
					     &NO_BLOB_FILTER,
					     false, "MotionDetector",
					     "If true then all motion blobs are accepted \
as possible moving people. Otherwise use constraints on aspect \
ratio.");
    
    sample_skip =
	configuration_manager->register_int("SAMPLE_SKIP", 1,
					    (int *) &sample_skip, false,
					    "MotionDetector",
					    "Spatial subsampling for motion image (1 means do not subsample)");
    
    use_multi_background_source =
	configuration_manager->register_bool("USE_MULTI_BACKGROUND_SOURCE", false, 
					     &use_multi_background_source,
					     true, "MotionDetector",
					     "Whether to use multi-layer background for temporary \
incorporation of static objects into the background.");
    
   
}//end of register_variables()



MotionDetector::MotionDetector(Inputs *inputs,
			       char *config_filename)    
{
    
/*************************************************************************
 *                                                                       *
 *  New concept options/features:                                        *
 *    conditionally use precalculated difference,                        *
 *    assume background is full size (ie size of input),                 *
 *    allow blurring & background updating through parameter options     *
 *    allow HSV experimental code through #ifdef HSV                     *
 *    Automatically PreProccesses the first frame                        *
 *                                                                       *
 ************************************************************************/    
    
    configuration_manager = new ConfigurationManager;
    
    //register variables with configuration manager
    register_configuration_variables();
    
    configuration_manager->parse_parameter_file(config_filename);
    
    // set up input
    input_source = inputs->get_video_image_source();
    
    // do not update background if external background input is used
    if (inputs->use_external_background_image_source())
    {
	do_background_updating = false;
	initial_background_image = NULL;
	initial_background_image_filename = NULL;
    }
    
    //
    //  1 - set up initial (or, constant) background image
    //
    if (initial_background_image_filename != NULL)
    {
	FILE *initial_background_image_file =
	    fopen(initial_background_image_filename, "rb");  // `b' may be important on non-UNIX

	if (initial_background_image_file != NULL)
	{
	    // FIXME: assuming RGB32 colour images here with no check
	    initial_background_image = new RGB32Image(input_source->get_xdim() / sample_skip,
						      input_source->get_ydim() / sample_skip);
	    
	    // read in file.  The only admissible format is PNM
	    if (PnmSource::read_pnm(initial_background_image_file,
	   	                    initial_background_image) == NULL)
	    {
		delete initial_background_image;
		initial_background_image = NULL;
	    }
	    fclose(initial_background_image_file);
	}
	else
	{
	    initial_background_image = NULL;
	}
    }
    else
	initial_background_image = NULL;

    // check whether there was an error creating the initial background image
    if (initial_background_image == NULL)
    {
	if (initial_background_image_filename != NULL)
	{
	    cerror << " MotionDetector::MotionDetector: Could not read initial background image "
		   << initial_background_image_filename << " " << endl;
	}
	
	if ((do_background_updating == false) &&
	    (inputs->use_external_background_image_source() == false))
	{
	    cerror << " MotionDetector::MotionDetector: No initial background image and no update requested.  Reverting to median filtered background... " << endl;    
	    do_background_updating = true;
	}
    }
    
// //     cdebug << "  >>>> MotionDetector::MotionDetector:  FIXME: do_background_updating was " 
// // 	   << do_background_updating
// // 	   << ", FORCED to false <<<<" << endl;
// //     do_background_updating = false; // FIXME
 
    if (do_background_updating == false)
    {
	if (inputs->use_external_background_image_source() == false)
	{
	    // no background updating and no external motion input: just use constant initial background
	    background = new PipeSource(new ConstantSource (initial_background_image), true);
	}
	else
	{
	    // no background updating but external background input available: just use it.
	    background = new PipeSource(inputs->get_external_background_image_source(), true);
	}
    }
    else
    {
	if (background_update_skip > 1)
	{
	    PipeSource *med_background =
		new MedianFilterSource(input_source, background_run_length,
			               initial_background_image);
	    background = new RepeatSource(background_update_skip, med_background);
	}
	else
	    background = new MedianFilterSource(input_source, background_run_length,
				                initial_background_image);
    }
    
    // check whether to use MultiBackgroundSource
    if (use_multi_background_source)
    {
	multi_background = new MultiBackgroundSource(background, initial_background_image);
	
	background = multi_background;
    }
    else
	multi_background = NULL;
    
    //
    // input set up.  set up difference and foreground (motion) image now...
    //
#ifdef HSV    
    setup_HSV32_to_RGB32_lookup_table();
    HSV_input = new HSV32Source(input_source);
    HSV_background = new ConstantSource(background->get_current()->to_HSV32(NULL));
    HSV_difference = new DifferenceSource(HSV_input, HSV_background);    
    HSV_input->set_title("HSV input");
    HSV_background->set_title("HSV Constant Background");
    HSV_difference->set_title("Difference HSV Fore-Background");
#else
    // 2 alternatives here:
    if (prefilter_difference)     
    {
	if (do_blur)
	{
	    blurred_input = new BlurSource(input_source);
	    blurred_background = new BlurSource(background);
	    filtered_background = new ColourFilterSource(blurred_background);
	    filtered_input = new ColourFilterSource(blurred_input);
	}
	else 
	{
	    filtered_background = new ColourFilterSource(background);
	    filtered_input = new ColourFilterSource(input_source);
	}
	filtered_difference = new DifferenceSource
	    (filtered_input,filtered_background);
	difference = filtered_difference;  // NB we need a valid PipeSource here!
    }
    else
    {
	if (do_blur)
	{
	    blurred_input = new BlurSource(input_source);
	    blurred_background = new BlurSource(background);
	    difference = new DifferenceSource(blurred_input,blurred_background);
	}
	else 
	{
	    difference = new DifferenceSource(input_source, background);
	}
	filtered_difference = new ColourFilterSource(difference);
    }
    
    thresholded_difference =  new ThresholdSource
	(filtered_difference, (unsigned char) (255.0 * diff_threshold));
    
    // set window titles...
    difference->set_title("Difference Fore-Background");
    filtered_difference->set_title("Filtered Difference");
    thresholded_difference->set_title("Thresholded Filtered Difference");
#endif  // ifdef HSV else
    
    if (median_filter_motion_image)
	foreground = new NeighbourSource(thresholded_difference,4);
    else
	foreground = thresholded_difference;
    
    foreground->set_title("Detected Foreground");  // override
//	camera->screen_output->set_show_thresholded_difference_images(false);
    
    // set other Window titles...
// now done in Camera.cc     input_source->set_title("Video Input");
    
    if (do_background_updating)
 	background->set_title("Median-filtered Background");
    else	
	background->set_title("Constant Background");    
    
    
    // set up additional motion mask (for time displays etc) if set...
    if (additional_motion_mask_filename != NULL)
    {
	// enforce this image format, let read_pnm complain if incompatible
	additional_motion_mask = 
	    new Grey8Image(input_source->get_xdim() / sample_skip,
			   input_source->get_ydim() / sample_skip);
	
	PnmSource::read_pnm(additional_motion_mask_filename, additional_motion_mask);
	
	if (additional_motion_mask == NULL)
	{
	    cerror << " MotionDetector::MotionDetector: Warning: Could not read additional motion mask from file "
		   << additional_motion_mask_filename << " " << endl;
	}
#ifndef NO_DISPLAY
	else
	    additional_motion_mask->set_title("Motion mask: Additional mask");
#endif  // ifndef NO_DISPLAY
    }
    else
	additional_motion_mask = NULL;  // no additional mask.
    
    // create inverted additional_motion_mask if possible:
    if (additional_motion_mask != NULL)
    {
	inverted_additional_motion_mask = (Grey8Image *) additional_motion_mask->invert();
    }

    // create motion_mask
    motion_mask = new
	Grey8Image(input_source->get_xdim() / sample_skip, 
		   input_source->get_ydim() / sample_skip);
    
    if (inverted_additional_motion_mask == NULL)
	motion_mask->clear(MARK);
    else
    {
	inverted_additional_motion_mask->copy(motion_mask);
    }
    
#ifndef NO_DISPLAY
    motion_mask->set_title("Motion mask");
#endif  // ifndef NO_DISPLAY

    // re-set threshold as it might have been changed interactively
    ((ThresholdSource*) thresholded_difference)
	-> threshold = (unsigned char) (255.0 * diff_threshold);
    
    if (do_motion_image_dilation)   // do dilation?
    {
	if (gap_size > 0)
	{
	    Grey8Image *filled_foreground = (Grey8Image*)
		foreground->get_current();
	    
	    filled_foreground->fix_holes(gap_size, filled_foreground);
	}
    }
    
    total_image_size = input_source->get_xdim() * input_source->get_ydim();
}

// update background image from get_video_image() in inputs, put into results
void MotionDetector::update_background_image(Inputs *inputs, Results *results)
{
    // let background->get_current() do the recalc() as necessary...
    results->set_background_image(background->get_current());
}

void MotionDetector::generate_motion_image(Inputs *inputs, Results *results)
{   
    // generate/update the new difference and motion images
    
    ((ThresholdSource*) thresholded_difference)
	-> threshold = (unsigned char) (255.0 * diff_threshold);
    
    if ((do_motion_image_dilation) && (gap_size > 0))  // do dilation?
    {
	Grey8Image *filled_foreground = (Grey8Image*)
	    foreground->get_current();
	
	filled_foreground->fix_holes(gap_size, filled_foreground);
    }

//
//  nts:  commented whole stuff out, our cameras do not move
//
//      // check for camera moved
//  	if (small_thresholded_differenced->no_marked >
//           (camera_moved_threshold * total_image_size))
//  	{
//  	    background->restart();
//  	    return;
//  	}
   
    
// tracked object are no longer masked out
//	mask_out_old_objects(results->get_tracked_objects());

    foreground->get_current()
	-> mask(additional_motion_mask, foreground->get_current()); 
    
    // Now we have a difference image, a binary motion image etc.
    // Store them in results:
    results->set_motion_image(foreground->get_current());
    results->set_difference_image(difference->get_current());
    results->set_thresholded_difference_image(thresholded_difference->get_current());
    results->set_filtered_difference_image(filtered_difference->get_current());
}



/////////////   Extract regions with moving blobs from motion image  ////////////////    
void MotionDetector::extract_new_regions(Inputs *inputs, Results *results)
{
    RegionSet *region_set = new RegionSet;  // result set
    
    // be invariant to image scaling / subsampling.  ratios are given as factor of image size
    unsigned int max_region_size = (unsigned int) (total_image_size * max_region_ratio);
    unsigned int min_region_size = (unsigned int) (total_image_size * min_region_ratio);
    
    unsigned int min_size = (merge_regions) ? 2 : min_region_size;  // FIXME: consider, why "2"?    
    
    region_set->grab_regions_and_subimages((Grey8Image*) foreground->get_current(),
					   min_size, max_region_size,
					   true);
    
    if (merge_regions)
    {
	region_set->merge_regions(merge_threshold);
	region_set->filter(min_region_size);
    }
    
    // filter out the regions whose height to width ratio are ``inhumane''
    height_width_ratio_filter_regions(region_set);
    
    // get current frame id and time from inputs
    frame_id_t current_frame_id = inputs->get_frame_id();
    frame_time_t current_frame_time = inputs->get_frame_time_in_ms();
    
    // mark all new regions as based on measurements
    for (region_set->start(); region_set->current_ok(); region_set->forward())
	region_set->get_current()->source = MEASUREMENT;
    
    // add observations to tracked object set in results
    results->get_tracked_objects()->add_observations(region_set,
						     current_frame_id,
						     current_frame_time);

    // de-allocate memory (add_observations takes over the List elements but not the List)
    region_set->delete_all();
    delete region_set;
}

void MotionDetector::process_frame(Inputs *inputs, Results *results,
				   unsigned int max_objects)
{
    
#ifdef ECCV2002
    if (! do_background_updating)
	((MultiBackgroundSource*) background)->force_recalc();  // FIXME: FIX for ECCV  No check
#endif
    
    ///// new Sep 18, 2001
    if (true) // if (additional_motion_mask != NULL)  // FIXME: put config variable
    {
	background->set_motion_mask(foreground->get_current());
    }
    
    // detect new objects...
    detect_new_objects(inputs, results);
}



//////   NB this is unused (summer 2002)
void MotionDetector::mask_out_old_objects(TrackedObjectSet *objects)
{
    if (inverted_additional_motion_mask != NULL)
    {
	// use inverted additional motion mask to initialise motion_mask...
	inverted_additional_motion_mask->copy(motion_mask);
    }
    else
    {
	assert (motion_mask != NULL);  // this was allocated in our constructor
	motion_mask->clear(MARK);
    }
    
// mask out tracked objects
//      for (Listitem<Profile> *curr = old_objects->first; curr != NULL;
//  	 curr = curr->next)
//      {
//  	unsigned int xmin; unsigned int xmax; unsigned int ymin; unsigned int ymax;
//  	curr->dat->get_enclosing_box(xmin,xmax, ymin, ymax);
    
//  	motion_mask->draw_rectangle
//  	    (xmin/sample_skip - 1,
//  	     xmax/sample_skip + 1, 
//  	     ymin/sample_skip - 1,
//  	     ymax/sample_skip + 1,
//  	     CLEAR_MARK);
//      }
    
}


void MotionDetector::height_width_ratio_filter_regions(RegionSet *res)
{
    if (NO_BLOB_FILTER)
	return;
    else
    {
	for (ListNode<Region> *curr = res->first; curr != NULL;)
	{
	    register realno ratio = ((realno) (curr->dat->yhi - curr->dat->ylo)) / 
		((realno) (curr->dat->xhi - curr->dat->xlo));
	    
	    if ((ratio > MAX_HEIGHT_TO_WIDTH) || (ratio < MIN_HEIGHT_TO_WIDTH))
	    {
		ListNode<Region> *bad_region = curr;
		curr = curr->next;
		res->destroy(bad_region);
	    }
	    else
		curr = curr->next;
	}
    }
}

} // namespace ReadingPeopleTracker
