#ifndef __INPUTS_H__
#define __INPUTS_H__

#include "ImageSource.h"

#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

// forward declarations
class Calibration;
class ConfigurationManager;
class XMLRegionSource;
class RegionSet;

class Inputs
{
    friend class Camera;
    friend class PeopleTracker;
    
private:
    //
    //  inputs: image sources and XML structures, camera calibration data
    //
    
    // video image source
    ImageSource *video_image_source;

    // external motion detection input.  NULL if not available.
    ImageSource *external_motion_image_source;      // motion images
    ImageSource *external_background_image_source;  // background images
    XMLRegionSource *external_xml_region_source;    // regions in XML

    // camera calibration 
    Calibration *calibration;

    // configuration options
    frame_id_t last_frame_id;
    bool skip_non_existing;
    frame_id_t max_skip_non_existing;  // max number of non-existing frames to skip
    bool quick_occlusion_reasoning;
    unsigned int occlusion_resample_shift;
   
    // file names
    char *calibration_filename;
    char *video_image_source_filename;
    char *external_motion_image_source_filename;
    char *external_xml_region_source_filename;
    char *external_background_image_source_filename;

public:
   
    Inputs(ConfigurationManager *configuration_manager);

    ~Inputs();
    
    // sets up video input, external motion detection input, camera calibration
    void setup_inputs();
    
    // proceed all inputs to given frame, return actual new frame id.
    frame_id_t proceed_to_frame(frame_id_t next_frame_id);

public:
    
    // static member functions (public helpers)

    // set up buffered slave and file input image source as specified by a filename
    // returns whether some valid input has been set up.
    static bool setup_image_source_input(char *input_filename,
					 ImageSource *&input_source,
					 bool skip_non_existing = false,
					 frame_id_t max_skip_non_existing = 3);
  
private:

    // private helper
    
    // set up buffered slave and file input XML region source as specified by a filename
    // returns whether some valid input has been set up.
    bool setup_xml_region_source_input(char *input_filename,
				       XMLRegionSource *&input_source) const;

public:

    // Accessors
    inline Image *get_video_image() const
	{
	    return video_image_source->get_current();
	}

    inline ImageSource *get_video_image_source() const
  	{
  	    return video_image_source;
  	}
    inline ImageSource *get_external_background_image_source() const
  	{
  	    return external_background_image_source;
	}
    
    inline frame_id_t get_frame_id() const
	{
	    return video_image_source->get_frame_id();
	}
    
    inline frame_time_t get_frame_time_in_ms() const
	{
	    return video_image_source->get_frame_time_in_ms();
	}
    
    inline frame_id_t get_last_frame_id() const
	{
	    return last_frame_id;
	}
    
    inline bool get_skip_non_existing() const
	{
	    return skip_non_existing;
	}
    
    inline unsigned int get_max_skip_non_existing() const
	{
	    return max_skip_non_existing;	    
	}

    inline Calibration *get_calibration() const
	{
	    return calibration;
	}

    inline Image *get_external_motion_image() const
	{
	    if (external_motion_image_source == NULL)
		return NULL;
	    
	    return external_motion_image_source->get_current();
	}
    
    inline Image *get_external_background_image() const
  	{
  	    if (external_background_image_source == NULL)
		return NULL;

	    return external_background_image_source->get_current();
  	}
 
    inline bool use_external_motion_image_source() const
	{
	    return (external_motion_image_source != NULL);
	}

    inline bool use_external_xml_region_source() const
	{
	    return (external_xml_region_source != NULL);
	}
    
    inline bool use_external_background_image_source() const
  	{
  	    return (external_background_image_source != NULL);
  	}
    
    RegionSet *get_external_xml_region_set() const;

    inline unsigned int get_occlusion_resample_shift() const
	{
  	    return occlusion_resample_shift;
  	}
};

} // namespace ReadingPeopleTracker

#endif
