// copied stuff from pte's Inputs.h  nts on Thu Oct 25 14:28:52 2001

#include "Inputs.h"

#include "Image.h"
#include "Calibration.h"
#include "XMLRegionSource.h"
#include "ConfigurationManager.h"
#include "text_output.h"
#include "BufferedSlaveImageSource.h"
#include "PnmSource.h"
#include "BufferedSlaveXMLRegionSource.h"
#include "NumberedFileSource.h"
#include "RegionSet.h"
#include "JPEGSource.h"

namespace ReadingPeopleTracker
{

Inputs::Inputs(ConfigurationManager *configuration_manager)
{
    video_image_source = NULL;
    
    // external motion input: set to NULL and false for now
    external_motion_image_source = NULL;
    external_xml_region_source = NULL;
    
    skip_non_existing =
	configuration_manager->register_bool("SKIP_NON_EXISTING",
					     true, &skip_non_existing,
					     true, "Inputs",
					     "Shall we tolerate (skip) non-existing input files ? \
Useful when you have 0000.jpg, 0001.jpg, 0007.jpg etc");
    
    max_skip_non_existing =
	configuration_manager->register_int("MAX_SKIP_NON_EXISTING", 3,
					    &max_skip_non_existing, true,
					    "Inputs",
					    "Maximum number of non-existing input files to \
tolerate (skip) before assuming end of sequence");
    
    video_image_source_filename =
	configuration_manager->register_string("VIDEO_IMAGE_SOURCE_FILENAME",
					       NULL, &video_image_source_filename,
					       false, "Inputs",
					       "Filename for a cameras video source.");
    
    last_frame_id = (frame_id_t)
	configuration_manager->register_int("LAST_FRAME_ID", 0,
					    &last_frame_id, false,
					    "inputs",
					    "Frame id (number) for the end frame of a video source.");
    
    calibration_filename =
	configuration_manager->register_string("CALIBRATION_FILENAME",
					       NULL, &calibration_filename,
					       false, "Inputs",
					       "Filename for a camera calibration data.");
    
    external_motion_image_source_filename =
	configuration_manager->register_string("MOTION_IMAGE_SOURCE_FILENAME",
					       NULL, &external_motion_image_source_filename,
					       false, "Inputs",
					       "Optional filename for external motion image input.");
    
    external_background_image_source_filename =
	configuration_manager->register_string("BACKGROUND_IMAGE_SOURCE_FILENAME",
					       NULL, &external_background_image_source_filename,
					       false, "Inputs",
					       "Optional (numbered) filename for external background image input.");
    
    external_xml_region_source_filename =
	configuration_manager->register_string("XML_REGION_SOURCE_FILENAME",
					       NULL, &external_xml_region_source_filename,
					       false, "Inputs",
					       "Optional (numbered) filename for external XML region input.");
    
    occlusion_resample_shift = 
	configuration_manager->register_int("OCCLUSION_RESAMPLE_SHIFT", 2,
					    &occlusion_resample_shift,
					    true, "Inputs", 
					    "Resample image dimensions by 2^n for occlusion handling");
    
    quick_occlusion_reasoning =
	configuration_manager->register_bool("QUICK_OCCLUSION_REASONING",
					     true, &quick_occlusion_reasoning,
					     true, "Inputs",
					     "Draw people as rectangles for speed in occlusion reasoning?");
    
}

Inputs::~Inputs()
{
    if (video_image_source != NULL) delete video_image_source;
    if (external_motion_image_source != NULL) delete external_motion_image_source;
    if (external_xml_region_source != NULL) delete external_xml_region_source;
    if (external_background_image_source != NULL) delete external_background_image_source;
    if (calibration != NULL) delete calibration;
}

void Inputs::setup_inputs()
{
    //  1 - Set up video input
    setup_image_source_input(video_image_source_filename,
			     video_image_source,
			     skip_non_existing,
			     max_skip_non_existing);
    
    if (video_image_source == NULL)
    {
	cerror << "Inputs::setup_inputs(): Error opening Video Image Source.  Cannot work without. "
	       << endl;
	exit(1);
    }    
    
    //  2 - Set up input from external motion detector if requested...
    //  2a - motion images
    setup_image_source_input(external_motion_image_source_filename,
			     external_motion_image_source,
			     skip_non_existing,
			     max_skip_non_existing);
    
    //  2b - background images
    setup_image_source_input(external_background_image_source_filename,
			     external_background_image_source,
			     skip_non_existing,
			     max_skip_non_existing);
    
    //  2c - XML Regions
    setup_xml_region_source_input(external_xml_region_source_filename,
				  external_xml_region_source);
    
    // 3 - set up camera calibration
    if (calibration_filename != NULL)
    {
	calibration = new Calibration(calibration_filename,
				      video_image_source->get_ydim());
    }
    else
	calibration = NULL;  // signal non-existence of Calibration
    
}

// set up buffered slave and input source as specified by a filename
//   returns whether some valid input has been set up.
bool Inputs::setup_image_source_input(char *input_filename,
				      ImageSource *&input_source,
				      bool skip_non_existing,            // = false
				      frame_id_t max_skip_non_existing)  // = 3
{
    // set default value NULL, thereby signalling not available [yet]
    input_source = NULL;
    
    // buffered slave (if available, it will be returned as the input_source)
    BufferedSlaveImageSource *slave_source = NULL;
    bool need_slave_source = false;
    
    if (input_filename == NULL)
	return false;
    
    //
    //  1 - check for special filename "slave".  The syntax for this is
    //      either "slave" or "slave," followed by a filename.
    if (strcasecmp(input_filename, "slave") == 0)
    {
	// use buffered slave input which is not fed input?
	cerror << "Inputs::setup_image_source_input: "
	       << "A slave without file input makes no sense. " << endl
	       << "  The requested input was " << input_filename << " " << endl;
	exit(1);
    }
    
    if (strncasecmp(input_filename, "slave,", 6) == 0)
    {
	// Use buffered slave input but fed from a file source.
	
	// in order to do that, we determine and create file source first,
	//   to get the image dimenstions.  Then we open the slave source.
	
	need_slave_source = true;  // remember that we need it
	
	// increase input_filename by strlen("slave,") to get the input_source name
	input_filename += 6;
    }
    
    // 
    //  2 - open input_source which reads images from hard disk
    //      we recognise JPEG, PPM, PGM (PNM formats may be `compress'ed or `gzip'ed)
    
    // determine file type by name (extension)...
    
    char *file_ext = strrchr(input_filename, '.');
    
    if ((strcasecmp(file_ext,".jpg") == 0) ||
	(strcasecmp(file_ext,".jpeg") == 0))
    {
	input_source = new JPEGSource(input_filename, skip_non_existing, max_skip_non_existing);
    }
    else
    {
	if ((strcasecmp(file_ext,".pnm") == 0) ||
	    (strcasecmp(file_ext,".pgm") == 0) ||
	    (strcasecmp(file_ext,".ppm") == 0))
	{
	    input_source = new PnmSource(input_filename, skip_non_existing, max_skip_non_existing);
	}	    
	else
	{
	    // check for compressed PNM
	    if (((strcasecmp(file_ext,".gz") == 0) ||
		 (strcasecmp(file_ext,".Z") == 0)) &&
		(strncasecmp(&file_ext[-4],".pnm",4) == 0) ||
		(strncasecmp(&file_ext[-4],".pgm",4) == 0) ||
		(strncasecmp(&file_ext[-4],".ppm",4) == 0))
	    {
		input_source = new PnmSource(input_filename, skip_non_existing, max_skip_non_existing);
	    }
	    else    
	    {
		cerror << " Inputs::setup_image_source_input: Error: "
		       << " Could not open the requested input file "
		       << input_filename << " ."
		       << endl;
		exit(1);
	    }
	}
    }
    
    //
    //  3 - create slave source if we need it.
    //
    if (need_slave_source)
    {
	// Open slave source to be fed from input_source.  Only now we have the image
	//   dimensions necessary to instantiate the BufferedSlaveImageSource class
	slave_source = new BufferedSlaveImageSource (input_source->get_xdim(),
						     input_source->get_ydim(),
						     input_source);
	
	// use buffered slave in place of the input source
	input_source = slave_source;
    }
    
    return true;  // everything set up OK.
}

// set up buffered slave and file input XML region source as specified by a filename
//   returns whether some valid input has been set up.
//
//  FIXME: currently, the only way of doing that is via BufferedSlaveXMLRegionSource since no
//  FIXME: other XMLRegionSource is implemented.  I think that is OK forever --- nts Apr 2002
bool Inputs::setup_xml_region_source_input(char *input_filename,
					   XMLRegionSource *&input_source) const
{
    // set default value NULL, thereby signalling not available [yet]
    input_source = NULL;
    
    // buffered slave (if available, it will be returned as the input_source)
    NumberedFileSource *file_source = NULL;
    bool need_file_source = false;
    
    if (input_filename == NULL)
	return false;
    
    //
    //  1 - check for special filename "slave".  The syntax for this is
    //      either "slave" or "slave," followed by a filename.
    if (strcasecmp(input_filename, "slave") == 0)
    {
	// use buffered slave input which is not fed input?
	cerror << "Inputs::setup_xml_region_source_input:  "
	       << "A slave without file input makes no sense. " << endl
	       << "  The requested input was " << input_filename << " " << endl;
	exit(1);
    }
    
    if (strncasecmp(input_filename, "slave,", 6) == 0)
    {
	// Use buffered slave input but fed from a file source.	
	
	// in order to do that, we determine and create file source first,
	//   to get the image dimenstions.  Then we open the slave source.	
	
	need_file_source = true;  // remember that we need it
	
	// increase input_filename by strlen("slave,") to get the input file name as usual
	input_filename += 6;
    }    
    //
    //  2 - create a NumberedFileSource which reads XML data from hard disk
    //
    
    if (need_file_source)
    {
	// we need valid image dimensions.  taking them from video_source ...
	if (video_image_source == NULL)
	{
	    cerror << "Inputs::setup_image_source_input: Cannot set up BufferedSlaveXMLRegionSource "
		   << "because no video image source has been set up. " << endl;
	    exit(1);
	}
	
	// set up file source for the slave.  This is a NumberedFileSource.
	
	file_source = new NumberedFileSource(input_filename);
	
	input_source = new BufferedSlaveXMLRegionSource(video_image_source->get_xdim(),
							video_image_source->get_ydim(),
							file_source);
    }
    else
    {
	cdebug << "Inputs::setup_image_source_input: XML Region input "
	       << "without NumberedFileSource not implemented. " 
	       << endl;
	
	bool this_is_implemented = false;
	assert (this_is_implemented == true);  // not implemented yet (not needed now)
	
	return false;  // disable input
    }
    
    return true;   // everything set up OK.
}

// proceed all inputs to frame given as next_frame_id, if possible.
//   returns actual new frame id.
frame_id_t Inputs::proceed_to_frame(frame_id_t next_frame_id)
{
    // we first proceed the video_image_source to the next frame id with
    //   an id >= the given one.  Then we use video_image_source as the
    //   definitive source for the frame id, proceeding other inputs to it.
    //   The rationale here is that if there is no video image for a given
    //   frame, no other source needs to provide data for that frame.
    
    //  1 - get next video image
    
    // first check for frame number wraparound (must be handled for ">" comparisons)
    if ((video_image_source->get_frame_id() > next_frame_id) &&
	(next_frame_id <= MAX_EXPECTED_FRAME_SKIP) &&
	(video_image_source->get_frame_id() >= MAX_FRAME_ID - MAX_EXPECTED_FRAME_SKIP))
    {
	// wrap around input sources, eg. frame_id 999999 -> 000000 within ADVISOR
	
	// video input source
	while (video_image_source->get_frame_id() > next_frame_id)
	    video_image_source->get_next();  // let image_source wrap around
	
	// for external sources, check whether they exist before advancing
	if (external_motion_image_source != NULL)
	    while (external_motion_image_source->get_frame_id() > next_frame_id) 
		external_motion_image_source->get_next();	
	
	if (external_xml_region_source != NULL)
	    while (external_xml_region_source->get_frame_id() > next_frame_id)
		external_xml_region_source->get_next();
	
	// NB  do not force wraparound of external_background_image_source, it might
	//     not have a newer image.  The check for a new image is done below.
    }
    
    // get next frame
    while (next_frame_id > video_image_source->get_frame_id())
	video_image_source->get_next();
    
    // adjust next_frame_id --- if video_image_source skips, so must the other sources
    next_frame_id = video_image_source->get_frame_id();
    
    if (external_motion_image_source != NULL)
	while (next_frame_id > external_motion_image_source->get_frame_id())
	    external_motion_image_source->get_next();
    
    // NB do not expect a new external background image every frame
    if (external_background_image_source != NULL)
    {
	if (external_background_image_source->get_source_type() == IST_SLAVE)
	{
	    // choose latest background image: if input from slave, until buffer is empty
	    while (((BufferedSlaveImageSource *) external_background_image_source)
		   -> get_buffer_entries() > 1)
		external_background_image_source->get_next();
	}
	else
	{
	    // normal ImageSource: use standard get_next()
	    while (next_frame_id > external_background_image_source->get_frame_id())
		external_background_image_source->get_next();
	}
    }
    
    if (external_xml_region_source != NULL)
	while (next_frame_id > external_xml_region_source->get_frame_id())
	    external_xml_region_source->get_next();
    
    return next_frame_id;
}

RegionSet *Inputs::get_external_xml_region_set() const
{
    // if there is no external_xml_region_source just signal this by returning NULL
    if (external_xml_region_source == NULL)
	return NULL;

    // get the regions from the external source.
    RegionSet *regions = external_xml_region_source->get_current();
    
    // These regions might still need some modifications: the XML data
    //   provided should be distributed into Observation variables.
    //   Also, XML data from INRIA is upside down (namely, top to bottom)
    
    // check if we have already modified these regions:
    if ((regions != NULL) && (regions->no_items > 0) &&
	(regions->first->dat->width == 0))      // unmodified?  (assuming actual width > 0)
    {
	// for each Region: copy data from XML structure
	ListNode<Region> *curr;
	
	for (curr = regions->first; curr != NULL; curr = curr->next)
	{
	    Region *region = curr->dat;
	    
	    if (Image::image_addressing_mode == IA_BOTTOM_TO_TOP)
	    {
	        // INRIA XML data uses IA_TOP_TO_BOTTOM so we have to flip the data...
		// flip dimensions according to the video image's height
		unsigned int height = video_image_source->get_ydim();
		
		unsigned int new_ymin = height - (region->xml_blob_data.ymax + 1);
		region->xml_blob_data.ymax = height - (region->xml_blob_data.ymin + 1);
		region->xml_blob_data.ymin = new_ymin;
	    }
	    
	    assert (region->xml_blob_data.ymin < region->xml_blob_data.ymax);   // otherwise min >= max
	    assert (region->xml_blob_data.xmin < region->xml_blob_data.xmax);   // otherwise min >= max
	    
	    // copy over data
	    region->xlo = region->xml_blob_data.xmin;
	    region->xhi = region->xml_blob_data.xmax;
	    region->ylo = region->xml_blob_data.ymin;
	    region->yhi = region->xml_blob_data.ymax;

	    // calculate width
	    region->width = region->xhi - region->xlo;
	    region->height = region->yhi - region->ylo;

	    // use centre, not centre of gravity (cog), for the origin (for now...)
	    region->origin.x = ((realno) (region->xhi + region->xlo)) / 2;
	    region->origin.y = ((realno) (region->yhi + region->ylo)) / 2;

	    // set source
	    region->source = MEASUREMENT;
	}
    }
    
    return regions;
}

} // namespace ReadingPeopleTracker
