///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  HumanFeatureTracker.cc (structure copied from ActiveShapeTracker.cc)    //
//                                                                           //
//  This tracker class tracks human features such as head etc                //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Fri Apr  6 15:08:24 BST 2001                                 //
//  Revision  : 1.0 of Wed Apr 18 18:06:11 BST 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
//  Changes:                                                                 //
//    nts: rev 1.0: initial working revision   Wed Apr 18 18:06:11 BST 2001  //
//    nts: rev 1.1: put in calibration         Thu Nov 15 09:38:47 GMT 2001  //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "HumanFeatureTracker.h"
#include "RGB32Image.h"
#include "limits.h"  // for INT_MAX
#include "Tracking.h"
#include "Calibration.h"
#include "Inputs.h"
#include "text_output.h"
#include "Region.h"
#include "TrackedObjectSet.h"
#include "Results.h"

namespace ReadingPeopleTracker
{

static const char *HumanFeatureTracker_Revision = "@(#) HumanFeatureTracker.cc, rev 0.1 of Wed Apr 18 18:06:11 BST 2001, Author Nils T Siebel, Copyright (c) 2001 The University of Reading";


// BaseTracker::register_base_configuration_variables() registeres some for us!
void HumanFeatureTracker::register_configuration_parameters()
{
    draw_head_search_area =
	configuration.register_bool("DRAW_HEAD_SEARCH_AREA", true,
				    &draw_head_search_area,
				    false, "HumanFeatureTracker",
				    "Draw the head search area?");
    
    draw_in_colour =
	configuration.register_bool("DRAW_COLOUR", true,
				    &draw_in_colour,
				    false, "HumanFeatureTracker",
				    "Use colour for drawing the results?");
    
    draw_head_bbox =
	configuration.register_bool("DRAW_HEAD_BBOX", true,
				    &draw_head_bbox,
				    false, "HumanFeatureTracker",
				    "Draw the head box?");
    
    draw_head_centre =
	configuration.register_bool("DRAW_HEAD_CENTRE", true,
				    &draw_head_centre,
				    false, "HumanFeatureTracker",
				    "Draw the head centre?");
    
    draw_shoulder_width =
	configuration.register_bool("DRAW_SHOULDER_WIDTH", true,
				    &draw_shoulder_width,
				    false, "HumanFeatureTracker",
				    "Draw the shoulder width?");
    
}

HumanFeatureTracker::HumanFeatureTracker(Calibration *the_calibration, char *config_filename)
{
    if (config_filename == NULL)
    {
	cerror << "HFT: HumanFeatureTracker: cannot instantiate without configuration file "
	       << endl;
	exit(1);
    }
    
    // member function registers appropriate configuration parameters with configuration manager
    register_configuration_parameters();
    
    // read from configuration file storing vales in correct place using configuration manager
    configuration.parse_parameter_file(config_filename);
    
    // get calibration (or NULL) from parameter
    calibration = the_calibration;
}


void HumanFeatureTracker::predict_old_objects(Results *previous_results)
{
    // no predictions since there is no tracking but only detection

    // only empty sets for now
    ListNode<TrackedObject> *curr_obj;
    
    // get HumanFeatures sets for each tracked object, destroy
    for (curr_obj = previous_results->get_tracked_objects()->first; curr_obj != NULL;
	 curr_obj = curr_obj->next)	
    {
	curr_obj->dat->features->destroy_all();
    }
    
    return;
}


void HumanFeatureTracker::track_old_objects(Inputs *inputs, Results *results)
{
    // no tracking of old ojects since there is no tracking but only detection

    return;
}

void HumanFeatureTracker::detect_new_objects(Inputs *inputs, Results *results)
{
    ListNode<TrackedObject> *curr_obj;
    ListNode<Region> *curr_reg;	
    
    HumanFeatureSet *new_features = new HumanFeatureSet;  // to store results temporarily
    
    // get regions from tracked object set, adding HumanFeatures for each
    for (curr_obj = results->get_tracked_objects()->first; curr_obj != NULL;
	 curr_obj = curr_obj->next)	
    {
	// process all regions in current object (should be only one)
	for (curr_reg = curr_obj->dat->regions->first; curr_reg != NULL;
	     curr_reg = curr_reg->next)	
	{
	    Region *region = curr_reg->dat;
	    
	    // region should be a measurement
	    if (region->source != MEASUREMENT)
		continue;
	    
	    // ignore regions which are incorporated into the background
	    if (region->incorporated_into_background)
		continue;
	    
	    if (region->region_img == NULL)
	    {
		cdebug << "HFT: Warning: no region image available for this region! "
		       << endl;
		continue;
	    }
	    
	    process_region(region, new_features);
	}
	curr_obj->dat->features->concat(new_features);  // move results to associated object
    }
    
    delete new_features;
}

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// do vertical histogram of region image ("significant" upper line only)   //
//   and scan y values for "peaks"                                         //
//                                                                         //
//   returns the number of peaks found (currently 0 to 10)                 //
//            (FIXME: silently ignores peaks other than the 10 leftmost)   //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
unsigned int HumanFeatureTracker::do_vertical_histogram(Image *region_image,
							VerticalHistogram *result)
{
    // TODO: at some point, this could be done in the difference image looking
    //       at consecutive pixel differences or something.
    
#ifdef DEBUG
    if (debug_level == 1)
	cdebug << "HFT: vert hist:" << endl;
#endif
//    region_image->display();
    
    assert (result != NULL);  // we need the position in the image from result
    
    //
    //  step one: calculate vertical histogram-type statistics.
    //
    //   not an actual histogram because we would like to ignore noise and
    //   "holes" in the lower part of the image.  In a way, we extract the
    //   "significant upper contour".
    
    // variables: geometric
    unsigned int x, y;
    unsigned int w = region_image->get_width();
    unsigned int h = region_image->get_height();
    
    // statistical:
    // (now statical, see header:) unsigned int *stats = new unsigned int[w];
    unsigned int min = INT_MAX;
    unsigned int max = 0;
    unsigned int leftmost = INT_MAX;  // leftmost, rightmost non-empty row
    unsigned int rightmost = 0;
    unsigned int sum = 0;
    
    unsigned int count;
    
    for (x=0; x < w; x++)
    {
	stats[x] = 0;
	
	//  go down row until hitting "significant" number of outline pixels
	const unsigned int significant = 3;

	assert(Image::image_addressing_mode == IA_BOTTOM_TO_TOP);
	
	for (y = h; y > 0; )     // FIXME: assuming BOTTOM_TO_TOP here
	{
	    y--;
	    if (*(region_image->get_pixel(x,y)) == MARK)  // hit something
	    {
		for (count = 1; count < significant; )  // increment done below
		{
		    if (y == 0)           // reached the image bottom
		    {
			stats[x] = count; // make sure we count the hit in
			
			// area width guess: ignore isolated pixels
			if ((x > 0) && (stats[x-1] != 0))
			{
			    if (leftmost > x)
			    {
				// ignore isolated pixels left of here
				leftmost = x - 1;
			    }
			    
			    //   if (rightmost < x)
			    rightmost = x;
			}
			
			break;
		    }
		    else
		    {
			y--;  // check next pixel  // FIXME: BOTTOM_TO_TOP
			
			if (*(region_image->get_pixel(x,y)) == MARK)
			    count++;                 // count this mark
			else
			    break;                   // no significant contour.
		    }
		}
		
		if (count >= significant)
		{
		    stats[x] = y + count;
		    
		    // overall "significant area" width ranges from leftmost to
		    //   rightmost; guessing width here, ignore isolated pixels
		    if ((x > 0) && (stats[x-1] != 0))
		    {
			if (leftmost > x)
			{
			    // ignore isolated pixels left of this point
			    leftmost = x - 1;
			}
			
			//   if (rightmost < x)
			rightmost = x;
		    }
		    
		    break;
		}
	    }
	}
    }
    
    realno cm;       // this should be a measure (in pixels) for around 1 cm (in world) 
    realno five_cm;  // this should be a measure (in pixels) for around 5 cm (in world)
    
    // if we have calibration we can get a good estimate in pixels
    if (calibration != NULL)
    {
	NagVector nag_origin(3);
	
	// head point the person in the image, NAG vector format, homogeneous
	nag_origin[0] = (result->xlo + result->xhi) / 2;
	nag_origin[1] = (result->ylo + result->yhi) / 2;
	nag_origin[2] = 1;
	
	five_cm = calibration->get_image_distance_from_width_in_cm
	    (nag_origin, 5, BaseTracker::TYPICAL_WORLD_HEIGHT);
	
	if (five_cm < 5)
	    five_cm = 5;  // use minimum of 5 pixels
    }
    else
	five_cm = 5;  // default value, ok for LUL camera 38 (centre) at PAL resolution
    
    cm = five_cm / 5;
    
    // check whether main area [leftmost;rightmost] is at least, say, 5 cm:
    if ((rightmost - leftmost < five_cm) || (leftmost > rightmost))
	return 0;                 //  better ignore 'cause it looks like noise
    
    // get sum, min and max ignoring area outside of [leftmost;rightmost]
    min = 10000;
    max = 0;
    sum = 0;
    
    for (x = leftmost; x <= rightmost; x++)
    {
  	register unsigned int value = stats[x];
  	if ((value > 0) && (min > value))  // ignore 0s
  	    min = value;
  	if (max < value)
  	    max = value;
  	sum += value;
    }
    
    if ((min > 5000) || ((max-min) < 6*cm))   // sanity check...
	return 0;                             // found nothing.
    
    result->width = w;
    result->height = h;
    result->leftmost = leftmost;
    result->rightmost = rightmost;
    result->min_value = min;
    result->max_value = max;

    
    //
    //  step two: look for peaks (heads?)...
    //
    unsigned int peak_count = 0;
    unsigned int lower_height = min;
    unsigned int upper_height;
    unsigned int upper_pos;
    
    for (x = leftmost; (x <= rightmost - 1) && (peak_count < 10); x++)
    {
	if (stats[x] > lower_height + 3*cm)  // going up?
	{
	    upper_height = stats[x];
	    upper_pos = x;
	    
	    // move all the way up, avoiding staying in plateaus...
	    for ( ; (x <= rightmost - 1) &&
		      (stats[x+1] + 6*cm > upper_height); x++)
	    {
		if (stats[x] > upper_height)
		{
		    upper_height = stats[x];   // remember local max
		    upper_pos = x;             // and leftmost max pixel
		}
	    }
	    
	    // ... and back again to where the peak starts...
	    for (count = 0; (x-count >= leftmost + 1) &&
		     (stats[x-count-1] + 6*cm > upper_height); count++) ;
	    
	    // we have traced the peak.  save the results and leave
	    //   further checks (width etc) to the head detection.
	    result->peak_width[peak_count] = count + 1;
	    result->peak_height[peak_count] = upper_height - lower_height;
	    result->peak_pos[peak_count] = x - (count+1)/2;
	    
//  	    for ( ; (upper_pos > leftmost) &&
//  		      (stats[upper_pos] + 2 >= upper_height); upper_pos--) ;
	    
	    for (count = 0; stats[upper_pos + count] // + 2
		     >= upper_height; count++) ;
	    
	    result->peak_max_pos[peak_count] = upper_pos + count/2;
	    result->peak_max_value[peak_count] =
		stats[result->peak_max_pos[peak_count]];
	    
	    peak_count++;             // count peaks...
	    lower_height = stats[x];  // to make sure we go down now
	}
	else
	{
	    if (stats[x] < lower_height)
		lower_height = stats[x];
	}
    }

    return peak_count;
}


// assuming region image is non-empty and more than 10 pixels in y direction!
// in fact, it probably won't work well for any region less than 36 pix in y
unsigned int HumanFeatureTracker::detect_heads(Region *region,
					       HumanFeatureSet *feature_set)
{
#ifdef DEBUG
    if (debug_level == 1)
	cdebug << "HFT: detect heads:" << endl;
#endif
    
    VerticalHistogram *v_hist = new VerticalHistogram;
    
    Image *reg_img = region->region_img;
    
    if (reg_img == NULL)  // better check...
    {
#ifdef DEBUG
	if (debug_level == 1)
	    cdebug << "HFT: no region image given. " << endl;
#endif
	return false;
    }
    
    unsigned int reg_height = reg_img->get_height();
    
    // check region for minimum size:
    if (reg_height < 36)
    {
#ifdef DEBUG
	if (debug_level == 1)
	    cdebug << "HFT: image given is too small (" << reg_img->get_width()
		   << " by " << reg_height << " pixels) " << endl;
#endif
	return false;
    }
    
    
    // we only use the upper part of the body for the histogram for head/detec
    unsigned int upper_body_height = reg_height / 3;
    
    if (upper_body_height < 20)  // make sure we use enough pixels for histogram!
    {
	upper_body_height = reg_height / 2;
	
	if (upper_body_height + 10 > reg_height)
	    upper_body_height = reg_height - 10;
    }
    
    Image *upper_reg_img = NULL;
    upper_reg_img = reg_img                // FIXME: assuming BOTTOM_TO_TOP here:
	-> get_subimage(0,reg_img->get_width() - 1,                 // full width
			reg_height - upper_body_height, reg_height - 1,  // upper
			upper_reg_img);  
    
#ifdef DEBUG
    if (debug_level == 1)
	cdebug << "HFT: subimage size " << upper_reg_img->get_width() << "x"
	       << upper_reg_img->get_height() << " pixels " << endl;    
#endif
    
    // initialise v_hist by filling whole area with 0s...
    memset((void *)v_hist, 0, sizeof(VerticalHistogram));
    
    // remember original position of region in video image...
    v_hist->xlo = region->xlo;
    v_hist->ylo = region->yhi + 1 - upper_body_height;
    v_hist->xhi = region->xhi;
    v_hist->yhi = region->yhi;
    
    unsigned int num_of_heads = 0;
    unsigned int number_of_peaks;
    
    //
    //  Step 1: create a vertical histogram of the region subimage
    //
    number_of_peaks = do_vertical_histogram(upper_reg_img, v_hist);
    
    delete upper_reg_img;
    upper_reg_img = NULL;
    
    if (number_of_peaks == 0)  // no detection!
    {
#ifdef DEBUG
	if (debug_level == 1)
	    cdebug << "HFT: no peaks :-(" << endl;
#endif	
	delete v_hist;
	return false;
    }
#ifdef DEBUG
    if (debug_level == 1)
	cdebug << "HFT: " << number_of_peaks << " peaks :-)" << endl;
#endif	
    
    
    //
    //  Step 2: use vertical histogram to detect peaks (heads?) in it
    //
    
    // use Calibration if available
    realno cm;       // this should be a measure (in pixels) for around 1 cm (in world) 
    
    // if we have calibration we can get a good estimate in pixels
    if (calibration != NULL)
    {
	NagVector nag_origin(3);
	
	// head point the person in the image, NAG vector format, homogeneous
	nag_origin[0] = (v_hist->xlo + v_hist->xhi) / 2;
	nag_origin[1] = (v_hist->ylo + v_hist->yhi) / 2;
	nag_origin[2] = 1;

	cm = calibration->get_image_distance_from_width_in_cm
	    (nag_origin, 1, BaseTracker::TYPICAL_WORLD_HEIGHT);
    }
    else
	cm = 1;  // default value, ok for LUL camera 38 (centre) at PAL resolution
    
    unsigned int count;
    HumanFeatures *current;
    
    for (count = 0; count < number_of_peaks; count++)
    {
	// check whether detected peak in the vertical histogram could be a head
	
	if ((v_hist->peak_width[count] > 10*cm) && (v_hist->peak_height[count] > 9*cm))
	{
	    feature_set->add(new HumanFeatures);
	    current = feature_set->get_last();  // the one we just appended
	    current->head = new HeadInfo;
	    
	    current->head->head_histogram = new VerticalHistogram;
	    *current->head->head_histogram = *v_hist;  // copy over data
	    current->head->x = v_hist->peak_pos[count];
	    current->head->x_abs = current->head->x + v_hist->xlo;
	    current->head->y = v_hist->peak_max_value[count] -
		(v_hist->peak_height[count] / 2);
	    current->head->y_abs = current->head->y + v_hist->ylo;
	    
	    current->head->width = v_hist->peak_width[count];
	    current->head->height = v_hist->peak_height[count];
	    
	    current->source = region->source;   // ie MEASUREMENT
	    num_of_heads++;
	}
    }
    delete v_hist;  // clean up memory
    return num_of_heads;
}

void HumanFeatureTracker::process_region(Region *new_region,
					 HumanFeatureSet *result_set)
{
    unsigned int number_of_heads;
    
#ifdef DEBUG
    if (debug_level == 1)
	cdebug << "HFT: init track:" << endl;
#endif
    
    unsigned int w = (unsigned int) new_region->width;
    unsigned int h = (unsigned int) new_region->height;
    
#ifdef DEBUG
    if (debug_level == 1)
	cdebug << "HFT: size " << w << "x" << h << " == " << w * h << " pixels ("
	       << (w * h) / (video_image->get_width() * video_image->get_height() + 1e-7)
	       << " of the image) "<< endl;
#endif
    
    number_of_heads = 0;
    number_of_heads = detect_heads(new_region, result_set);
    
#ifdef DEBUG
    if (debug_level == 1)
    {
	if (number_of_heads > 0) // found a head!
	    cdebug << "HFT: detected " << number_of_heads << " heads. " << endl;
	else
	    cdebug << "HFT: found no head :-( " << endl;  
    }
#endif
    
}


// tracking loop for each new image
void HumanFeatureTracker::process_frame(Inputs *inputs,
					Results *results,
					unsigned int max_objects)
{
#ifdef DEBUG
    if (debug_level == 1)
	cdebug << "HFT: main loop:" << endl;
#endif
    
    video_image = inputs->get_video_image();
//    difference_image = results->get_difference_image();
    motion_image = results->get_motion_image();
    
    predict_old_objects(results);
    track_old_objects(inputs, results);
    detect_new_objects(inputs, results);
}

} // namespace ReadingPeopleTracker
