/***************************************************************
 * C - C++ Header
 *
 * File : 	EdgeDetector.h
 *
 * Module :	EdgeDetector
 *
 * Author : 	A M Baumberg (CoMIR)
 *
 * Creation Date : Thu May 16 14:01:03 1996 
 *
 * Comments : 	general edge/contrast etc detection
 *
 ***************************************************************/


#ifndef __EDGE_DETECTOR_H__
#define __EDGE_DETECTOR_H__

#include "tracker_defines_types_and_helpers.h"
#include "RGB32Image.h"   // for RGB32pixel
#include "Profile.h"

namespace ReadingPeopleTracker
{

// forward declarations
class ActiveShapeTracker;
class PipeSource;
class OcclusionHandler;
class Point2;
class ActiveModel;

// base class 
class EdgeDetector
{
protected:
    realno max_scale;			// maximum scale for subsampling
    
    realno significance_threshold;	// minimum significant response
    
    virtual realno get_maximum_response()
	{
	    return 0;
	}
    
    static realno max_scale_height_ratio;
    static realno max_scale_width_ratio;
    
    // pointer to the calling ActiveShapeTracker so that we can access its configuration parameters
    ActiveShapeTracker *tracker;
    
    static const realno MAX_NORM_RGB_DISTANCE;
    static const realno MAX_RGB_DISTANCE;
    static const realno MAX_GREY_DISTANCE;
    
public: 
    // Names of edge detection methods
    static char *edge_detection_method_names[];  // array of strings with names
    
public:
    
    void set_significance_threshold(realno frac = 0.0)
	{
	    significance_threshold = frac * get_maximum_response();
	}
    
    EdgeDetector()
	{
	    // NOTHING
	} 
    
    // find_edge: find an edge in a window (interval) :-
    // look on a line r = 'pos' + k 'search_line' with |k| < 'window_size'
    // the search direction, although it does not need to be normal, is given by 'normal'
    // the parametric value for the curve point is 'u'
    //
    // return the measurement 'p_obs' 
    // and the discrete measurement variance 'var'
    // return EDGE_GOOD if a significant edge was found
    
    virtual edge_status_t find_edge(realno u, Point2 &pos, Point2 &search_line,
				    Point2 &normal, realno window_size, 
				    OcclusionHandler *occlusion_map, Profile *curr,
				    Point2 &p_obs, realno &var,
				    ActiveModel *active_model) = 0;
    
//     virtual edge_status_t find_edge(realno, Point2& , Point2&, Point2&,
// 				    realno,	OcclusionHandler*,
// 				    Profile*, Point2& , realno& )
// 	{
// 	    return EDGE_GOOD;
// 	}
    
    
    // get object dependencies before performing a batch of edge detections
    // e.g.  need the object size to determine maximum edge size (maximum
    // subsample scale)
    
    virtual void setup_batch(Profile *curr) 
	{ 
	    max_scale = MIN(max_scale_width_ratio * curr->width,
			    max_scale_height_ratio * curr->height);
	    
	    if (max_scale < 1.0)
		max_scale = 1.0;
	}
    
    // the actual edge detection methods
    static bool grey_simple(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model);
    
    static bool grey_sobel(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model);
    
    static bool rgb_simple(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model);
    
    static bool rgb_sobel(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model);
    
    static bool grey_moving_edge(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model);
    
    static bool rgb_moving_edge(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model);
    
    static bool grey_foreground_edge(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model);
    
    static bool rgb_foreground_edge(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model);
    
};

class GenericEdgeDetector : public EdgeDetector
{
    
protected:
    
    realno curr_u_val;
    
    bool (*edge_response) (int, int, int, Point2&, realno &, ActiveModel *);
    
    edge_search_method_t search_mthd;
    
public:
    
    GenericEdgeDetector(edge_search_method_t srch = STATISTICAL_EDGE_SEARCH) 
	{
	    search_mthd = srch;
	}
    
    virtual edge_status_t find_edge(realno u, Point2 &pos, Point2 &search_line,
			    Point2 &normal, realno window_size, 
			    OcclusionHandler *occlusion_map, Profile *curr,
			    Point2 &p_obs, realno &var,
			    ActiveModel *active_model);
    
    // statistical method:
    // look along the profile and obtain an edge response at each point.
    // combine to obtain the mean and variance for the measurement
    // process
    edge_status_t stat_edge(Point2 &pos, Point2 &search_line,
			    Point2 &normal, realno window_size, 
			    OcclusionHandler *occlusion_map, Profile *curr,
			    Point2 &p_obs, realno &var,
			    ActiveModel *active_model);
    
    // maximum method:
    // look along the profile and obtain an edge response at each point.
    // find the maximum response
    edge_status_t max_edge(Point2 &pos, Point2 &search_line,
			   Point2 &normal, realno window_size, 
			   OcclusionHandler *occlusion_map, Profile *curr,
			   Point2 &p_obs, realno &var,
			   ActiveModel *active_model);
    
    // nearest method:
    // find the nearest significant edge
    edge_status_t nearest_edge(Point2 &pos, Point2 &search_line,
			       Point2 &normal, realno window_size, 
			       OcclusionHandler *occlusion_map, Profile *curr,
			       Point2 &p_obs, realno &var,
			       ActiveModel *active_model);
    
};


class SobelEdgeDetector : public GenericEdgeDetector
{
protected:
    
public:
    
    SobelEdgeDetector(realno thresh = 0.0,
		      edge_search_method_t mthd = STATISTICAL_EDGE_SEARCH, 
		      ActiveShapeTracker *the_tracker = NULL);
    
    virtual inline realno get_maximum_response();

    virtual edge_status_t find_edge(realno u, Point2 &pos, Point2 &search_line,
				    Point2 &normal, realno window_size, 
				    OcclusionHandler *occlusion_map, Profile *curr,
				    Point2 &p_obs, realno &var,
				    ActiveModel *active_model)
	{
	    bool this_method_has_been_reimplemented = false;
	    
	    assert (this_method_has_been_reimplemented == true);

	    return EDGE_OCCLUDED;  // be elusive...
	}

};

// Edge (1D)

class SimpleEdgeDetector : public GenericEdgeDetector
{
protected:
    
public:
    
    SimpleEdgeDetector(realno thresh = 0.0, 
		       edge_search_method_t mthd = STATISTICAL_EDGE_SEARCH, 
		       ActiveShapeTracker *the_tracker = NULL);

    inline realno get_maximum_response();
    
};

// Moving Edge
//
// looking at spatial and temporal derivative
// but don't assume background image is available

class MovingEdgeDetector : public GenericEdgeDetector
{
protected:
    
//     Image *short_sequence[2];
//     // short_sequence[0] == previous image frame
//     // short_sequence[1] == current image frame
    
public:
    
    MovingEdgeDetector(realno thresh = 0.0,
		       edge_search_method_t mthd = STATISTICAL_EDGE_SEARCH, 
		       ActiveShapeTracker *the_tracker = NULL);
    
    inline realno get_maximum_response();
    
};

// ForegroundEdge (signed edge)


class ForegroundEdgeDetector : public GenericEdgeDetector
{
    
public:
    
    realno thresh_background;
    realno thresh_foreground;
    
public:
    
    // a foreground edge is detected if
    // d1 < thresh_lower, d2 > thresh_upper
    // where
    // d1 = movement detected to the left of the edge
    // d2 = movement detected to the right of the edge
    // scaled such that 0 <= d_i <= 1
    ForegroundEdgeDetector(realno thresh_lower = 0.5,
			   realno thresh_upper = 0.5,
			   edge_search_method_t mthd = MAXIMUM_EDGE_SEARCH, 
			   ActiveShapeTracker *the_tracker = NULL);
    //  nts: changed default mthd, was STATISTICAL_EDGE_SEARCH 
    
    inline realno get_maximum_response();
};

class ColourBlob
{
private:
    static int cblob_glwin;
    
public:
    
    //NagVector mean_colour;
    //NagVector new_mean;
    
    RGB32pixel theColour;
    bool theColour_valid;
    
    RGB32pixel newColour;
    bool newColour_valid;
    
    //NagMatrix E_xxT;
    
    //NagMatrix inv_covariance;
    
    ColourBlob() 
	{
	    newColour_valid = false;
	    theColour_valid = false;
	}
    
    ~ColourBlob()
	{
	    // nothing
	} 
    
    void add_observation(RGB32pixel *observed_colour)
	{ 
	    if (newColour_valid == false)
		newColour_valid = true;
	    
	    newColour = *observed_colour; 
	}
    
    void update();
    realno colour_distance(RGB32pixel *col);
    // normalised colour distance
    realno norm_colour_distance(RGB32pixel *col);
    
    void visualise(realno u = 0); 
    
};

class ColourEdgeDetector : public EdgeDetector
{
public:
    
    ColourBlob *cblobs;
    ColourBlob *current_blob;
    //realno col_gain;
    
    int no_cblobs;
    
    realno edge_threshold;
    
    ColourEdgeDetector(int no_colour_blobs = 20)
	{
	    no_cblobs = no_colour_blobs;
	}
    
    virtual edge_status_t find_edge(realno u, Point2 &pos, Point2 &search_line,
				    Point2 &normal, realno window_size, 
				    OcclusionHandler *occlusion_map, Profile *curr,
				    Point2 &p_obs, realno &var,
				    ActiveModel *active_model);
    
    virtual RGB32pixel *edge_response(realno x, realno y, 
				      int grid_scale, Point2 n,
				      realno &response) = 0;
    
    void setup_batch(Profile *curr);
};


// Use a combination of colour image subtraction and
// inside colour (from previous frames)
class ColourForegroundEdgeDetector : public ColourEdgeDetector
{
    
private:
    realno thresh_background;
    realno thresh_foreground;
    
public:
    
    ColourForegroundEdgeDetector(realno thresh_lower = 0.5,
				 realno thresh_upper = 0.5,
				 ActiveShapeTracker *the_tracker = NULL,
				 int no_cblobs = 20);
    
    RGB32pixel *edge_response(realno x, realno y, int grid_scale,
			      Point2 n, realno &repsonse); 
    
    inline realno get_maximum_response();
    
};

// Uses a combination of *normalised *colour subtraction
// and inside normalised colour (from previous frames)
class NormalisedColourForegroundEdgeDetector : public ColourEdgeDetector
{
    
public:
    
    realno thresh_background;
    realno thresh_foreground;
    
    NormalisedColourForegroundEdgeDetector(realno thresh_lower = 0.5,
					   realno thresh_upper = 0.5,
					   ActiveShapeTracker *the_tracker = NULL,
					   int no_cblobs = 20);
    
    RGB32pixel *edge_response(realno  x, realno y, int grid_scale,
			      Point2 n, realno &repsonse); 
    
    inline realno get_maximum_response();
    
};

} // namespace ReadingPeopleTracker

#endif
