/***************************************************************
 * C++ source
 *
 * File : 	EdgeDetector.cc
 *
 * Module :	EdgeDetector
 *
 * Author : 	A M Baumberg (CoMIR)
 *
 * Creation Date : Fri May 17 14:17:32 1996
 *
 * Comments : 	
 *
 ***************************************************************/


// include files

#include "OcclusionHandler.h"
#include "EdgeDetector.h"
#include "Image.h"
#include "PipeSource.h"
#include "Grey8Image.h"
#include "RGB32Image.h"
#include "ActiveShapeTracker.h"
#include "text_output.h"
#include "ActiveModel.h"

#ifndef NO_DISPLAY
#ifdef USE_GL
#include <gl.h>
#else
#include <vogl.h>
#endif
#endif   // #ifndef NO_DISPLAY

#include <cassert>

namespace ReadingPeopleTracker
{

// definition and initialisation of static member variables
realno EdgeDetector::max_scale_height_ratio = 0.05;
realno EdgeDetector::max_scale_width_ratio = 1.0;
const realno EdgeDetector::MAX_NORM_RGB_DISTANCE = sqrt(2.0);
const realno EdgeDetector::MAX_RGB_DISTANCE = sqrt(3.0 * 255.0 * 255.0);
const realno EdgeDetector::MAX_GREY_DISTANCE = 255.0;

#ifndef NO_DISPLAY
int ColourBlob::cblob_glwin = NULLWIN;
#endif
// Names of edge detection methods
char *EdgeDetector::edge_detection_method_names[] =
{
    "COLOUR_FOREGROUND_EDGE",
    "COLOUR_EDGE",
    "MOVING_EDGE",
    "SIMPLE_EDGE",
    "SOBEL_EDGE",
    "FOREGROUND_EDGE",
    "NORMALISED_COLOUR_FOREGROUND_EDGE",
    NULL
};

// pixelwise edge / contrast detection functions ...

bool EdgeDetector::grey_simple(int x, int y, int step, Point2 &n,
			       realno &edge, ActiveModel *active_model)
{
    int dx = real_to_int(n.x * step);
    int dy = real_to_int(n.y * step);
    
    int xr = x + dx;
    int yr = y + dy;
    int xl = x - dx;
    int yl = y - dy;
    Image *img = active_model->get_video_image();
    
    if (!((img->check_coords(xr,yr)) && (img->check_coords(xl,yl))))
	return false;
    
    int c1 = *img->get_pixel(xr,yr);
    int c2 = *img->get_pixel(xl,yl);
    
    edge = (realno) abs(c1 - c2);
    return true;
}

bool EdgeDetector::grey_sobel(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model)
{
    int xr = x + step;
    int xl = x - step;
    int yr = y + step;
    int yl = y - step;
    Image *img = active_model->get_video_image();
    
    if (!((img->check_coords(xr,yr)) && (img->check_coords(xl,yl))))
	return false;
    
    int c1 = *img->get_pixel(xl,yl);
    int c2 = *img->get_pixel(x,yl);
    int c3 = *img->get_pixel(xr,yl);
    int c4 = *img->get_pixel(xl,y);
    //  int c5 = *img->get_pixel(x,y);
    int c6 = *img->get_pixel(xr,y);
    int c7 = *img->get_pixel(xl,yr);
    int c8 = *img->get_pixel(x,yr);
    int c9 = *img->get_pixel(xr,yr);
    
    int v_response = c1 - c7 + (2 * (c2 - c8)) + c3 - c9;
    int h_response = c1 - c3 + (2 * (c4 - c6)) + c7 - c9;
    
    //  edge = fabs((h_response * n.x) + (v_response * n.y));
    edge = sqrt((float)(h_response * h_response) + (v_response * v_response));
    
    return true;
}

bool EdgeDetector::rgb_simple(int x, int y, int step,
			      Point2 &n, realno &edge, ActiveModel *active_model)
{
    int dx = real_to_int(n.x * step);
    int dy = real_to_int(n.y * step);
    
    if ((dx == 0) && (dy == 0))
    {
	cdebug << "rgb_simple: dx == dy == 0.  This should not happen. " << endl
	       << "  n = " << n << ", step = " << step << endl;
    }
    
    int xr = x + dx;
    int yr = y + dy;
    int xl = x - dx;
    int yl = y - dy;
    Image *img = active_model->get_video_image();
    
    if (!((img->check_coords(xr,yr)) && (img->check_coords(xl,yl))))
	return false;
    
    RGB32pixel *p1 = (RGB32pixel*) img->get_pixel(xr,yr);
    RGB32pixel *p2 = (RGB32pixel*) img->get_pixel(xl,yl);
    
    int rdiff = p1->red - p2->red;
    int gdiff = p1->green - p2->green;
    int bdiff = p1->blue - p2->blue;
    
    edge = sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
    
    return true;
}

bool EdgeDetector::rgb_sobel(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model)
{
    int xr = x + step;
    int xl = x - step;
    int yr = y + step;
    int yl = y - step;

    Image *img = active_model->get_video_image();

    if (!((img->check_coords(xr,yr)) && (img->check_coords(xl,yl))))
	return false;
    
    RGB32pixel *c1 = (RGB32pixel*) img->get_pixel(xl,yl);
    RGB32pixel *c2 = (RGB32pixel*) img->get_pixel(x,yl);
    RGB32pixel *c3 = (RGB32pixel*) img->get_pixel(xr,yl);
    RGB32pixel *c4 = (RGB32pixel*) img->get_pixel(xl,y);
    //  RGB32pixel *c5 = (RGB32pixel*) img->get_pixel(x,y);
    RGB32pixel *c6 = (RGB32pixel*) img->get_pixel(xr,y);
    RGB32pixel *c7 = (RGB32pixel*) img->get_pixel(xl,yr);
    RGB32pixel *c8 = (RGB32pixel*) img->get_pixel(x,yr);
    RGB32pixel *c9 = (RGB32pixel*) img->get_pixel(xr,yr);
    
    int r_v_response =
	c1->red - c7->red + (2 * (c2->red - c8->red)) + c3->red - c9->red;
    int g_v_response =
	c1->green - c7->green + (2 * (c2->green - c8->green)) +
	c3->green - c9->green;
    int b_v_response = c1->blue - c7->blue + (2 * (c2->blue - c8->blue))
	+ c3->blue - c9->blue;
    
    int r_h_response =
	c1->red - c3->red + (2 * (c4->red - c6->red)) + c7->red - c9->red;
    int g_h_response = c1->green - c3->green +
	(2 * (c4->green - c6->green)) +
	c7->green - c9->green;
    int b_h_response = c1->blue - c3->blue +
	(2 * (c4->blue - c6->blue)) + c7->blue - c9->blue;
    
    realno r_edge = (r_h_response * n.x) + (r_v_response * n.y);
    realno g_edge = (g_h_response * n.x) + (g_v_response * n.y);
    realno b_edge = (b_h_response * n.x) + (b_v_response * n.y);
    
    edge = (sqrt((float)(r_edge * r_edge + g_edge * g_edge + b_edge * b_edge)));
    return true;
    
    
}

bool EdgeDetector::grey_moving_edge(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model)
{
    
    int dx = real_to_int(n.x * step);
    int dy = real_to_int(n.y * step);
    
    int xr = x + dx;
    int yr = y + dy;
    int xl = x - dx;
    int yl = y - dy;
    
    Image *prev = active_model->get_previous_video_image();
    Image *curr = active_model->get_video_image();
    
    
    if (!((curr->check_coords(xr,yr)) && (curr->check_coords(xl,yl))))
	return false;
    
    int c1 = *curr->get_pixel(xr,yr);
    int c2 = *curr->get_pixel(xl,yl);
    int c3 = *curr->get_pixel(x,y);
    int c4 = *prev->get_pixel(x,y);
    
    edge = (realno) (abs(c1 - c2) + abs(c3 - c4));
    return true;
}

bool EdgeDetector::rgb_moving_edge(int x, int y, int step, Point2 &n, realno &edge, ActiveModel *active_model)
{
    
    int dx = real_to_int(n.x * step);
    int dy = real_to_int(n.y * step);
    
    int xr = x + dx;
    int yr = y + dy;
    int xl = x - dx;
    int yl = y - dy;
    
    Image *prev = active_model->get_previous_video_image();
    Image *curr = active_model->get_video_image();
    
    if (!((curr->check_coords(xr,yr)) && (curr->check_coords(xl,yl))))
	return false;
    
    RGB32pixel *p1 = (RGB32pixel*) curr->get_pixel(xr,yr);
    RGB32pixel *p2 = (RGB32pixel*) curr->get_pixel(xl,yl);
    RGB32pixel *p3 = (RGB32pixel*) curr->get_pixel(x,y);
    RGB32pixel *p4 = (RGB32pixel*) prev->get_pixel(x,y);
    
    int rdiff = p1->red - p2->red;
    int gdiff = p1->green - p2->green;
    int bdiff = p1->blue - p2->blue;
    
    realno spatial_edge =
	sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
    
    rdiff = p3->red - p4->red;
    gdiff = p3->green - p4->green;
    bdiff = p3->blue - p4->blue;
    realno temporal_edge =
	sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
    
    edge = spatial_edge + temporal_edge;
    
    return true;
}


bool EdgeDetector::grey_foreground_edge(int x, int y, int step, Point2 &n,
					realno &edge, ActiveModel *active_model)
{
    ForegroundEdgeDetector *parent = (ForegroundEdgeDetector*) active_model->get_edge_detector();

    int dx = real_to_int(n.x * step);
    int dy = real_to_int(n.y * step);
    
    int xr = x + dx;
    int yr = y + dy;
    int xl = x - dx;
    int yl = y - dy;

    Image *back = active_model->get_background_image();
    Image *front = active_model->get_video_image();
    
    if (!((front->check_coords(xr,yr)) && (front->check_coords(xl,yl))))
	return false;
    
    int c1 = *front->get_pixel(xr,yr);
    int c2 = *back->get_pixel(xr,yr);
    int c3 = *front->get_pixel(xl,yl);
    int c4 = *back->get_pixel(xl,yl);
    
    int d1 = abs(c1 - c2);
    int d2 = abs(c3 - c4);
    
    if ((d2 <= parent->thresh_foreground) ||
	(d1 >= parent->thresh_background))
	edge = 0;
    else
	edge = (realno) (d2 - d1);
    
    return true;
}


bool EdgeDetector::rgb_foreground_edge(int x, int y, int step, Point2 &n,
				       realno &edge, ActiveModel *active_model)
{
    ForegroundEdgeDetector *parent = (ForegroundEdgeDetector*) active_model->get_edge_detector();
    int dx = real_to_int(n.x * step);
    int dy = real_to_int(n.y * step);
    
    int xr = x + dx;
    int yr = y + dy;
    int xl = x - dx;
    int yl = y - dy;
    Image *back = active_model->get_background_image();
    Image *front = active_model->get_video_image();
    
    if (!((front->check_coords(xr,yr)) && (front->check_coords(xl,yl))))
	return false;
    
    RGB32pixel *p1 = (RGB32pixel*) front->get_pixel(xr,yr);
    RGB32pixel *p2 = (RGB32pixel*) back->get_pixel(xr,yr);
    RGB32pixel *p3 = (RGB32pixel*) front->get_pixel(xl,yl);
    RGB32pixel *p4 = (RGB32pixel*) back->get_pixel(xl,yl);
    
    int rdiff = p1->red - p2->red;
    int gdiff = p1->green - p2->green;
    int bdiff = p1->blue - p2->blue;
    
    realno d1 = sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
    
    rdiff = p3->red - p4->red;
    gdiff = p3->green - p4->green;
    bdiff = p3->blue - p4->blue;

    realno d2 = sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
    
    if ((d2 <= parent->thresh_foreground) ||
	(d1 >= parent->thresh_background))
	edge = 0;
    else
	edge = d2 - d1;
    
    return true;
}


edge_status_t
GenericEdgeDetector::find_edge(realno u, Point2 &pos,
			       Point2 &search_line,
			       Point2 &normal, realno window_size,
			       OcclusionHandler *occlusion_map,
			       Profile *currx,
			       Point2 &p_obs, realno &var,
			       ActiveModel *active_model)
{
    curr_u_val = u;
    
    // ugly switch cannot easily be avoided !
    switch (search_mthd)
    {
    case NEAREST_EDGE_SEARCH:
	return nearest_edge(pos, search_line, normal, window_size,
			    occlusion_map, currx, p_obs, var, tracker->get_active_model());
    case STATISTICAL_EDGE_SEARCH:
	return stat_edge(pos, search_line, normal, window_size,
			 occlusion_map, currx, p_obs, var, tracker->get_active_model());
    default:
    case MAXIMUM_EDGE_SEARCH:
	return max_edge(pos, search_line, normal, window_size,
			occlusion_map, currx, p_obs, var, tracker->get_active_model());
	
    }
    return EDGE_BAD;
}


// mechanisms for combining pixel edge information

edge_status_t
GenericEdgeDetector::stat_edge(Point2 &pos, Point2 &search_line,
			       Point2 &normal, realno window_size,
			       OcclusionHandler *occlusion_map,
			       Profile *currx,
			       Point2 &p_obs, realno &var,
			       ActiveModel *active_model)
{
    int nsubs; // sampling size
    
    if (window_size > max_scale)  // this will generally be the case
	nsubs = (int) (0.5 + (window_size / max_scale));
    else
	nsubs = 1;
    
    realno edge_scale = window_size / ((realno) nsubs);
    int grid_scale = (int) (edge_scale + 0.5);
    var = 0;
    realno sum_e = 0;
    realno sum_d = 0;
    realno sq_sum = 0;
    realno max_e = -1e30, best = 0;
    Point2 pos_base = pos + Point2(0.5, 0.5);
//    Point2 line_offset = edge_scale * search_line;
    
    for (int i = -nsubs;i <= nsubs; i++)
    {
	realno d_i = i * edge_scale;
	
	realno e_i = significance_threshold-1.0;
	Point2 pos_i = pos_base + (d_i * search_line);
	if (occlusion_map->is_occlusion(pos_i.x, pos_i.y))
	    return EDGE_OCCLUDED;
	
//       if (!(occlusion_map->is_occlusion(pos_i.x, pos_i.y) ||
// 	    occlusion_map->is_occlusion(pos_i.x + line_offset.x,
// 					pos_i.y + line_offset.y) ||
// 	    occlusion_map->is_occlusion(pos_i.x - line_offset.x,
// 					pos_i.y - line_offset.y)))
	if (!(*edge_response)((int) pos_i.x, (int) pos_i.y, grid_scale, normal,
			      e_i, tracker->get_active_model()))
	    return EDGE_OCCLUDED;
	realno new_e_i = 0;
	
	if (e_i > significance_threshold)
	{
	    new_e_i = e_i;
	    if (e_i > max_e)
	    {
		max_e = e_i;
		best = d_i;
	    }
	}
	
	sum_e += new_e_i;
	sum_d += d_i * new_e_i;
	sq_sum += d_i * d_i * new_e_i;
	
    }
    
    if (sum_e == 0)
	return EDGE_BAD;
    
    realno mean = sum_d / sum_e;
    //realno mean = best;
    var = 0.25 * ((sq_sum / sum_e) - (mean * mean));
    p_obs = pos_base + mean * search_line;
    // var += 0.25 * edge_scale * edge_scale;
    return EDGE_GOOD;
}

edge_status_t
GenericEdgeDetector::max_edge(Point2 &pos, Point2 &search_line,
			      Point2 &normal, realno window_size,
			      OcclusionHandler *occlusion_map,
			      Profile *currx,
			      Point2 &p_obs, realno &var,
			      ActiveModel *active_model)
{
    int nsubs; // sampling size
    
    if (window_size > max_scale)  // this will generally be the case
	nsubs = (int) (0.5 + (window_size / max_scale));
    else
	nsubs = 1;
    
    realno edge_scale = window_size / ((realno) nsubs);
    int grid_scale = (int) (edge_scale + 0.5);
    var = 0;
    realno max_e = -significance_threshold;
    
    Point2 pos_base = pos + Point2(0.5, 0.5);
    for (int i = -nsubs;i <= nsubs; i++)
    {
	Point2 offset =  (i * edge_scale) * search_line;
	
	realno e_i;
	Point2 pos_i = pos_base + offset;
	if (occlusion_map->is_occlusion(pos_i.x, pos_i.y))
	    return EDGE_OCCLUDED;
	
	if (!(*edge_response)((int) pos_i.x, (int) pos_i.y, grid_scale, normal,
			      e_i, tracker->get_active_model()))
	    return EDGE_OCCLUDED;
	
	if (e_i > max_e)
	{
	    max_e = e_i;
	    p_obs = pos_i;
	}
	
    }
    
    if (max_e <= significance_threshold)
	return EDGE_BAD;
    
    // var = 0.25 * edge_scale * edge_scale;
    return EDGE_GOOD;
}

edge_status_t
GenericEdgeDetector::nearest_edge(Point2 &pos, Point2 &search_line,
				  Point2 &normal, realno window_size,
				  OcclusionHandler *occlusion_map,
				  Profile *currx,
				  Point2 &p_obs, realno &var,
				  ActiveModel *active_model)
{
    int nsubs; // sampling size
    
    if (window_size > max_scale)  // this will generally be the case
	nsubs = (int) (0.5 + (window_size / max_scale));
    else
	nsubs = 1;
    
    realno edge_scale = window_size / ((realno) nsubs);
    int grid_scale = (int) (edge_scale + 0.5);
    var = 0;
//    realno max_e = -significance_threshold;
    realno e_i;
    
    Point2 pos_base = pos + Point2(0.5, 0.5);
    p_obs = pos_base;
    
    if (occlusion_map->is_occlusion(pos_base.x, pos_base.y))
	return EDGE_OCCLUDED;  ///  FIXME: nts: was EDGE_BAD --- ????
    
    if (!(*edge_response)((int) pos_base.x, (int) pos_base.y, grid_scale,
			  normal, e_i, tracker->get_active_model()))
	return EDGE_BAD;
    
    if (e_i > significance_threshold)
	return EDGE_GOOD;
    
    for (int i = 1; i <= nsubs; i++)
    {
	Point2 offset =  (i * edge_scale) * search_line;
	Point2 pos_i = pos_base + offset;
	
	if (occlusion_map->is_occlusion(pos_i.x, pos_i.y))
	    return EDGE_OCCLUDED;
	
	if (!(*edge_response)
	    ((int) pos_i.x, (int) pos_i.y, grid_scale, normal, e_i, tracker->get_active_model()))
	    return EDGE_OCCLUDED;
	
	if (e_i > significance_threshold)
	{
	    p_obs = pos_i;
	    return EDGE_GOOD;
	}
	
	pos_i = pos_base - offset;
	if (occlusion_map->is_occlusion(pos_i.x, pos_i.y))
	    return EDGE_OCCLUDED;
	
	if (!(*edge_response)((int) pos_i.x, (int) pos_i.y, grid_scale, normal,
			      e_i, tracker->get_active_model()))
	    return EDGE_OCCLUDED;
	if (e_i > significance_threshold)
	{
	    p_obs = pos_i;
	    return EDGE_GOOD;
	}
    }
    
    return EDGE_BAD;
    
}


SobelEdgeDetector::SobelEdgeDetector(realno thresh,
				     edge_search_method_t mthd,
				     ActiveShapeTracker *the_tracker) :
    GenericEdgeDetector(mthd)
{
    assert (the_tracker != NULL);
    tracker = the_tracker;
    
    set_significance_threshold(thresh);
    
    if (tracker->get_active_model()->get_video_image_type() == GREY8)
	edge_response = &grey_sobel;
    else
	if (tracker->get_active_model()->get_video_image_type() == RGB32)
	    edge_response = &rgb_sobel;
	else
	{
	    cerror << "SobelEdgeDetector::SobelEdgeDetector: "
		   << "Invalid image type, cannot work. " << endl;
	    exit(1);    
	}
}

inline realno SobelEdgeDetector::get_maximum_response()
{
    if (tracker->get_active_model()->get_video_image_type() == GREY8)
	return 4.0 * 255;
    else
	if (tracker->get_active_model()->get_video_image_type())
	    return 2.0 * MAX_RGB_DISTANCE;
	else
	{
	    cerror << "SobelEdgeDetector::get_maximum_response(): "
		   << "Invalid image type, expect trouble" << endl;
	    return 0;
	}
}

SimpleEdgeDetector::SimpleEdgeDetector(realno thresh, edge_search_method_t mthd,
				       ActiveShapeTracker *the_tracker)
    : GenericEdgeDetector(mthd)
{
    assert (the_tracker != NULL);
    tracker = the_tracker;
    
    if (tracker->get_active_model()->get_video_image_type() == GREY8)
	edge_response = &grey_simple;
    else
	if (tracker->get_active_model()->get_video_image_type() == RGB32)
	    edge_response = &rgb_simple;
	else
    	{
	    cerror << "SimpleEdgeDetector::SimpleEdgeDetector: "
		   << "Invalid image type, cannot work. " << endl;
	    exit(1);    
	}

    set_significance_threshold(thresh);
}

inline realno SimpleEdgeDetector::get_maximum_response()
{
    if (tracker->get_active_model()->get_video_image_type() == GREY8)
	return 255;
    else
	if (tracker->get_active_model()->get_video_image_type() == RGB32)
	    return MAX_RGB_DISTANCE;
	else
	{
	    cerror << "SobelEdgeDetector::get_maximum_response(): "
		   << "Invalid image type, expect trouble" << endl;
	    return 0;
	}
}


MovingEdgeDetector::MovingEdgeDetector(realno thresh,
				       edge_search_method_t mthd,
				       ActiveShapeTracker *the_tracker)
    : GenericEdgeDetector(mthd)
{
    assert (the_tracker != NULL);
    tracker = the_tracker;
    
    if (tracker->get_active_model()->get_video_image_type() == RGB32)
	edge_response = &rgb_moving_edge;
    else
	if (tracker->get_active_model()->get_video_image_type() == GREY8)
	    edge_response = &grey_moving_edge;
	else
    	{
	    cerror << "MovingEdgeDetector::MovingEdgeDetector: "
		   << "Invalid image type, cannot work. " << endl;
	    exit(1);    
	}

    set_significance_threshold(thresh);
}

inline realno MovingEdgeDetector::get_maximum_response()
{
    if (tracker->get_active_model()->get_video_image_type() == GREY8)
	return 2.0 * 255;
    else
	if (tracker->get_active_model()->get_video_image_type() == RGB32)
	    return 2.0 * MAX_RGB_DISTANCE;
	else
	{
	    cerror << "MovingEdgeDetector::get_maximum_response(): "
		   << "Invalid image type, expect trouble" << endl;
	    return 0;
	}
}


ForegroundEdgeDetector::ForegroundEdgeDetector(realno thresh_lower,
					       realno thresh_upper,
					       edge_search_method_t mthd,
					       ActiveShapeTracker *the_tracker)
    : GenericEdgeDetector(mthd)
{
    assert (the_tracker != NULL);
    tracker = the_tracker;
    
    if (tracker->get_active_model()->get_video_image_type() == RGB32)
	edge_response = &rgb_foreground_edge;
    else
	edge_response = &grey_foreground_edge;
    
    set_significance_threshold(0.0);

    thresh_background = thresh_lower * get_maximum_response();
    thresh_foreground = thresh_upper * get_maximum_response();
}

realno ForegroundEdgeDetector::get_maximum_response()
{
    if (tracker->get_active_model()->get_video_image_type() == GREY8)
	return MAX_GREY_DISTANCE;
    else
	if (tracker->get_active_model()->get_video_image_type() == RGB32)
	    return MAX_RGB_DISTANCE;
	else
	{
	    cerror << "ForegroundEdgeDetector::get_maximum_response(): "
		   << "Invalid image type, expect trouble" << endl;
	    return MAX_RGB_DISTANCE;
	}
}


// FIXME:  nts: how is this colour distance (2-norm) motivated?
realno ColourBlob::colour_distance(RGB32pixel *col)
{
    if (theColour_valid == false)
	return 0;
    
    int rdiff = col->red - theColour.red;
    int gdiff = col->green - theColour.green;
    int bdiff = col->blue - theColour.blue;
    
    return sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
}


realno ColourBlob::norm_colour_distance(RGB32pixel *col)
{
    if (theColour_valid == false)
	return 0;
    
    realno sum1 = 1 + theColour.red + theColour.green + theColour.blue;
    realno r1 = theColour.red / sum1;
    realno g1 = theColour.green / sum1;
    realno b1 = theColour.blue / sum1;
    
    realno sum2 = 1 + col->red + col->green + col->blue;
    realno r2 = col->red / sum2;
    realno g2 = col->green / sum2;
    realno b2 = col->blue / sum2;
    
    r1 -= r2; g1 -= g2; b1 -= b2;
    return sqrt((float)(r1*r1 + g1*g1 + b1*b1));
}

void ColourBlob::update()
{
    if (newColour_valid == false)
	return;
    
    if (theColour_valid == false)
    {
	theColour_valid = true;
	theColour = newColour;
    }
    else
    {
	// sorry, optimised a bit, less readable now -- nts
	// this is how one of the 3 r,g,b parts looked like before:
	
	// int gdiff = newColour->green - theColour->green;
	// if (ABS(gdiff) > 4)
	//	   gdiff = 4 * ZSGN(gdiff);
	// else
	//	   gdiff = ZSGN(gdiff);
	// theColour -> green += gdiff;
	
	register int rdiff = newColour.red - theColour.red;
	if (rdiff > 0)
	    if (rdiff > 4)
		rdiff = 4;
	    else
		rdiff = 1;
	else
	    if (rdiff != 0)
		if (rdiff < -4)
		    rdiff = -4;
		else
		    rdiff = -1;
	
	theColour.red += rdiff;
	
	register int gdiff = newColour.green - theColour.green;
	if (gdiff > 0)
	    if (gdiff > 4)
		gdiff = 4;
	    else
		gdiff = 1;
	else
	    if (gdiff != 0)
		if (gdiff < -4)
		    gdiff = -4;
		else
		    gdiff = -1;
	
	theColour.green += gdiff;
	
	register int bdiff = newColour.blue - theColour.blue;
	if (bdiff > 0)
	    if (bdiff > 4)
		bdiff = 4;
	    else
		bdiff = 1;
	else
	    if (bdiff != 0)
		if (bdiff < -4)
		    bdiff = -4;
		else
		    bdiff = -1;
	
	theColour.blue += bdiff;
    }
}



void ColourBlob::visualise(realno u)
{
#ifndef NO_DISPLAY
    if (theColour_valid == false)
	return;
    
    if (cblob_glwin == NULLWIN)
    {
	prefsize(200,200);
	// nts: this gives an error in Ygl version 4.x
	// foreground();
	cblob_glwin = winopen("Colour Blobs");
	winset(cblob_glwin);
	reshapeviewport();
	prefsize(200,200);
	winconstraints();
#ifdef USE_GL
	RGBmode();
	gconfig();  // nts: have to do this after RGBmode and before cpack
	cpack(0x00ffffff);
	gflush();
#endif
	gconfig();
#ifdef USE_GL
	RGBcolor(0,0,0);
#endif
	clear();
    }
    
    winset(cblob_glwin);
    
#ifdef USE_GL
    RGBmode();
    gconfig();  // nts: have to do this after RGBmode and before RGBcolor
    RGBcolor((short) theColour.red, (short) theColour.green, (short)
	     theColour.blue);
    
#endif
    circf(100+(50*sin(2 * 3.142 * u)), 100+(50*cos(2 * 3.142 * u)), 10.0);
#endif   // #ifndef NO_DISPLAY
}



edge_status_t
ColourEdgeDetector::find_edge(realno u, Point2 &pos,
			      Point2 &search_line,
			      Point2 &normal, realno window_size,
			      OcclusionHandler *occlusion_map,
			      Profile *currx,
			      Point2 &p_obs, realno &var,
			      ActiveModel *active_model)
{
    int indx = (int) (u * no_cblobs);
    if (indx >= no_cblobs)	// should not happen
    {
	cerror << " ColourEdgeDetector::find_edge(): indx out of range " << endl;
	exit(1);
    }
    current_blob = &cblobs[indx];
    
#ifdef DEBUG
    if (tracker->debug_level == 1)
    {
	current_blob->visualise(indx / (realno) no_cblobs);
    }    
#endif
    
    int nsubs; // sampling size
    
    if (window_size > max_scale)  // this will generally be the case
	nsubs = (int) (0.5 + (window_size / max_scale));
    else
	nsubs = 1;
    
    realno edge_scale = window_size / ((realno) nsubs);
    int grid_scale = (int) (edge_scale + 0.5);
    var = 0;
    realno max_e = -significance_threshold;
    
    Point2 pos_base = pos + Point2(0.5, 0.5);
    Point2 best_offset(0,0);
    
    RGB32pixel *best_pix = NULL;
//   realno sum_e = 0;
//   realno sum_d = 0;
//   realno sq_sum = 0;
    for (int i = -nsubs; i <= nsubs; i++)
    {
//	realno d_i = i * edge_scale;
	Point2 offset =  (i * edge_scale) * search_line;;
	
	realno e_i;
	Point2 pos_i = pos_base + offset;
	
	if (occlusion_map->is_occlusion(pos_i.x, pos_i.y))
	    return EDGE_OCCLUDED;
	
	RGB32pixel *inside_pix =
	    edge_response((int) pos_i.x, (int) pos_i.y, grid_scale, normal, e_i);
	//if (inside_pix == NULL) return EDGE_OCCLUDED;
	
	realno new_e_i = 0;
	if ((inside_pix != NULL) && (e_i > significance_threshold))
	{
	    new_e_i = e_i;
	    if (e_i > max_e)
	    {
		max_e = e_i;
		best_pix = inside_pix;
		best_offset = offset;
	    }
	}
//       sum_e += new_e_i;
//       sum_d += d_i * new_e_i;
//       sq_sum += d_i * d_i * new_e_i;
	
    }
    
    if (best_pix == NULL)
	return EDGE_BAD;
    
    current_blob->add_observation(best_pix);
    
//   realno mean = sum_d / sum_e;
//   p_obs = pos + mean * search_line;
//   var = 0.25 * ((sq_sum / sum_e) - (mean * mean));
    
    p_obs = pos + best_offset;
    
    return EDGE_GOOD;
}

void ColourEdgeDetector::setup_batch(Profile *curr)
{
    if (curr->colour_info == NULL)
    {
	curr->colour_info = new ColourBlob[no_cblobs];
	curr->no_cblobs = no_cblobs;  // remember array size
    }
    else
    {
	for (int i = 0; i < no_cblobs; i++)
	    curr->colour_info[i].update();
    }
    
    cblobs = curr->colour_info;
    this->EdgeDetector::setup_batch(curr);
}


ColourForegroundEdgeDetector::ColourForegroundEdgeDetector(realno thresh_lower, realno thresh_upper,
							   ActiveShapeTracker *the_tracker, int the_no_cblobs)
    : ColourEdgeDetector(the_no_cblobs)
{
    assert (the_tracker != NULL);
    tracker = the_tracker;
    
    set_significance_threshold(0.0);

    thresh_background = thresh_lower * get_maximum_response();
    thresh_foreground = thresh_upper * get_maximum_response();
}


RGB32pixel*
ColourForegroundEdgeDetector::edge_response(realno x, realno y,
					    int step, Point2 n,
					    realno &edge)
{
    // FIXME: could use other methods than sqrt of sum squared for differencing
    
    int dx = real_to_int(n.x * step);
    int dy = real_to_int(n.y * step);
    
    int xr = (int) (x + dx);
    int yr = (int) (y + dy);
    int xl = (int) (x - dx);
    int yl = (int) (y - dy);

    Image *vidimage = tracker->get_active_model()->get_video_image();
    Image *refimage = tracker->get_active_model()->get_background_image();

    // check whether either of the points is off image
    if ((vidimage->check_coords(xr,yr) == false) || (vidimage->check_coords(xl,yl) == false))
	return NULL;
    
#ifdef DEBUG
     if (tracker->debug_level == 1)
 	cdebug << "edge (" << x << "," << y
 	     << "): step +/- (" << dx << "," << dy << ")  ";
#endif
    
    realno leftdiff, rightdiff;
    RGB32pixel *vid_leftpixel, *vid_rightpixel;
    RGB32pixel *ref_leftpixel, *ref_rightpixel;
    RGB32pixel *diff_leftpixel, *diff_rightpixel;
    
    // this is the position to test whether it is inside the person:
    vid_leftpixel = (RGB32pixel*) vidimage->get_pixel(xl,yl);
    
    // check whether a pre-calculated difference image is available (else we get NULL)
    Image *diffimage = tracker->get_active_model()->get_difference_image();
    
    if (diffimage != NULL)   // we have a precalculated difference image!
    {
	// we use these pixels in the difference image...
	diff_leftpixel = (RGB32pixel*) diffimage->get_pixel(xl,yl);
	diff_rightpixel = (RGB32pixel*) diffimage->get_pixel(xr,yr);
	
	// measure for left pixel...
	leftdiff = sqrt((float)(SQUARE(diff_leftpixel->red) +
				SQUARE(diff_leftpixel->green) +
				SQUARE(diff_leftpixel->blue)));
	
	// measure for right pixel...
	rightdiff = sqrt((float)(SQUARE((int)diff_rightpixel->red) +
				 SQUARE((int)diff_rightpixel->green) +
				 SQUARE((int)diff_rightpixel->blue)));
	
#ifdef DEBUG
  	if (tracker->debug_level == 1)
  	    cdebug << " pre: ";  // tell user where the numbers come from
#endif
    }
    else  // we don't have precalculated difference so calculate on the fly
    {
	// we need these pixels...
	vid_rightpixel = (RGB32pixel*) vidimage->get_pixel(xr,yr);
	ref_leftpixel = (RGB32pixel*) refimage->get_pixel(xl,yl);
	ref_rightpixel = (RGB32pixel*) refimage->get_pixel(xr,yr);
	
	// measure for left pixel...
	int rdiff = vid_leftpixel->red - ref_leftpixel->red;
	int gdiff = vid_leftpixel->green - ref_leftpixel->green;
	int bdiff = vid_leftpixel->blue - ref_leftpixel->blue;
	
	leftdiff = sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
	
	// measure for right pixel...
	rdiff = vid_rightpixel->red - ref_rightpixel->red;
	gdiff = vid_rightpixel->green - ref_rightpixel->green;
	bdiff = vid_rightpixel->blue - ref_rightpixel->blue;
	
	rightdiff = sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
	
#ifdef DEBUG
  	if (tracker->debug_level == 1)
  	    cdebug << " fly: ";  // tell user where the numbers come from
#endif
    }
    
    // the calculation of the difference is the same...
    
    if ((leftdiff <= thresh_foreground) || (rightdiff >= thresh_background))
    {
	edge = 0;
	
#ifdef DEBUG
  	if (tracker->debug_level == 1)
  	    cdebug << "edge = 0  because "
  		 << leftdiff << " <= " << thresh_foreground << " or "
  		 << rightdiff << " >= " << thresh_background << endl;
#endif
    }
    else
    {
	realno col_dist = current_blob->colour_distance(vid_leftpixel);
	
	if (col_dist == 0)  // seems we don't have colour blobs yet
	{
	    edge = (leftdiff - rightdiff);   //  so we cannot use that info
	    
#ifdef DEBUG
  	    if (tracker->debug_level == 1)
  		cdebug << "edge = " << (leftdiff - rightdiff) << endl;
#endif
	}
	else
	{
	    // we know what colour to expect inside the person ("left") so
	    // let's use this to make tracking (hopefully) more robust...
	    edge = (2 * (leftdiff - rightdiff) - col_dist) / 3;
	    
#ifdef DEBUG
  	    if (tracker->debug_level == 1)
  		cdebug << "edge = (2 * (" << leftdiff << "-" << rightdiff
  		     << ") - " << col_dist << ") / 3  == " << edge << endl;
#endif
	}
    }
    
    return vid_leftpixel;
}

inline realno ColourForegroundEdgeDetector::get_maximum_response()
{
    return // nts: changed weighting so had to change this...  was " 2.0 * "
	MAX_RGB_DISTANCE;
}


NormalisedColourForegroundEdgeDetector::NormalisedColourForegroundEdgeDetector(realno thresh_lower,
									       realno thresh_upper,
									       ActiveShapeTracker *the_tracker,
									       int the_no_cblobs)
    : ColourEdgeDetector(the_no_cblobs)
{
    assert (the_tracker != NULL);
    tracker = the_tracker;
    
    set_significance_threshold(0.0);

    thresh_background = thresh_lower * get_maximum_response();
    thresh_foreground = thresh_upper * get_maximum_response();
}

RGB32pixel *NormalisedColourForegroundEdgeDetector::edge_response(realno x, realno y,
								  int step, Point2 n,
								  realno &edge)
{
    int dx = real_to_int(n.x * step);
    int dy = real_to_int(n.y * step);
    
    int xr = (int) (x + dx);
    int yr = (int) (y + dy);
    int xl = (int) (x - dx);
    int yl = (int) (y - dy);
    
    Image *vidimage = tracker->get_active_model()->get_video_image();
    Image *refimage = tracker->get_active_model()->get_background_image();

    if (!((vidimage->check_coords(xr,yr)) && (vidimage->check_coords(xl,yl))))
	return NULL;
    
    RGB32pixel *vid_rightpixel = (RGB32pixel*) vidimage->get_pixel(xr,yr);
    RGB32pixel *ref_rightpixel = (RGB32pixel*) refimage->get_pixel(xr,yr);
    RGB32pixel *vid_leftpixel = (RGB32pixel*) vidimage->get_pixel(xl,yl);
    RGB32pixel *ref_leftpixel = (RGB32pixel*) refimage->get_pixel(xl,yl);
    
    realno sum1 = 1 +
	vid_rightpixel->red + vid_rightpixel->green + vid_rightpixel->blue;
    realno sum2 = 1 +
	ref_rightpixel->red + ref_rightpixel->green + ref_rightpixel->blue;
    realno sum3 = 1 +
	vid_leftpixel->red + vid_leftpixel->green + vid_leftpixel->blue;
    realno sum4 = 1 +
	ref_leftpixel->red + ref_leftpixel->green + ref_leftpixel->blue;
    
    realno rdiff = (vid_rightpixel->red / sum1) - (ref_rightpixel->red / sum2);
    realno gdiff = (vid_rightpixel->green / sum1) - (ref_rightpixel->green / sum2);
    realno bdiff = (vid_rightpixel->blue / sum1) - (ref_rightpixel->blue / sum2);
    realno rightdiff = sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
    
    rdiff = (vid_leftpixel->red / sum3) - (ref_leftpixel->red / sum4);
    gdiff = (vid_leftpixel->green / sum3) - (ref_leftpixel->green / sum4);
    bdiff = (vid_leftpixel->blue / sum3) - (ref_leftpixel->blue / sum4);
    realno leftdiff = sqrt((float)(rdiff * rdiff + gdiff * gdiff + bdiff * bdiff));
    
    if ((leftdiff <= thresh_foreground) || (rightdiff >= thresh_background))
	edge = 0;
    else
    {
	edge = (leftdiff - rightdiff) +
	    MAX_NORM_RGB_DISTANCE -
	    current_blob->norm_colour_distance(vid_leftpixel);
    }
    
    return vid_leftpixel;
}

inline realno NormalisedColourForegroundEdgeDetector::get_maximum_response()
{
    return 2.0 * MAX_NORM_RGB_DISTANCE;
}

} // namespace ReadingPeopleTracker
