///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Camera.h  ---  A class which holds everything connected with a camera    //
//                                                                           //
//  A Camera object holds everything associated with a camera:               //
//   Inputs, Calibration, the actual Tracking class which generates and      //
//   stores results, a ConfigurationManager class etc.                       //
//                                                                           //
//  The Camera class starts a thread for each Camera class and waits for     //
//   input to be fed by the parent.  Results from Tracking, are generated    //
//   and the availability of new Results is signalled.                       //
//                                                                           //
//  Authors   : Initial version with no threads by Philip Elliott (pte)      //
//              redesign and all further changes by Nils T Siebel (nts)      //
//  Created   : Initial working revision on Tue Sep 25 16:39:45 BST 2001     //
//  Revision  : 1.2 of Thu Nov 29 16:28:21 GMT 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef __CAMERA_H__
#define __CAMERA_H__

// NB: under WIN32, please use the free ``pthreads for win32'' library,
//     see http://sources.redhat.com/pthreads-win32/
#include <pthread.h>  // for IEEE POSIX 1003.1c --- 1995 threads
#include <cassert>

#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

// forward declarations...
class ConfigurationManager;
class Inputs;
class ScreenOutput;
class Results;
class MovieStore;
class Tracking;
class Image;
class RegionSet;

// Camera class: contains all information individual to a camera and has a thread to process

class Camera
{    
    friend class PeopleTracker;
    
private:
    bool camera_enabled;  // whether do_processing() should be active
    bool quiet_mode;      // whether to disable all screen output and user input
    
    ConfigurationManager *configuration_manager;

    // POSIX threading stuff
    pthread_t thread_id;  // unique thread id: this will usually be an unsigned long int
    
    // this is the threaded method waiting for data and doing all the processing
    void *do_processing(void *unused);

    // we require some overhead here ...
    static void *start_processing(void *camera_object_ptr)
	{
	    assert(camera_object_ptr != NULL);
	    return ((Camera *) camera_object_ptr) -> do_processing(NULL);
	}

    camera_id_t camera_id;    // unique numerical camera id which does not change over time
    char *camera_name;        // unique camera name eg. HALL01.C02
    char *camera_description; // full textual camera name (description string)

    frame_id_t max_keep_alive;  // how many frames to keep unseen predicted objects
    frame_id_t min_draw_age;    // how many frames to wait before drawing new objects
    
    realno camera_confidence;
    
    // current frame id and frame id increment per frame
    frame_id_t current_frame_id;
    frame_id_t frame_id_delta;

    frame_time_t current_frame_time_in_ms;
    frame_id_t frame_count;  // wraparound in 27 years at 5 fps --- OK.
    
    // classes
    Inputs *inputs;
    
    ScreenOutput *screen_output;

    Tracking *tracking;

    // results from current frame (being generated) and from last frame (completed)
    Results *results;
    Results *previous_results;

    // pointers to current external image and XML RegionSet input data
    Image *current_video_image;
    RegionSet *current_external_xml_region_set;

    // output movie (video image with overlayed tracking results) for demos
    char *output_movie_filename;
    MovieStore *output_movie;

    // output movie (binary motion image with motion detection results) for demos
    char *motion_movie_filename;
    MovieStore *motion_movie;

public:
    // Constructor and Destructor
    Camera(char* camera_config_filename, bool the_quiet_mode = false);

    ~Camera();

    // start a thread which does all processing as data arrives.  returns thread id
    pthread_t start_thread();
    
    // Accessors and Modifiers
    inline char *get_camera_name() const
	{
	    return camera_name;
	}
    inline camera_id_t get_camera_id() const
	{
	    return camera_id;
	}

    inline frame_id_t get_frame_id() const
	{
	    return current_frame_id;
	}

    inline frame_time_t get_frame_time_in_ms() const
	{
	    return current_frame_time_in_ms;
	}

    // check this to find out end of video input sequence
    inline bool get_camera_enabled() const
	{
	    return camera_enabled;
	}

private:

    // private helpers

    // advance frame id by frame_id_delta, handling wraparound
    void calculate_next_frame_id();
    
    // retrieve new data sets if they are available
    void get_new_data_sets();

    // register our configuration parameters with 
    void register_configuration_parameters();
};

} // namespace ReadingPeopleTracker

#endif
