///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Camera.cc  ---  A class which holds everything connected with a camera   //
//                                                                           //
//  A Camera object holds everything associated with a camera:               //
//   Inputs, Calibration, the actual Tracking class which generates and      //
//   stores results, a ConfigurationManager class etc.                       //
//                                                                           //
//  The Camera class starts a thread for each Camera class and waits for     //
//   input to be fed by the parent.  Results from Tracking, are generated    //
//   and the availability of new Results is signalled.                       //
//                                                                           //
//  Authors   : Initial version with no threads by Philip Elliott (pte)      //
//              Redesign and all further changes by Nils T Siebel (nts)      //
//  Created   : Initial working revision on Tue Sep 25 16:39:45 BST 2001     //
//  Revision  : 1.2 of Thu Nov 29 16:28:21 GMT 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <limits.h>
#include <string.h>  // for strrchr()

// we want to use sched_yield() if available.  POSIX systems on which
// sched_yield is available  define _POSIX_PRIORITY_SCHEDULING in <unistd.h>.
#ifndef WIN32
#include <unistd.h>
#endif

#ifdef _POSIX_PRIORITY_SCHEDULING
#include <sched.h>
#else
#ifdef __GNUC__
#warning POSIX 1003.1b-1993 (aka POSIX.4) sched_yield() seems to be unavailable, disabling it...
#endif    // ifdef  __GNUC__
#endif    // ifdef  _POSIX_PRIORITY_SCHEDULING else

#include "Camera.h"

#include "Tracking.h"
#include "ConfigurationManager.h"
#include "Inputs.h"
#include "ScreenOutput.h"
#include "MovieStore.h"
#include "Results.h"
#include "text_output.h"
#include "MotionDetector.h"
#include "ActiveShapeTracker.h"

namespace ReadingPeopleTracker
{

static const char *Camera_Revision = "@(#) Camera.cc, rev 1.2 of Thu Nov 29 16:28:21 GMT 2001, Authors Philip Elliott and Nils T Siebel, Copyright (c) 2001--2002 The University of Reading";


Camera::Camera(char *camera_config_filename, bool the_quiet_mode)
{
    camera_name = NULL;
    camera_description = NULL;
    quiet_mode = the_quiet_mode;
    
    // we cannot run without configuration file
    if (camera_config_filename == NULL)
    {
	cerror << "Camera::Camera: No configuration file specified. " << endl;
	exit(1);
    }
    
    //setup configuration manager
    configuration_manager = new ConfigurationManager;
    
    // register our own configuration parameters
    register_configuration_parameters();
    
    //create classes for storage of the configuration data
    //constructors for the classes register their variables with configuration_manager
    inputs = new Inputs(configuration_manager);

    if (quiet_mode == false)
	screen_output = new ScreenOutput(configuration_manager);
    else
	screen_output = NULL;
    
    tracking = new Tracking(configuration_manager);
    
    //Read parameter file data into their respective homes
    configuration_manager->parse_parameter_file(camera_config_filename);
    
    // replace underscores with space in CameraConfiguration's camera_name.
    // this is because the file cannot have spaces in the variable values
    if (camera_description != NULL)
	for (char *c = camera_description; *c != '\0'; c++)
	    if (*c == '_')
		*c = ' ';
    
    // allocate Results classes
    results = new Results;
    previous_results = new Results;
   
    // remember that the current (empty) previous_results need no more processing
    previous_results->status = RESULTS_PROCESSED;
     
    frame_id_delta = 1;  // default to frame id increment 1 per frame
    
///////////////////////// Set up All Inputs ///////////////////////////
    
    // sets up video input, external motion detection input, camera calibration
    inputs->setup_inputs();
    
    // initialise pointers to current external image and XML RegionSet input data
    current_video_image = NULL;
    current_external_xml_region_set = NULL;
    
//////////////////////// Set up Output Movie (results) //////////////////////////
    
    output_movie = NULL;  // default to NULL (no output movie)
    
    if ((output_movie_filename != NULL) && (output_movie_filename[0] != '\0'))
    {
	// try detecting the desired format by checking the filename extension
	char *file_ext = strrchr(output_movie_filename, '.');
	
	if ((file_ext == NULL) ||
	    (strcasecmp(file_ext,".pnm") == 0) || 
	    (strcasecmp(file_ext,".ppm") == 0) || 
	    (strcasecmp(file_ext,".pgm") == 0) || 
	    (strcasecmp(file_ext,".pbm") == 0))
	{
	    output_movie = new MovieStore (MOVIESTORE_PNM, output_movie_filename);
	}
	else
	    if ((strcasecmp(file_ext,".jpeg") == 0) || 
		(strcasecmp(file_ext,".jpg") == 0))
            {
                output_movie = new MovieStore (MOVIESTORE_JPEG, output_movie_filename);
	    }
	    else
	    {
		cerror << "Camera::Camera: Could not detect file format for output movie named "
		       << output_movie_filename << " " << endl;
		exit(1);
	    }
    }
    
//////////////////////// Set up Motion Movie //////////////////////////
    
    motion_movie = NULL;  // default to NULL (no motion movie)
    
    if ((motion_movie_filename != NULL) && (motion_movie_filename[0] != '\0'))
    {
	// try detecting the desired format by checking the filename extension
	char *file_ext = strrchr((char *)motion_movie_filename, '.');
	
	if ((file_ext == NULL) ||
	    (strcasecmp(file_ext,".jpeg") == 0) || 
	    (strcasecmp(file_ext,".jpg") == 0))
            {
                motion_movie = new MovieStore (MOVIESTORE_JPEG, motion_movie_filename);
	    }
	else
	    if ((strcasecmp(file_ext,".pnm") == 0) || 
		(strcasecmp(file_ext,".ppm") == 0) || 
		(strcasecmp(file_ext,".pgm") == 0) || 
		(strcasecmp(file_ext,".pbm") == 0))
	    {
                motion_movie = new MovieStore (MOVIESTORE_PNM, motion_movie_filename);
	    }
	    else
	    {
		cerror << "Camera::Camera: Could not detect file format for motion movie named "
		       << motion_movie_filename << " " << endl;
		exit(1);
	    }
    }
//////////////////////////// Set up Trackers ///////////////////////////
    
    // trackers use our configuration filename as a base for their filenames
    tracking->setup_motion_detector_and_trackers(inputs, results, camera_config_filename);
    
    // remember that we still need to start the thread by setting thread_id to 0
    thread_id = 0;
   
    camera_enabled = false;   // we are not set up yet, there is no processing thread
 
    frame_count = 0;
}


//
// this is the threaded method waiting for data and doing all the processing
//
void *Camera::do_processing(void *unused)
{
    // first set cancellation parameters: we would like to enable cancelling of
    //   this thread but only after results for the current frame are generated
    int old_cancel_state;
    int old_cancel_type;
    
    // cancellation enabled but deferred until we call pthread_testcancel() ...
    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &old_cancel_state);
    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, &old_cancel_type); 
   
    camera_enabled = true;   // flag which can be checked by parent class (PeopleTracker)
 
    while (true)
    {
	// advance frame id by frame_id_delta, handling wraparound
	calculate_next_frame_id();    // ... which updates current_frame_id and time
	
	// check whether last frame id is reached
	if (current_frame_id > inputs->get_last_frame_id())
	{
	    // end of sequence requested by configuration file
	    cinfo << " Camera::do_processing: End of video sequence detected, disabling camera. "
		  << endl;
	    
	    camera_enabled = false;   // NB flag to be checked by parent class (PeopleTracker) !
	    
	    thread_id = 0;    // signal no thread present	    
	    
	    pthread_exit(0);  // EXIT THREAD
	    
	    /* NOTREACHED */
	}

	// proceed all inputs to the new current_frame_id
	frame_id_t actual_new_frame_id = inputs->proceed_to_frame(current_frame_id);
	
	cdebug << " >>>> Camera::do_processing: next frame calculated as "
	       << current_frame_id 
	       << ", got " << actual_new_frame_id
	       << "  <<<< " << endl;
	
	// mark current results as not yet finished
	results->status = RESULTS_UNDEFINED;

	// check whether last frame id is reached
	if (actual_new_frame_id > inputs->get_last_frame_id())
	{
	    // end of sequence requested by configuration file
	    cinfo << " Camera::do_processing: End of video sequence detected, disabling camera. "
		  << endl;
	    
	    camera_enabled = false;   // NB flag to be checked by parent class (PeopleTracker) !
	    
	    thread_id = 0;    // signal no thread present	    
	    
	    pthread_exit(0);  // EXIT THREAD
	    
	    /* NOTREACHED */
	}

	// if the video image source is missing a frame, actual_new_frame_id is larger
	if (actual_new_frame_id > current_frame_id)
	{
	    // no video image for this frame  ---  skip frame
	    cerror << " video image for frame " << current_frame_id 
		   << " unavailable (got frame " << current_video_image->get_frame_id()
		   << "). skipping frame... " << endl;
	    
	    cdebug << " >>>>>>>>>>>>>>>>>   FIXME:  PREDICT ALL.  ETC?   <<<<<<<<<<<<<< " << endl;
	    
	    continue;
	}
	
	// retrieve new data sets (pointers to images / RegionSet) if they are available
	get_new_data_sets();
	
	// We have requested one data set each.  Let's see what do we have (missing data?)
	if (current_video_image == NULL)
	{
	    // assume no more video images (end of sequence)
	    cinfo << " Camera::do_processing: End of video sequence detected, disabling camera. "
		  << endl;
	    
	    camera_enabled = false;   // NB flag to be checked by parent class (PeopleTracker) !
	    
	    thread_id = 0;    // signal no thread present	    
	    
	    pthread_exit(0);  // EXIT THREAD
	    
	    /* NOTREACHED */
	}
	
	assert (actual_new_frame_id == current_frame_id);                 // should hold after the above
	assert (current_video_image->get_frame_id() == current_frame_id); // should hold after the above
	
	// run motion detector (or use external MD input if available) for motion images
	tracking->generate_motion_image(inputs, results);
	Image *motion_image;
	
	// make a copy of the motion image if it needs to be stored in a movie later 
	if (motion_movie != NULL)
	    motion_image = results->get_motion_image()->copy();
	else
	    motion_image = NULL;   // unused, set to NULL to detect possible use
	    
#ifndef NO_DISPLAY

	// all images are now generated.  update image displays:
	if (quiet_mode == false)
	    screen_output->update_displays(inputs, results);
#endif

	// now we can extract the motion data from the motion images (NB clears the image!)
	tracking->extract_motion_data(inputs, results);

#ifndef NO_DISPLAY
	// draw motion detection output into the motion image (on screen)
	if (quiet_mode == false)
	    screen_output->draw_motion_data(inputs, results);
#endif
	
	// make motion movie if requested
	if (motion_movie != NULL)
	{
	    // use the copy of the motion_image saved above
	    motion_image->set_draw_colour(127);

	    frame_id_t motion_image_frame_id = results->get_motion_image()->get_frame_id();	    
	    
	    ListNode<TrackedObject> *object;
	    ListNode<Region> *region;
	    
	    // for each object in the results:
	    for (object = results->get_tracked_objects()->start();
		 object != NULL; object = object->next)
	    {
		// if object is visible then draw all measured regions in the object:
		
		for (region = object->dat->regions->first;
		     region != NULL; region = region->next)
		{
		    if ((region->dat->source == MEASUREMENT) &&
			(region->dat->frame_last_detected == motion_image_frame_id))
		    {
			motion_image->draw_horizontal(region->dat->ylo,
						      region->dat->xlo, region->dat->xhi);
			motion_image->draw_horizontal(region->dat->yhi,
						      region->dat->xlo, region->dat->xhi);
			motion_image->draw_vertical(region->dat->xlo,
						    region->dat->ylo, region->dat->yhi);
			motion_image->draw_vertical(region->dat->xhi,
						    region->dat->ylo, region->dat->yhi);
			
		    }
		}
	    }
	    
	    motion_movie->add_a_frame(motion_image);
	    
	    delete motion_image;
	}
	    
	cdebug << " >>>> Camera::do_processing: "
	       << results->get_tracked_objects()->no_items << " new measurements in results, " 
	       << previous_results->get_tracked_objects()->no_items << " objects in previous_results. " 
	       << " Running trackers...  <<<< " << endl;
	
	// do actual tracking
	tracking->run_trackers(inputs, results);	

	cdebug << " >>>> Camera::do_processing: ...done. "
	       << " got Results for frame " << current_frame_id
	       << " <<<< " << endl;

	// check whether all objects have valid (>0) object ids, create if not.
	results->get_tracked_objects()->check_and_create_ids();

	// set the frame id which Results correspond to
	results->frame_id_of_results = current_frame_id;
	
	// now remove old tracks and those that were marked for deletion
	results->get_tracked_objects()->remove_old_tracks(current_frame_id,
							  max_keep_alive);

	//
	// now results are complete
	//
	results->status = RESULTS_COMPLETE;

	// wait until the previous results have been processed/output by PeopleTracker
	if (previous_results->status != RESULTS_PROCESSED)       // unlikely, but check
	{
	    // previous_results not processed yet: wait for signal that they are
	    pthread_mutex_lock(&previous_results->status_modification);
	    //
	    while (previous_results->status != RESULTS_PROCESSED)
	    {
		pthread_cond_wait(&previous_results->status_has_changed,
				  &previous_results->status_modification);
	    }
	    //
	    pthread_mutex_unlock(&previous_results->status_modification);
	}
	
	// now we can erase previous_results and copy results there,
	// indicating the completeness of Results to PeopleTracker thread which is waiting
	//
	// NB the operator= does the signaling that previous_results->status has changed
	*previous_results = *results;
	

	cdebug << " >>>> Camera::do_processing: results have "
	       << results->get_tracked_objects()->no_items  << " tracked objects . "
	       << "  Drawing...  <<<< " << endl;

#ifndef NO_DISPLAY
#ifdef DEBUG
	// if trackers have drawn debug/demo info then we should clear that
	if (((tracking->get_active_shape_tracker() != NULL) &&
	     (tracking->get_active_shape_tracker()->get_debug_level() > 0)) ||
// 	    ((tracking->get_region_tracker() != NULL) &&
// 	     (tracking->get_region_tracker()->get_debug_level() > 0)) ||
// 	    ((tracking->get_human_feature_tracker() != NULL) &&
// 	     (tracking->get_human_feature_tracker()->get_debug_level() > 0)) ||
	    ((tracking->get_motion_detector() != NULL) &&
	     (tracking->get_motion_detector()->get_debug_level() > 0)))
	{
	    // redraw images
	    if (quiet_mode == false)
		screen_output->update_displays(inputs, results);
	}
#endif
	
	// draw results into images (which were displayed above)
	if (quiet_mode == false)
	    screen_output->draw_results(inputs, results, min_draw_age);
	// NB using previous_results would be a race

        if (quiet_mode == false)
            gflush();
         usleep(700*1000);      /////////////////////////////////////////////
	
#ifdef DEBUG
	if (quiet_mode == false)
	    gflush();
	// usleep(300*1000);      /////////////////////////////////////////////
#endif   //  ifdef DEBUG
	
#endif   //  ifndef NO_DISPLAY

	// make output movie if requested
	if (output_movie != NULL)
	{
	    results->get_tracked_objects()->draw_all_in_image
		(inputs->get_video_image(), min_draw_age);

	    output_movie->add_a_frame(inputs->get_video_image());
	}
	
	// create a cancelation point.  We will be cancelled here iff our parent
	//   (PeopleTracker) requests it using pthread_cancel( <our thread_id> )
	//   This might happen if they want to disable this camera.
	pthread_testcancel();
	
	// check whether POSIX 1003.1b-1993 (aka POSIX.4) scheduling is avaliable...
#ifdef _POSIX_PRIORITY_SCHEDULING
	// yield CPU to other cameras or to PeopleTracker to finish their tracking...
	sched_yield();
#endif
	
	frame_count++;
    }
    
    /* NOTREACHED */
}


// start a thread which does all processing as data arrives.  returns thread id
pthread_t Camera::start_thread()
{
    if (thread_id != 0)
    {
	// we have been called before.  check whether the thread had been cancelled...
	
	// in order to check that, we will try to cancel the thread ourselves:
	//   FIXME: can you think of another way of doing this w/out using another mutex?
	int return_value = pthread_cancel(thread_id);
	
	if (return_value == 0)
	{
	    // no error occurred => thread exists and it was running!
	    // issue an error message and start it again.
	    cerror << " Camera " << camera_name
		   << " start_thread() requested when thread was already running! " << endl;
	}
	else
	{
	    // thread_id or cancellation is invalid: maybe we have been cancelled before
	    cinfo << " restarting Camera  " << camera_name << " ... " << endl;
	}
    }
    
    // we require some overhead here, see static member function start_processing
    int result = pthread_create(&thread_id, NULL, &Camera::start_processing, this);
    
    if (result != 0)
    {
	// an error occurred
	cerror << " could not start thread for camera " << camera_name
	       << ",  error code " << result << ". "<< endl;
    }
    
    return thread_id;
}

Camera::~Camera()
{
    // cancel the thread if we have one
    if ((thread_id != 0) &&              // we have a thread
	(pthread_self() != thread_id))   // and it is not us (just checking...)
	pthread_cancel(thread_id);
    
    if (inputs != NULL)  delete inputs;
    if (screen_output != NULL)  delete screen_output;
    if (configuration_manager != NULL)  delete configuration_manager;
    if (tracking != NULL)  delete tracking;
    if (results != NULL)  delete results;
    if (previous_results != NULL)  delete previous_results;
    if (output_movie != NULL) delete output_movie;
}

void Camera::calculate_next_frame_id()
{
    // determine next (target) frame id
    if (frame_count == 0)  // first frame?
    {
	// set first current_frame_id according to current (first) video image
	current_frame_id = inputs->get_frame_id();
    }
    else
    {
	// increment current_frame_id by frame_id_delta (which is most probably 1)
	current_frame_id += frame_id_delta;
	
#if (MAX_FRAME_ID - FRAME_ID_T_MAX) > 0
	// the following makes no sense if MAX_FRAME_ID == FRAME_ID_T_MAX
	if (current_frame_id > MAX_FRAME_ID)
	{
	    // wraparound frame id
	    current_frame_id %= (MAX_FRAME_ID + 1);
	    
	    // make sure we can wrap aroud the other sources (should be no problem)
	    assert (MAX_FRAME_ID > frame_id_delta + MAX_EXPECTED_FRAME_SKIP);
	}
#endif   // if (MAX_FRAME_ID - FRAME_ID_T_MAX) > 0
    }
    
    // get current frame time from inputs
    current_frame_time_in_ms = inputs->get_frame_time_in_ms();
}

void Camera::get_new_data_sets()
{
    // get next image / XML RegionSet from each of the 4 inputs as necessary
    //   the Inputs class will set the pointers to NULL if they are not available
    current_video_image = inputs->get_video_image();

    cdebug << " >>>> Camera::get_new_data_sets: "
	   << " calling inputs->get_external_xml_region_set()...  <<<< "
	   << endl;    

    current_external_xml_region_set = inputs->get_external_xml_region_set();

    if (current_external_xml_region_set == NULL)
	cdebug << " >>>> Camera::get_new_data_sets: "
	       << " no external_xml_region_set available  <<<< "
	       << endl;
    else
	cdebug << " >>>> Camera::get_new_data_sets: "
	       << " external_xml_region_set contains "
	       << current_external_xml_region_set->no_items
	       << " entries:- "
	       << *current_external_xml_region_set
	       << " <<<< "
	       << endl;
}



void Camera::register_configuration_parameters()
{
    camera_id = (camera_id_t)
	configuration_manager->register_int("CAMERA_ID", 0,
					    (int *) &camera_id,
					    true, "Camera", 
					    "Unique numerical camera id, constant over time.");
    
    camera_description =
	configuration_manager->register_string("CAMERA_DESCRIPTION",
					       "Camera_NN", &camera_description,
					       true, "Camera",
					       "Meaningful textual description of the camera.  Use underscore for space.");
    
    camera_confidence =
	configuration_manager->register_real("CAMERA_CONFIDENCE", 1.0,
					     &camera_confidence,
					     false, "Camera",
					     "Accuracy weighting used during data fusion.  Unused for now, please set to 1.0.");   
    
    output_movie_filename =
	configuration_manager->register_string("OUTPUT_MOVIE_FILENAME",
					       NULL, &output_movie_filename,
					       true, "Camera",
					       "Filename for output movie (video with overlayed results) for demos.");
    
    motion_movie_filename =
	configuration_manager->register_string("MOTION_MOVIE_FILENAME",
					       NULL, &motion_movie_filename,
					       true, "Camera",
					       "Filename for output movie (motion detection data) for demos.");
    
    max_keep_alive =
	configuration_manager->register_int("MAX_KEEP_ALIVE", 2,
					    (int *) &max_keep_alive,
					    false, "Camera",
					    "Keep unseen tracked objects for this many frames.");

    min_draw_age =
	configuration_manager->register_int("MIN_DRAW_AGE", 1,
					    (int *) &min_draw_age,
					    false, "Camera",
					    "Display new objects after this many frames (0 = always draw).");

}

} // namespace ReadingPeopleTracker
