////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//  Calibration.h                                                             //
//                                                                            //
//  This class reads a calibration (4x3 projection) matrix from file and      //
//    provides methods to use this calibration data for image measurements    //
//                                                                            //
//  The matrix file must be in NAG format, image addressing mode is expected  //
//    to be IA_TOP_TO_BOTTOM, and using the unit [cm] for world coordinates.  //
//                                                                            //
//  Author    : Rmi Belin (rb)  supervised by  Nils T Siebel (nts)           //
//  Created   : Wed Jul 18 13:50:21 BST 2001                                  //
//  Revision  : 1.0 of Tue Jul 31 11:37:44 BST 2001                           //
//  Copyright : The University of Reading                                     //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#ifndef __CALIBRATION_H__
#define __CALIBRATION_H__

#include <fstream>

#include "NagMatrix.h"
#include "Point2.h"
#include "Image.h"

namespace ReadingPeopleTracker
{

class Calibration
{
private:
    
    unsigned int image_height;
    
    // 3x4 homogenous projection matrix  (image_vector = WorldToImage * world_vector)
    NagMatrix *WorldToImage;
    
    ImageAddressing matrix_image_addressing_mode;
    
public:
    
    Calibration(char *filename,
		unsigned int the_image_height);

    //for all of the STIB and TMB matrixes ImageAddressing = IA_TOP_TO_BOTTOM
    
    void get_matrix_from_file (char *filename);
    
    NagVector get_ip_from_world (const NagVector &worldpos) const;    
    
    // Calibration interfaces: versions using Point2 for image points
    NagVector get_world_from_ip (const Point2 &ip, const realno height) const;
    
    realno get_image_distance_from_width_in_cm (const Point2 &left_ip,
						const realno width_in_cm,
						const realno height) const;
    
    realno get_image_distance_from_height_in_cm (const Point2 &upper_ip,
						 const realno height_in_cm,
						 const realno height) const;
    
    realno get_world_distance_from_height_in_pixels (const Point2 &upper_ip,
						     const realno height_in_pixels,
						     const realno height) const;
    
    realno get_world_distance_from_width_in_pixels (const Point2 &left_ip,
						    const realno width_in_pixels,
						    const realno height) const;
    
    // Calibration interfaces: versions using a homogenous NagVector(3) for image points
    NagVector get_world_from_ip (NagVector ip,
				 const realno height) const;
    
    realno get_image_distance_from_width_in_cm (NagVector left_ip,
						const realno width_in_cm,
						const realno height) const;
    
    realno get_image_distance_from_height_in_cm (NagVector upper_ip,
						 const realno height_in_cm,
						 const realno height) const;
    
    realno get_world_distance_from_height_in_pixels (NagVector upper_ip,
						     const realno height_in_pixels,
						     const realno height) const;
    
    realno get_world_distance_from_width_in_pixels (NagVector left_ip,
						    const realno width_in_pixels,
						    const realno height) const;
    
    realno get_distance_to_camera_from_worldpos(const NagVector &worldpos) const;
    
    NagVector get_camera_position() const;

};

} // namespace ReadingPeopleTracker

#endif
