////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//  Calibration.cc                                                            //
//                                                                            //
//  This class reads a calibration (4x3 projection) matrix from file and      //
//    provides methods to use this calibration data for image measurements    //
//                                                                            //
//  The matrix file must be in NAG format.  Its image addressing mode is      //
//    determined with a simple test.                                          //
//                                                                            //
//  Author    : Rmi Belin (rb)  supervised by  Nils T Siebel (nts)           //
//  Created   : Wed Jul 18 13:50:21 BST 2001                                  //
//  Revision  : 1.0 of Tue Jul 31 12:21:08 BST 2001                           //
//  Copyright : The University of Reading                                     //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <cmath>          // for fmax()
#include <string.h>       // for strcmp()
#include <cassert>

#include <fstream>

#include "Calibration.h"
#include "text_output.h"

namespace ReadingPeopleTracker
{

static const char *Calibration_Revision = "@(#) Calibration.cc, rev 1.0 of Tue Jul 31 12:21:08 BST 2001, Authors Rmi Belin and Nils T Siebel, Copyright (c) 2000--2001 The University of Reading";

Calibration::Calibration(char *filename,
			 unsigned int the_image_height)
{
    assert(the_image_height > 0);
    
    if ((filename == NULL) || (strlen(filename) == 0))
    {
	cerror << "Calibration::Calibration(): Error:"
	       << " No or empty calibration file name given. " << endl;
	exit(1);
    }
    
    WorldToImage = new NagMatrix(3,4);
    get_matrix_from_file(filename);
    image_height = the_image_height;
    
    // determine the matrix's image addressing mode
    
    // first get a world point in the centre of the image
    NagVector image_centre(3);
    
    image_centre[0] = (4/3) * the_image_height / 2;
    image_centre[1] = the_image_height / 2;
    image_centre[2] = 1;
    
    NagVector lower_wp(4);
    
    lower_wp = get_world_from_ip(image_centre, 0);  
    
    // now test whether a point higher up in world is higher up in the image
    NagVector upper_wp(4);
    upper_wp = lower_wp;
    upper_wp[2] += 100;  // FIXME: assuming 100 world units (e.g. cm) is reasonable
    
    NagVector lower_ip(3);
    NagVector upper_ip(3);
    
    // before calling get_ip_from_world, set matrix_image_addressing_mode temporarily
    // in a way that disables the correction of v components in that method
    matrix_image_addressing_mode = Image::image_addressing_mode;
    lower_ip = get_ip_from_world(lower_wp);
    upper_ip = get_ip_from_world(upper_wp);
    
    // simple comparison of v coordinates to see which one is larger
    if (lower_ip[1] > upper_ip[1])
    {
	matrix_image_addressing_mode = IA_TOP_TO_BOTTOM;	
	cdebug << " Calibration::Calibration: detected TOP_TO_BOTTOM calibration matrix. "
	       << endl;
    }
    else
    {
	matrix_image_addressing_mode = IA_BOTTOM_TO_TOP;
	cdebug << " Calibration::Calibration: detected BOTTOM_TO_TOP calibration matrix. "
	       << endl;
    }
    
}

void Calibration::get_matrix_from_file(char *filename)
{
    ifstream ifs(filename);
    if (ifs.bad())
    {
	cerror << "Calibration::get_matrix_from_file(): Warning:"
	       << " Cannot open calibration file. " << filename << " " << endl;
	exit(1);
    }
    char marker[256];
    ifs >> marker;
    if (strcmp(marker,"Calibration_Matrix") != 0)
    {
	cerror << "Calibration::get_matrix_from_file(): Warning:"
	       << " Bad calibration file " << filename << " " << endl;
	exit(1);
    }
    ifs >> *WorldToImage;
    return;
}


NagVector 
Calibration::get_ip_from_world(const NagVector &worldpos) const
{
    assert (worldpos.get_size() == 4);              // expect homogeneous vector (x,y,z,1)
    assert (fabs(worldpos.read(3) - 1) < 0.00001);  // expect homogeneous vector (x,y,z,1)
    
    NagVector ip(3);
    WorldToImage->multiply(worldpos,ip);  // ip = WorldToImage * worldpos
    ip[0] = ip[0] / ip[2];                // homogenise the vector
    ip[1] = ip[1] / ip[2];
    ip[2] = 1;

    // adjust image point according to the matrix's image addressing mode.
    if (Image::image_addressing_mode != matrix_image_addressing_mode)
    {
	// correct v term in the image point ip = (u,v,1)
	ip[1] = image_height - (ip[1] + 1);
    }
    
    return ip;
}


//  
// Starting from the equality
//
//    (u,v,1) = WorldToImage * (x,y,z,1)  //  valid up to a scale
//
// where * WorldToImage is the 3x4 projection matrix
//       * (u,v) the image position
//       * (x,y,z) the world position
//
// one can eliminate problems with scaling factors by deriving
//
//    u * ( w20*x + w21*y + w22*z + w23 ) = w00*x + w01*y + w02*z + w03
//         and
//    v * ( w20*x + w21*y + w22*z + w23 ) = w10*x + w11*y + w12*z + w13
//
// with wij = WorldToImage->read(i,j)
//
// These lower two equations are the underlying equations for the following methods
//

NagVector Calibration::get_world_from_ip (NagVector ip, realno height) const
{
    assert (ip.get_size() == 3);              // expect homogeneous vector (u,v,1)
    assert (fabs(ip.read(2) - 1) < 0.00001);  // expect homogeneous vector (u,v,1)
    assert (height >= 0);
    
    // first adjust image point according to the matrix's image addressing mode.
    if (Image::image_addressing_mode != matrix_image_addressing_mode)
    {
	// correct v term in the image point ip = (u,v,1)
	ip[1] = image_height - (ip[1] + 1);
    }
    
    // A is the 2x2 matrix where u and v (known image position), and the z component
    // (known world height) are already inserted.  What remains are the x and y
    // components of the underlying equation above.
    NagMatrix A(2,2);
    
    // first row of A: first equation above.
    A.set(0,0,ip[0]*WorldToImage->read(2,0)-WorldToImage->read(0,0));  // x
    A.set(0,1,ip[0]*WorldToImage->read(2,1)-WorldToImage->read(0,1));  // y
    
    // second row of A: second equation above.
    A.set(1,0,ip[1]*WorldToImage->read(2,0)-WorldToImage->read(1,0));  // x
    A.set(1,1,ip[1]*WorldToImage->read(2,1)-WorldToImage->read(1,1));  // y
    
    // right hand side (RHS) of the above equations: vector B has all the known variables.
    NagMatrix b(2,1);
    
    b.set(0,0,(WorldToImage->read(0,2)-WorldToImage->read(2,2)*ip[0])*height     // z
	  + WorldToImage->read(0,3)                                      // constants
	  - WorldToImage->read(2,3)*ip[0]);                              // constants
    
    b.set(1,0,(WorldToImage->read(1,2)-WorldToImage->read(2,2)*ip[1])*height     // z
	  + WorldToImage->read(1,3)                                      // constants
	  - WorldToImage->read(2,3)*ip[1]);                              // constants
    
    // now solve A * X = b      <----  X = (x,y) 
    
    // X is the result in world coordinates (x,y) with z missing
    NagMatrix X(2,1);
    
    A.solve_equations(b,X);  // this solves A * X = B
    
    // assign result to new `world' vector, inserting the known z (height)
    NagVector world(4);
    world[0] = X.read(0,0);
    world[1] = X.read(1,0);
    world[2] = height;
    world[3] = 1;         //  make homogeneous coordinates
    
    return world;
    
}
realno Calibration::get_image_distance_from_width_in_cm(NagVector left_ip,
							realno width_in_cm,
							realno height) const
{
    assert (left_ip.get_size() == 3);              // expect homogeneous vector (u,v,1)
    assert (fabs(left_ip.read(2) - 1) < 0.00001);  // expect homogeneous vector (u,v,1)
    
    NagVector wp(4);
    wp = get_world_from_ip(left_ip,height);
    // NB the get_world_from_ip method handles image addressing for us
    
    // now determine x, then u coordinate of the right point
    realno right_x = wp[0] + width_in_cm;
    realno right_u = (WorldToImage->read(0,0)*right_x
		      + WorldToImage->read(0,1)*wp[1]
		      + WorldToImage->read(0,2)*wp[2]
		      + WorldToImage->read(0,3))
	/
	(WorldToImage->read(2,0)*right_x
	 + WorldToImage->read(2,1)*wp[1]
	 + WorldToImage->read(2,2)*wp[2]
	 + WorldToImage->read(2,3));
    
    return fabs(left_ip[0]-right_u);
}

realno Calibration::get_image_distance_from_height_in_cm(NagVector upper_ip,
							 realno height_in_cm,
							 realno upper_height) const
{
    assert (upper_ip.get_size() == 3);              // expect homogeneous vector (u,v,1)
    assert (fabs(upper_ip.read(2) - 1) < 0.00001);  // expect homogeneous vector (u,v,1)
    
    NagVector upper_wp(4);
    upper_wp = get_world_from_ip (upper_ip, upper_height);
    // NB the get_world_from_ip method handles image addressing for us
    
    // lower_wp which is simply height_in_cm below upper_wp
    NagVector lower_wp(4);
    lower_wp = upper_wp;
    lower_wp[2] -= height_in_cm;

    // determine lower_ip from lower_wp
    NagVector lower_ip(3);    
    lower_ip = get_ip_from_world(lower_wp);
    // NB the get_ip_from_world method handles image addressing for us

    return fabs (upper_ip[1]-lower_ip[1]);
}

realno Calibration::get_world_distance_from_width_in_pixels(NagVector left_ip,
							    realno width_in_pixels,
							    realno height) const
{
    assert (left_ip.get_size() == 3);              // expect homogeneous vector (u,v,1)
    assert (fabs(left_ip.read(2) - 1) < 0.00001);  // expect homogeneous vector (u,v,1)
    
    NagVector wp(4);
    wp = get_world_from_ip(left_ip,height);

    // now determine u, then x coordinate of the right point
    realno right_u = left_ip[0]+width_in_pixels;    
    realno right_x = ((WorldToImage->read(0,1)-right_u*WorldToImage->read(2,1))*wp[1]
		      + (WorldToImage->read(0,2)-right_u*WorldToImage->read(2,2))*wp[2]
		      + WorldToImage->read(0,3)-right_u*WorldToImage->read(2,3))
	/ (right_u*WorldToImage->read(2,0)-WorldToImage->read(0,0));
    
    return fabs(wp[0]-right_x);
}


realno Calibration::get_world_distance_from_height_in_pixels(NagVector upper_ip,
							     realno height_in_pixels,
							     realno height) const
{
    assert (upper_ip.get_size() == 3);              // expect homogeneous vector (u,v,1)
    assert (fabs(upper_ip.read(2) - 1) < 0.00001);  // expect homogeneous vector (u,v,1)
    assert (height_in_pixels > 0);    
    
    //  1 - get world position of the upper image point
    NagVector wp(4);
    wp = get_world_from_ip(upper_ip,height);
    // NB the get_world_from_ip method handles image addressing for us
    
    // adjust upper_ip according to the matrix's image addressing mode
    if (Image::image_addressing_mode != matrix_image_addressing_mode)
    {
	// correct v term in the upper image point upper_ip
	upper_ip[1] = image_height - (upper_ip[1] + 1);
    }
    
    //  2 - get z component lower_z of the lower image point (with v coordinate
    //      lower_v), assuming that x and y are the same as in upper_wp
    realno lower_v;
    realno lower_z;
    
    // determine the lower point's v according to the matrix's image addressing mode
    if (matrix_image_addressing_mode == IA_TOP_TO_BOTTOM)
	lower_v = upper_ip[1] + height_in_pixels;
    else
	lower_v = upper_ip[1] - height_in_pixels;
    
    // calculate the z coordinate of the lower point
    lower_z = ((WorldToImage->read(1,0)-lower_v*WorldToImage->read(2,0))*wp[0]
	       + (WorldToImage->read(1,1)-lower_v*WorldToImage->read(2,1))*wp[1]
	       + WorldToImage->read(1,3)-lower_v*WorldToImage->read(2,3))
	/ (lower_v*WorldToImage->read(2,2)-WorldToImage->read(1,2));
    
    return fabs(wp[2]-lower_z);
}


NagVector Calibration::get_camera_position() const
{
    NagMatrix A(3,3);
    WorldToImage->get_block(0,2,0,2,A);
    NagMatrix b(3,1);
    WorldToImage->get_block(0,2,3,3,b);
    b[0] = - b[0];
    b[1] = - b[1];
    b[2] = - b[2];
    NagMatrix camera_pos(3,1);
    WorldToImage->solve_equations(b,camera_pos);
    //to get to camera position we look for the null-space of WorldToImage.
    
    NagVector camera_position(4);
    camera_position[0] = camera_pos.read(0,0);
    camera_position[1] = camera_pos.read(1,0);
    camera_position[2] = camera_pos.read(2,0);
    camera_position[3] = 1;
    return camera_position;
    
}
realno Calibration::get_distance_to_camera_from_worldpos(const NagVector &worldpos) const
{
    assert (worldpos.get_size() == 4);              // expect homogeneous vector (x,y,z,1)
    assert (fabs(worldpos.read(3) - 1) < 0.00001);  // expect homogeneous vector (x,y,z,1)
    
    NagVector camera_pos(4);
    camera_pos = get_camera_position();
    NagVector diff(3);
    diff[0] = worldpos[0] - camera_pos[0];
    diff[1] = worldpos[1] - camera_pos[1];
    diff[2] = worldpos[2] - camera_pos[2];
    
    return sqrt(diff[0]*diff[0]+diff[1]*diff[1]+diff[2]*diff[2]);
}


// Calibration interfaces: versions using Point2 for image points -> simple wrappers
NagVector
Calibration::get_world_from_ip (const Point2 &ip, const realno height) const
{
    NagVector nag_ip(3);
    nag_ip[0] = ip.x;
    nag_ip[1] = ip.y;
    nag_ip[2] = 1;       // make homogenous coordinates
    
    return get_world_from_ip(nag_ip, height);    
}

realno Calibration::get_image_distance_from_width_in_cm (const Point2 &left_ip,
							 const realno width_in_cm,
							 const realno height) const
{
    NagVector nag_ip(3);
    nag_ip[0] = left_ip.x;
    nag_ip[1] = left_ip.y;
    nag_ip[2] = 1;       // make homogenous coordinates
    
    return get_image_distance_from_width_in_cm(nag_ip, width_in_cm, height);    
}


realno Calibration::get_image_distance_from_height_in_cm (const Point2 &upper_ip,
							  const realno height_in_cm,
							  const realno height) const
{
    NagVector nag_ip(3);
    nag_ip[0] = upper_ip.x;
    nag_ip[1] = upper_ip.y;
    nag_ip[2] = 1;       // make homogenous coordinates
    
    return get_image_distance_from_height_in_cm(nag_ip, height_in_cm, height);    
}

realno Calibration::get_world_distance_from_height_in_pixels (const Point2 &upper_ip,
							      const realno height_in_pixels,
							      const realno height) const
{
    NagVector nag_ip(3);
    nag_ip[0] = upper_ip.x;
    nag_ip[1] = upper_ip.y;
    nag_ip[2] = 1;       // make homogenous coordinates
    
    return get_world_distance_from_height_in_pixels(nag_ip, height_in_pixels, height);    
}

realno Calibration::get_world_distance_from_width_in_pixels (const Point2 &left_ip,
							     const realno width_in_pixels,
							     const realno height) const
{
    NagVector nag_ip(3);
    nag_ip[0] = left_ip.x;
    nag_ip[1] = left_ip.y;
    nag_ip[2] = 1;       // make homogenous coordinates
    
    return get_world_distance_from_width_in_pixels(nag_ip, width_in_pixels, height);    
}

} // namespace ReadingPeopleTracker
