
// the actual shape fitting is done by an ActiveModel class using an EdgeDetector

#ifndef __ACTIVE_SHAPE_TRACKER_H__
#define __ACTIVE_SHAPE_TRACKER_H__

#include "BaseTracker.h"
#include "tracker_defines_types_and_helpers.h"  // for frame_id_t

namespace ReadingPeopleTracker
{

// forward declarations
class Image;
class ActiveModel;
class Inputs;
class MotionDetector;
class Outputs;
class Calibration;
class EdgeDetector;
class Profile;
class ProfileSet;
class Camera;

class ActiveShapeTracker : public BaseTracker
{
    friend class ActiveModel;  // our friend needs some parameters from us
 
private:
    
    // the actual shape fitting is done by an ActiveModel using an EdgeDetector
    ActiveModel *active_model;
   
    // ActiveModel's MovingEdgeDetector might need this, we will supply it
    Image *previous_video_image;

    ///////////////////////////////////////////////////////////
    //   configuration variables                             //
    ///////////////////////////////////////////////////////////

    // constants for optimisation scheme
    // maximum M-distance to mean in shape space
    realno cloud_size;
    
    // shape model noise parameters
    realno noise_scalar;
    realno initial_noise;
    
    // measurement parameters
    realno measure_fac_sd;
    realno measure_const_sd;

    unsigned int default_subdivisions;
    unsigned int default_model_depth;

    // the no of observations per free parameter
    unsigned int bunch_size;

    // use a hierarchical fitting scheme
    bool use_hierarchical_search;
    // use the mahalonbis search direction
    bool do_m_search;

    // try and fix the maximum gain around this value or above
    realno minimum_best_gain;
    realno FITNESS_THRESHOLD;
    realno VERY_POOR_FIT;
    unsigned int SMALL_NO_FRAMES;
    
    realno MIN_SIGNIFICANCE;
    realno terminate_thresh;


    // edge detector and parameters
    EdgeDetector *edge_detector;
    realno edge_threshold;
    char *edge_detection_method;
    
    unsigned int TEMPORAL_SKIP;

    // uncertainties for Kalman filtering
    // relative to the initial object height
    
    realno init_pos_sd;
    realno init_vel_sd;
    realno noise_vel_sd;
    realno noise_acc_sd;
    
    realno init_scale_sd;       // the std dev of scale estimate
    realno noise_scale_sd;	// the std dev of scale noise term
    realno init_theta_sd;	// the std dev of initial theta estimate
    realno noise_theta_sd;      // the std dev of theta noise term
    
    bool use_calibration;
    bool show_occlusion_images;    

    // whether to remove theoriginal measurement when creating new hypotheses using calibration
    bool new_hypothesis_replaces_measurement;
    
    // for ActiveModel class:
    char *pca_model_filename;


public:
    //new Constructor to accept a camera pointer as input as well as config filename
    //passes camera pointer to BaseTracker   
    ActiveShapeTracker(Inputs *inputs,
		       MotionDetector *motion_detector,
		       char *config_filename);
    
    // use inputs and current/previous results to obtain and store new results
    void process_frame(Inputs *inputs,
		       Results *results,
		       unsigned int max_objects = 32); 
    
    // post processing: clean up tracks etc in results
    void post_process_frame(Inputs *inputs,
			    Results *results);

    inline frame_id_t get_small_number_frames() const
	{
	    return SMALL_NO_FRAMES;
	}
    
    // this will be used by EdgeDetector classes to access current images
    inline ActiveModel *get_active_model() const
	{
	    return active_model;
	}
    
private:    

    // re-defined virtual methods, from ShapeTracker
    void predict_old_objects(Results *previous_results);
    void track_old_objects(Inputs *inputs, Results *results);
    void detect_new_objects(Inputs *inputs, Results *results);

    // other private methods
    void initialise_track(Profile *curr_prf);
    void create_hypotheses_using_calibration(Profile *profile, ProfileSet *results);
    void compare_two_and_disable_bad_profile(Profile *prf1, Profile *prf2);
    void register_configuration_parameters();

    EdgeDetector *setup_edge_detector(Inputs *inputs,
				      MotionDetector *motion_detector,
				      char *detection_method);
    
};

} // namespace ReadingPeopleTracker

#endif
