/***************************************************************
 * C++ source
 *
 * File : 	ActiveShapeTracker.cc
 *
 * Module :	ActiveShapeTracker
 *
 * Author : 	A M Baumberg (CoMIR)
 *
 * Creation Date : Tue Oct 22 15:06:09 1996 
 *
 * Comments : 	High level people tracker
 *
 ***************************************************************/


// includes

#include <cassert>

#ifndef NO_DISPLAY
#ifdef DEBUG
#include "os_specific_things.h"  // for sleep()
#endif
#endif

#include "ActiveShapeTracker.h"
#include "TrackedObjectSet.h"
#include "EdgeDetector.h"
#include "Image.h"
#include "Calibration.h"
#include "Inputs.h"
#include "ScreenOutput.h"
#include "text_output.h" // for cerror etc
#include "ActiveModel.h"
#include "Results.h"
#include "MotionDetector.h"
#include "text_output.h"
#include "PeopleTracker.h"
#include "OcclusionImage.h"

namespace ReadingPeopleTracker
{

// BaseTracker::register_base_configuration_variables() registeres some for us!
void ActiveShapeTracker::register_configuration_parameters()
{
    cloud_size = 
	configuration.register_real("PCA_CLOUD_SIZE", 2.0,
				    &cloud_size,
				    false, "ActiveModel", 
				    "The maximum Mahalanobis distance to the \
mean shape (in std deviations)");
    
    noise_scalar = 
	configuration.register_real("SHAPE_NOISE_SD", sqrt(0.2),
				    &noise_scalar,
				    false, "ActiveModel",
				    "The noise term for each shape parameter \
in std deviations per frame");

    initial_noise =
	configuration.register_real("INIT_SHAPE_SD", 1.0,
				    &initial_noise,
				    false, "ActiveModel",
				    "The initial uncertainty of each shape \
parameter in std deviations");
    
    measure_fac_sd =
	configuration.register_real("MEASURE_FAC_SD", 0.1,
				    &measure_fac_sd,
				    false, "ActiveModel",
				    "The standard deviation of measurements \
proportional to the search window size");
    
    measure_const_sd =
	configuration.register_real("MEASURE_CONST_SD", 0.5,
				    &measure_const_sd,
				    false, "ActiveModel",
				    "An additional constant value added to the \
std deviation of the measurement process (in pixels)");
    
    default_subdivisions =
	configuration.register_int("SPLINE_SUBDIVISIONS", 8,
				   &default_subdivisions,
				   true, "ActiveModel",
				   "The number of sample points between spline \
control points");
    
    default_model_depth = 
	configuration.register_int("MODEL_DEPTH", 10,
				   &default_model_depth,
				   true, "ActiveModel",
				   "The number of shape parameters used");
    
    bunch_size =
	configuration.register_int("BUNCH_SIZE", 2,
				   &bunch_size,
				   false, "ActiveModel",
				   "The number of measurement made \
per free parameter");
    
    
    use_hierarchical_search = 
	configuration.register_bool("HIERARCHICAL", true,
				    &use_hierarchical_search,
				    false, "ActiveModel",
				    "Vary the number of shape paramaters and \
measurements as the fitting process converges");
    
    do_m_search =
	configuration.register_bool("MAHALANOBIS_DIRECTION", true,
				    &do_m_search,
				    false, "ActiveModel",
				    "Search along the Mahalanobis optimal search \
direction");
    
    minimum_best_gain = 
	configuration.register_real("MIN_GAIN", 0.5,
				    0.0, 1.0,
				    &minimum_best_gain,
				    false, "ActiveModel",
				    "The desired initial gain and the minimum \
subsequent gain for parameter fitting");
    
//////////////////////////////////////////////////
    
    FITNESS_THRESHOLD  =
	configuration.register_real("START_FITNESS_THRESHOLD", 0.85,
				    0.0, 1.0,
				    &FITNESS_THRESHOLD,
				    false, "ActiveShapeTracker",
				    "Minimum fitness score to initialise track");
    
    VERY_POOR_FIT =
	configuration.register_real("TRACK_FITNESS_THRESHOLD", 0.4,
				    0.0, 1.0,
				    &VERY_POOR_FIT,
				    false, "ActiveShapeTracker",
				    "Minimum fitness score to maintain track");
    
    SMALL_NO_FRAMES =
	configuration.register_int("SMALL_NO_FRAMES", 3,
				   &SMALL_NO_FRAMES,
				   false, "ActiveShapeTracker",
				   "Number of frames before track is accepted");
    
    MIN_SIGNIFICANCE =
	configuration.register_real("MIN_SIGNIFICANCE", 0.3,
				    0.0, 1.0,
				    &MIN_SIGNIFICANCE,
				    false, "ActiveShapeTracker",
				    "Fraction of curve that must be visible for tracking");
    
    terminate_thresh =
	configuration.register_real("TERMINATION_THRESHOLD", 100,
				    &terminate_thresh,
				    false, "ActiveShapeTracker",
				    "Maximum SD of ground plane position (in cm)");
    
    init_scale_sd =
	configuration.register_real("INIT_SCALE_SD", 0.015,
				    &init_scale_sd,
				    false, "ActiveShapeTracker",
				    "SD of initial estimate of object size \
relative to initial estimate");
    
    noise_scale_sd =
	configuration.register_real("NOISE_SCALE_SD", 0.05,
				    &noise_scale_sd,
				    false, "ActiveShapeTracker",
				    "SD of scale noise term relative to initial scale");
    
    TEMPORAL_SKIP =
	configuration.register_int("FRAME_SKIP", 1,
				   &TEMPORAL_SKIP,
				   true, "ActiveShapeTracker",
				   "Temporal resampling for object detection (1==no skip)");
    
    edge_threshold =
	configuration.register_real("EDGE_THRESHOLD", 0.05,
				    0.0, 1.0, &edge_threshold,
				    false, "ActiveShapeTracker",
				    "A significance threshold for edge detection");
    
    edge_detection_method =
	configuration.register_string("EDGE_DETECTION", "COLOUR_FOREGROUND_EDGE",
				      &edge_detection_method, true,
				      "ActiveShapeTracker",
				      "The type of edges used for tracking",
				      EdgeDetector::edge_detection_method_names);
    
    init_pos_sd =
	configuration.register_real("INIT_POS_SD", 0.1,
				    &init_pos_sd,
				    false, "ActiveShapeTracker",
				    "Initial standard deviation of position\
 relative to object height");
    
    init_vel_sd =
	configuration.register_real("INIT_VEL_SD", 0.1,
				    &init_vel_sd,
				    false, "ActiveShapeTracker",
				    "Initial standard deviation of velocity\
 relative to object height");
    
    noise_vel_sd =
	configuration.register_real("REL_VEL_SD", 0.1,
				    &noise_vel_sd,
				    false, "ActiveShapeTracker",
				    "standard deviation of random velocity\
 term relative to object height");

    noise_acc_sd =
	configuration.register_real("REL_ACC_SD", 0.1,
				    &noise_acc_sd,
				    false, "ActiveShapeTracker",
				    "standard deviation of random acceleration\
 term relative to object height");    
    
    init_theta_sd =
	configuration.register_real("INIT_THETA_SD", 0.015,
				    &init_theta_sd,
				    false, "ActiveShapeTracker",
				    "Standard deviation of initial estimate of\
 rotation angle");
    
    noise_theta_sd =
	configuration.register_real("NOISE_THETA_SD", 0.0,
				    &noise_theta_sd,
				    false, "ActiveShapeTracker",
				    "Standard deviation of theta noise term");    
    
    pca_model_filename =
	configuration.register_string("PCA_MODEL_FILENAME",
				      NULL, &pca_model_filename, true,
				      "ActiveShapeTracker",
				      "An optional eigenshape model file.  If \
not set, the default model is used.");
    
    new_hypothesis_replaces_measurement
	= configuration.register_bool("NEW_HYPOTHESIS_REPLACES_MEASUREMENT", true,
				    &new_hypothesis_replaces_measurement,
				    false, "ActiveShapeTracker",
				    "whether to remove the original measurement when creating new hypotheses using calibration");
   
}//end of register_configuration_parameters()


ActiveShapeTracker::ActiveShapeTracker(Inputs *inputs,
				       MotionDetector *motion_detector,
				       char *config_filename)
{
    calibration = inputs->get_calibration();
    
    if (calibration != NULL)
	use_calibration = true;
    else
	use_calibration = false;
    
    if (config_filename == NULL)
    {
	cerror << "ActiveShapeTracker: cannot instantiate without configuration file "
	       << endl;
	exit(1);
    }
    
    //member function registers appropriate configuration parameters with configuration manager
    register_configuration_parameters();
    
    //read from configuration file storing vales in correct place using configuration manager
    configuration.parse_parameter_file(config_filename);
    
    previous_video_image = NULL;

    // set up active model (before setting up the edge detector)
    active_model = new ActiveModel(this, inputs->get_video_image_source()->get_image_type());

    // set up edge detector using the ShapeTracker configuration var edge_detection_method
    edge_detector = setup_edge_detector(inputs, motion_detector, edge_detection_method);
    
    active_model->set_edge_detector(edge_detector);
    
    cdebug << " >>>  ActiveShapeTracker::ActiveShapeTracker: active_model->set_video_image(inputs->get_video_image());  <<< " << endl;
    
    active_model->set_occlusion_handler (
	new OcclusionImage((Grey8Image *)motion_detector->get_inverted_additional_motion_mask()));
    
    cdebug << " >>>  ActiveShapeTracker::ActiveShapeTracker: active_model->set_occlusion_handler(tracking->get_occlusion_image()); changed to new OcclusionImage(motion_detector->get_inverted_additional_motion_mask()) <<< "
	   << endl;
}


void ActiveShapeTracker::predict_old_objects(Results *results)
{
    // update Kalman predictions (position) for tracked `Profile's.
    // new: we now also get new `Profile's which are not tracked yet (ie no filters)
    //      so there cannot be predictions.  These will just be ignored.
    Profile *curr_prf;
    
//     frame_id_t frame_id = results->get_motion_image()->get_frame_id();
    
    ListNode<TrackedObject> *curr_obj;
    
    TrackedObjectSet *objects = results->get_tracked_objects();
    
    // get old profiles from tracked object set, predicting each
    for (curr_obj = objects->first; curr_obj != NULL; curr_obj = curr_obj->next)	
    {
	// get profile(s) for this tracked object (usually only one)
	ProfileSet *profiles = curr_obj->dat->profiles;
	
	for (profiles->start(); profiles->current_ok(); profiles->forward())
	{
	    // predict profile
	    curr_prf = profiles->get_current();
	    
	    // first check whether the surrounding region is incorporated into the background
	    if ((curr_obj->dat->regions->no_items > 0) &&
		(curr_obj->dat->regions->first->dat->incorporated_into_background))
	    {
		// no prediction, just change status...
		curr_prf->source = PREDICTION;
		curr_prf->is_visible = true;
		// curr_prf->frame_last_detected = frame_id - 1;
		continue;
	    }
	    
	    if (curr_prf->filters_initialised == false)
	    {
		// FIXME: we should not be here.  we are here because of the problem in Profile::operator=, quod vide.
		cdebug << "ActiveShapeTracker::predict_old_objects(): "
		       << " object id " << curr_obj->dat->id
		       << " Warning: curr->dat->filters_initialised == false"
		       << endl;
		
		initialise_track(curr_prf);
		
//  		continue;  // ignore uninitialised Profile (from splitting?)
//		// the Profile will be initialised/tracked in detect_new_objects()
	    }
	    
	    active_model->set_profile(curr_prf);
	    
	    assert(curr_prf->pos_filter != NULL);  // cannot predict uninitialised Profile
	    
	    curr_prf->update_shape_filters();
	    active_model->apply_virtual_shape();
	    
	    DynamicKalmanTwoD *origin_filter =
		(DynamicKalmanTwoD*) curr_prf->pos_filter;
	    
	    origin_filter->
		reset_noise_covariance(SQUARE(noise_vel_sd * curr_prf->height),
				       SQUARE(noise_vel_sd * curr_prf->height),
				       SQUARE(noise_acc_sd * curr_prf->height),
				       SQUARE(noise_acc_sd * curr_prf->height));
	    
	    Point2 origin = origin_filter->get_prediction();
	    curr_prf->origin = origin;
	    
	    curr_prf->a_filter->set_polar_noise_cov(SQUARE(noise_scale_sd),
						    SQUARE(noise_theta_sd));
	    curr_prf->a_filter->update_state();
	    
	    // mark profile as predicted
	    curr_prf->source = PREDICTION;
	    
	    active_model->b_to_x();
	}
    }
}


void ActiveShapeTracker::track_old_objects(Inputs *inputs, Results *results)
{
    // new: we now also get new `Profile's which are not tracked yet
    //      so they do not have valid shape nor filters
    
#ifdef DEBUG
    inputs->get_video_image()->draw_in_image();
#endif
    
    // get frame id and time from  video image
    frame_id_t frame_id = inputs->get_frame_id();
    frame_time_t frame_time = inputs->get_frame_time_in_ms();
    
    ListNode<TrackedObject> *curr_obj;
    
    TrackedObjectSet *objects = results->get_tracked_objects();
    
    // get old, predicted profiles from tracked object set, tracking each
    for (curr_obj = objects->first; curr_obj != NULL; curr_obj = curr_obj->next)	
    {
	// get profile(s) for this tracked object (usually only one)
	ProfileSet *profiles = curr_obj->dat->profiles;
	
	for (ListNode<Profile> *curr = profiles->first; curr != NULL; )
	{
	    // first check whether the surrounding region is incorporated into the background
	    if ((curr_obj->dat->regions->no_items > 0) &&
		(curr_obj->dat->regions->first->dat->incorporated_into_background))
	    {
		// ignore profile (no tracking)
		curr = curr->next;
		continue;
	    }
	    
	    if (curr->dat->filters_initialised == false)
	    {
		cdebug << "ActiveShapeTracker::track_old_objects(): "
		       << " object id " << curr_obj->dat->id
		       << " Warning: curr->dat->filters_initialised == false"
		       << endl;
		
		initialise_track(curr->dat);
		// the Profile will be tracked in detect_new_objects()
	    }
	    
	    cdebug << "ActiveShapeTracker::track_old_objects(): Tracking profile in"
		   << " object id " << curr_obj->dat->id << endl;
	    
	    assert (curr->dat->source == PREDICTION);
	    
	    Profile *curr_prf = curr->dat;
	    
	    // track profile using our ActiveModel
	    active_model->set_profile(curr_prf);
	    
	    assert(curr_prf->pos_filter != NULL);  // cannot predict uninitialised Profile
	    
	    curr_prf->fitness = active_model->track_profile(curr_obj->dat->id);
	    
	    curr_prf->track_error =
		sqrt(curr_prf->pos_filter->get_uncertainty());
	    
	    // update variables like xlo, xhi, width etc
	    curr_prf->update_size_variables();
	    
	    cdebug << "AST: positional variance " << curr_prf->track_error << endl
		   << "AST: observed " << active_model->no_observed
		   << " out of unoccluded " << active_model->no_looks
		   << " out of " << active_model->no_sampled
		   << endl	    
		   << "AST: fitness " << curr_prf->fitness << endl;
	    
	    // check whether track is ok (fit enough to keep)
	    bool is_ok = true;
	    
	    // if track is fairly new and the fit is poor then kill it
	    if ((frame_id - curr_prf->frame_first_detected < SMALL_NO_FRAMES) && 
		(curr_prf->fitness < FITNESS_THRESHOLD))
		is_ok = false;
	    
	    // if the positional uncertainty has grown too large, kill it
	    if (curr_prf->track_error > terminate_thresh)
		is_ok = false;
	    
	    // if the fit is really bad and a significant part of the
	    // contour is observable (unoccluded), kill it
	    if ((curr_prf->fitness < VERY_POOR_FIT) &&
		(active_model->get_significance() > MIN_SIGNIFICANCE)) 
		is_ok = false;
	    
	    if (is_ok == false)
	    {
		if (frame_id - curr_prf->frame_first_detected < SMALL_NO_FRAMES)
		{
		    cdebug << "AST: Killing object " << curr_obj->dat->id
			   << " (alive " << frame_id - curr_prf->frame_first_detected << ") :"
			   << " track error = " << curr_prf->track_error
			   << ", fitness = " << curr_prf->fitness
			   << endl
			   << "AST: observed " << active_model->no_observed
			   << " out of " << active_model->no_looks
			   << endl;

		    curr = curr->next;
		    profiles->destroy(curr_prf);
		}
		else
		{
		    cdebug << "AST: Disabling object " <<  curr_obj->dat->id
			   << " (alive " << frame_id - curr_prf->frame_first_detected << ") :"
			   << " track error = " << curr_prf->track_error
			   << ", fitness = " << curr_prf->fitness
			   << endl
			   << "AST: observed " << active_model->no_observed
			   << " out of " << active_model->no_looks
			   << endl;
		    
		    curr_prf->is_visible = false;
		    curr_prf->source = PREDICTION;  // stay at predicted

		    curr = curr->next;
		}
	    }
	    else
	    {
		// track is ok and visible now
		
		// store tracking status in profile
		curr_prf->source = MEASUREMENT;
		curr_prf->is_visible = true;
		curr_prf->frame_last_detected = frame_id;
		curr_prf->time_last_detected = frame_time;
		
  		// this means that the corresponding TrackedObject is also visible
  		curr_obj->dat->is_visible = true;
		
		// increment
		curr = curr->next;
	    }
	}
    }
}

void ActiveShapeTracker::detect_new_objects(Inputs *inputs, Results *results)
{
    TrackedObjectSet *objects = results->get_tracked_objects();
    
    if (objects->no_items >= max_objects)
    {
	cdebug << " ActiveShapeTracker::detect_new_objects: "
	       << " no more detection because we already have max_objects == "
	       << max_objects;
 	return;
    }
    
#ifdef DEBUG
    inputs->get_video_image()->draw_in_image();      ////////////
#endif
    
    // get regions from tracked object set, applying a shape to and tracking each
    for (objects->start(); objects->current_ok(); objects->forward())
    {
	RegionSet *regions = objects->get_current()->regions;
	ProfileSet *new_profiles;
	
	// first check whether the surrounding region is incorporated into the background
	if ((regions->no_items > 0) && (regions->first->dat->incorporated_into_background))
	    continue;  // ignore!
	
	//
	//  1 - get regions and convert them into profiles using RegionSet::to_profiles()
	//
	
	// if there are new regions convert them to profiles...
	if (regions->no_items > 0)
	{
	    // convert regions to profile by tracing outline
	    new_profiles = regions->to_profiles();
	    
// FIXME: not available    
//  	    // scale coordinates if motion image is subsampled
//  	    new_profiles->scale_coords (tracking->get_motion_detector()->sample_skip);
//	    
//	    // filter all profiles for this object according to height/width ratio
//	    height_width_ratio_filter_profiles(new_profiles);
	    
//	    assert(curr_obj->dat->profiles->no_items == 0);
	    
	    // add observations temporarily to tracked object (removed later if not ok)
	    objects->get_current()->profiles->concat(new_profiles);

	    delete /* the now emptied */ new_profiles;
	}
	
	//
	//  2 - get heads from human features and initialise profiles from them
	//
	
	HumanFeatureSet *features = objects->get_current()->features;
	
	if (features->no_items > 0)
	{
	    new_profiles = new ProfileSet;
	    
	    for (features->start(); features->current_ok(); features->forward())
	    {
		HeadInfo *head_info = features->get_current()->head;
		
		realno height;  // new Profile's height
		
		if (use_calibration)
		{
		    NagVector nag_origin(3);
		    
		    // centre of the head in the image, NAG vector format, homogeneous
		    nag_origin[0] = head_info->x_abs;
		    nag_origin[1] = head_info->y_abs;
		    nag_origin[2] = 1;
		    
		    height = calibration->get_image_distance_from_height_in_cm
			(nag_origin, TYPICAL_WORLD_HEIGHT, TYPICAL_WORLD_HEIGHT);
		}
		else
		{
		    // get the new profile's height from the first region  // FIXME: why first???
		    assert (objects->get_current()->regions->no_items == 1);
		    
		    height = objects->get_current()->regions->first->dat->height;
		}
		
		// create new profile from head info
		Profile *new_profile;
		
		// check ImageAdressing mode before adding up coordinates...
		if (Image::image_addressing_mode == IA_BOTTOM_TO_TOP)
		    new_profile =
			new Profile(head_info->x_abs,
				    head_info->y_abs + head_info->y - height/2,
				    head_info->width,  // width does not matter here
				    height);
		else
		    new_profile =
			new Profile(head_info->x_abs,
				    head_info->y_abs - head_info->y + height/2,
				    head_info->width,  // width does not matter here
				    height);
		
		cdebug << "ActiveShapeTracker::detect_new_objects() : new profile from head! "
		       << endl;
		
		// adjust status
		new_profile->source = features->get_current()->source;  // ie MEASUREMENT
		
		// if we have Calibration the size will be corrected further down
		new_profiles->add(new_profile);
	    }
	    
	    objects->get_current()->profiles->concat(new_profiles);	    
	    delete /* the now emptied */ new_profiles;
	}
	
	ProfileSet *profiles = objects->get_current()->profiles;
	signed int index;  // index into profiles
	
	//
	//  3 - use calibration to create new hypotheses for each profile
	//
	if (use_calibration)
	{
	    int no_items = profiles->no_items;  // to avoid recursion when appending
	    
	    for (index = 0; index < no_items; index++)
	    {
		Profile *curr_prf = (*profiles)[index];
		
		// ignore old objects (already tracked in track_old_objects())
		if (curr_prf->filters_initialised)
		    continue;
		
		// non-measurements should not be here
		assert (curr_prf->source == MEASUREMENT);
		
		create_hypotheses_using_calibration(curr_prf, profiles);
	    }
	}
	
	//
	//  4 - track these new and any other uninitialised profiles for this object
	//
	for (index = 0; index < profiles->no_items; index++)
	{
	    Profile *curr_prf = (*profiles)[index];
	    
	    // ignore old objects (already tracked in track_old_objects())
	    if (curr_prf->filters_initialised)
		continue;
	    
	    // non-measurements should be ignored in regions->to_profiles(), 
	    assert (curr_prf->source == MEASUREMENT);
	    
	    // initialise filters and get model hypothesis from our ActiveModel
	    initialise_track(curr_prf);
	    
	    // convert coordinates
	    active_model->b_to_x();
	    
	    // track profile using our ActiveModel
	    curr_prf->fitness = active_model->track_profile(objects->get_current()->id);
	    
	    curr_prf->track_error = 
		sqrt(curr_prf->pos_filter->get_uncertainty());
	    
	    // update variables like xlo, xhi, width etc
	    curr_prf->update_size_variables();
	    
	    if ((curr_prf->track_error < terminate_thresh) && 
		(curr_prf->fitness > FITNESS_THRESHOLD))
	    {
		// accept track
		
		// store tracking status in profile
		curr_prf->source = MEASUREMENT;
		curr_prf->is_visible = true;
		curr_prf->frame_first_detected =
		    curr_prf->frame_last_detected =
		    inputs->get_frame_id();
		curr_prf->time_first_detected =
		    curr_prf->time_last_detected =
		    inputs->get_frame_time_in_ms();
		
		// remember that the corresponding TrackedObject is thereby visible
		objects->get_current()->is_visible = true;
	    }
	    else 
	    {
		// reject track
		profiles->destroy(index);
		index--;  // no increment necessary to get to next profile
	    }
	}
    }
}

// clean up old tracks from multiple hypotheses of the same tracked person etc
void ActiveShapeTracker::post_process_frame(Inputs *inputs, Results *results)
{
    // check all tracked objects for multiple occurances of tracked profiles.
    //   if there is more than one and they resemble the same person then remove
    //   all but the best track.
    // FIXME: ...and set all tracker pointers to sensible values
    
    TrackedObjectSet *objects = results->get_tracked_objects();
    ProfileSet *profiles;              // profiles for one object to be checked
    signed int index1, index2;         // indices into profiles or objects
    Profile *prf1, *prf2;              // the actual profiles indexed
    TrackedObject *obj1, *obj2;        // the actual objects indexed
    ListNode<TrackedObject> *curr_obj; // tracked object item to be checked    
    
    //
    //  1 - clean up profiles within each tracked object
    //
    
    // get profiles from tracked object set and keep only the best
    for (curr_obj = objects->first; curr_obj != NULL; curr_obj = curr_obj->next)	
    {
	// get profile(s) for this tracked object (might well be more than one)
	profiles = curr_obj->dat->profiles;
	
	// ignore object unless there are at least two profiles here
	if (profiles->no_items < 2)
	    continue;
	
 	// ignore background objects here
 	if ((curr_obj->dat->regions->no_items > 0) &&
 	    (curr_obj->dat->regions->first->dat->incorporated_into_background))
 	    continue;
	
	// check each combination of profiles doing pair-wise comparisons
	for (index1 = 0; index1 + 1 < profiles->no_items; index1++)
	{
	    prf1 = (*profiles)[index1];
	    
	    // ignore disabled profiles
	    if (prf1->fitness <= 0)
		continue;
	    
	    for (index2 = index1 + 1; index2 < profiles->no_items; index2++)
	    {
		prf2 = (*profiles)[index2];
		
		// ignore disabled profiles
		if (prf2->fitness <= 0)
		    continue;
		
		// do the actual comparison: if the two profiles match disable
		// the worse fitting profile by setting its fitness to 0
		compare_two_and_disable_bad_profile(prf1, prf2);
	    }
	}
    }
    
    // after marking profiles for deletion destroy all these and other bad profiles
    for (index1 = 0; index1 < objects->no_items; index1++)  // look in all objects:
    {
	obj1 = (*objects)[index1];
	
	for (index2 = 0; index2 < obj1->profiles->no_items; )  // all profiles in object:
	{
	    prf1 = (*obj1->profiles)[index2];
	    
	    if (prf1->fitness <= 0)
	    {
		obj1->profiles->destroy(prf1);
		
		cdebug << "ActiveShapeTracker::post_process_frame() 1: "
		       << "cleaned away bad/duplicate profile within object "
		       << obj1->id << " because its fitness was " << prf1->fitness << " "
		       << endl;

		// NB: The TrackedObject will be cleaned away elsewhere if it is empty now
	    }
	    else
		index2++;
	}
    }
    
    //
    //  2 - split objects containing more than one (distinct after the above) person
    //
    
    // check each object for the number of profiles
    for (index1 = 0; index1 + 1 < objects->no_items; index1++)
    {
	obj1 = (*objects)[index1];
	
	// only check objects which have 1 region and at least 2 profiles
	if (obj1->regions->no_items != 1)
	    continue;
	
	if (obj1->profiles->no_items < 2)
	    continue;
	
	if (obj1->profiles->no_items > 2)
	    cdebug << " ActiveShapeTracker::post_process_frame() 2: Warning: "
		   << obj1->profiles->no_items << " profiles (can only handle 2) "
		   << endl;
	
//	assert (obj1->profiles->no_items == 2);   // cannot handle more for now // FIXME: use a while loop
	
	if (obj1->profiles->first->dat->source != MEASUREMENT)  // FIXME: do not check 1st only
	    continue;
	
	// do not work on background objects
	if (obj1->regions->first->dat->incorporated_into_background)
	    continue;
	
	// split object into two by copying data over
	TrackedObject *new_obj = new TrackedObject;
	*new_obj = *obj1;
	
	// disable one profile in one object, the other one in the other
	obj1->profiles->destroy(obj1->profiles->last->dat);
	new_obj->profiles->destroy(new_obj->profiles->first->dat);
	
  	// get the two remaining (distinct) profiles
  	prf1 = obj1->profiles->first->dat;
  	prf2 = new_obj->profiles->first->dat;
	
	// adjust associated regions
	Region *reg1 = obj1->regions->first->dat;
	Region *reg2 = new_obj->regions->first->dat;
	
	// now correct position, size and status of the regions matched to the profiles
	reg1->Observation::operator=(*prf1);
	reg1->old_origin = reg1->origin;
	reg1->direction.x = reg1->direction.y = 0;
	
	reg2->Observation::operator=(*prf2);
	reg2->old_origin = reg2->origin;
	reg2->direction.x = reg2->direction.y = 0;
	
	new_obj->frames_static = obj1->frames_static;
	
	// create a new id for the newly created object and add it to the tracked objects
	new_obj->id = PeopleTracker::create_new_id();
	objects->add(new_obj);
	
	cdebug << "ActiveShapeTracker::post_process_frame() 2: split object "
	       << obj1->id << " into two objects, new object id " << new_obj->id << " " << endl;
    }
    
    //
    //  3 - clean up tracked objects if they correspond to the same tracked person
    //
    
    // check each combination of profiles doing pairwise comparisons
    for (index1 = 0; index1 + 1 < objects->no_items; index1++)
    {
	obj1 = (*objects)[index1];
	
	// only check objects which have a Profile
	if (obj1->profiles->no_items == 0)
	    continue;
	
// FIXME: put an if	assert (obj1->profiles->no_items == 1);   // should be ok after the above clean-up
	
	// ignore background objects here
	if ((obj1->regions->no_items > 0) &&
	    (obj1->regions->first->dat->incorporated_into_background))
	    continue;
	
	prf1 = obj1->profiles->first->dat;
	
	for (index2 = index1 + 1; index2 < objects->no_items; index2++)
	{
	    obj2 = (*objects)[index2];
	    
	    // only check objects which have a Profile
	    if (obj2->profiles->no_items == 0)
		continue;
	    
// FIXME: put an if	    assert (obj2->profiles->no_items == 1);   // should be ok after the above clean-up
	    
	    // ignore background objects here
	    if ((obj2->regions->no_items > 0) &&
		(obj2->regions->first->dat->incorporated_into_background))
		continue;
	    
	    prf2 = obj2->profiles->first->dat;
	    
	    // do the actual comparison: if the match, disable the worse fit
	    compare_two_and_disable_bad_profile(prf1, prf2);
	    
	    // move over tracking data to older track from disabled profile
	    // NB assumes lower id for older objects (should be ok)
	    if ((prf1->fitness <= 0) && (obj1->id < obj2->id))
	    {
		Profile *tmp_prf = prf1;
		prf1 = prf2;
		prf2 = tmp_prf;
	    }
	    else 
		if ((prf2->fitness <= 0) && (obj2->id < obj1->id))
		{
		    Profile *tmp_prf = prf1;
		    prf1 = prf2;
		    prf2 = tmp_prf;
		}
	}
    }
    
    // after marking profiles for deletion destroy all these and other bad profiles
    for (index1 = 0; index1 < objects->no_items; index1++)  // look in all objects:
    {
	obj1 = (*objects)[index1];
	
	for (index2 = 0; index2 < obj1->profiles->no_items; )  // all profiles in object:
	{
	    prf2 = (*obj1->profiles)[index2];
	    
	    if (prf2->fitness <= 0)
	    {
		obj1->profiles->destroy(prf2);
		cdebug << "ActiveShapeTracker::post_process_frame() 3: cleaned away bad profile in object "
		       << obj1->id << " " << endl;
		
		// NB: The TrackedObject will be cleaned away elsewhere if it is empty now
	    }
	    else
		index2++;
	}
    }
}

void ActiveShapeTracker::compare_two_and_disable_bad_profile(Profile *prf1, Profile *prf2)
{    
//      const realno ox1 = (prf1->xlo + prf1->xhi) /2;
//      const realno oy1 = (prf1->ylo + prf1->yhi) /2;
//      const realno w1 =  prf1->width;
//      const realno h1 =  prf1->height;
//
//      const realno ox2 = (prf2->xlo + prf2->xhi) /2;
//      const realno oy2 = (prf2->ylo + prf2->yhi) /2;
//      const realno max_delta_x = fmax(w1,prf2->width) / 3;  // max deviation in x
//      const realno max_delta_y = fmax(h1,prf2->height) / 3; // max deviation in y
//    
//      // first, check whether prf1 and prf2 are at the same position in the image
//      if (! ((ox2 > ox1 - max_delta_x) && (ox2 < ox1 + max_delta_x) &&
//  	   (oy2 > oy1 - max_delta_y) && (oy2 < oy1 + max_delta_y)))
//  	return;
    
    const realno ox1 = prf1->origin.x;
    const realno oy1 = prf1->origin.y;
    const realno ox2 = prf2->origin.x;
    const realno oy2 = prf2->origin.y;
    
    // TODO:  CHECK CHECKS
    
    // first, check whether prf1 and prf2 are at the same position in the image
    if (! (((ox2 > prf1->xlo) && (ox2 < prf1->xhi) &&
	    (oy2 > prf1->ylo) && (oy2 < prf1->yhi)) ||
	   ((ox1 > prf2->xlo) && (ox1 < prf2->xhi) &&
	    (oy1 > prf2->ylo) && (oy1 < prf2->yhi))))
  	return;
    
    // TODO:  CHECK DIRECTION HERE (crossing people!) !!!!
   
    // NB assuming that size is already checked (people at the same position
    //     in to image have around the same pixel size so if they had not one
    //     of them had a bad fit and would be removed)
    
    if (prf2->fitness + 0.05 < prf1->fitness)  // prf1 clearly fits better
    {
	prf2->fitness = 0;  // mark for deletion

	prf1->frame_first_detected = min(prf1->frame_first_detected,
					 prf2->frame_first_detected);
	prf1->time_first_detected = min(prf1->time_first_detected,
					prf2->time_first_detected);
	return;
    }
    
    if (prf1->fitness + 0.05 < prf2->fitness)  // prf2 clearly fits better
    {
	prf1->fitness = 0;  // mark for deletion

	prf2->frame_first_detected = min(prf1->frame_first_detected,
					 prf2->frame_first_detected);
	prf2->time_first_detected = min(prf1->time_first_detected,
					prf2->time_first_detected);
	return;
    }
    
    // shapes are similarly good: keep the older one
    if (prf2->frame_first_detected + 3 < prf1->frame_first_detected)  // prf1 is much older
    {
	prf2->fitness = 0;  // mark for deletion

	prf1->frame_first_detected = min(prf1->frame_first_detected,
					 prf2->frame_first_detected);
	prf1->time_first_detected = min(prf1->time_first_detected,
					prf2->time_first_detected);
	return;
    }
    
    if (prf1->frame_first_detected + 3 < prf2->frame_first_detected)  // prf2 is much older
    {
	prf1->fitness = 0;  // mark for deletion

	prf2->frame_first_detected = min(prf1->frame_first_detected,
					 prf2->frame_first_detected);
	prf2->time_first_detected = min(prf1->time_first_detected,
					prf2->time_first_detected);
	return;
    }
    
    // shapes are similarly good and old : keep the one with the lowest positional variance
    if (prf1->track_error > prf2->track_error)
    {
	prf1->fitness = 0;  // mark for deletion

	prf2->frame_first_detected = min(prf1->frame_first_detected,
					 prf2->frame_first_detected);
	prf2->time_first_detected = min(prf1->time_first_detected,
					prf2->time_first_detected);
    }
    else
    {
	prf2->fitness = 0;  // mark for deletion

	prf1->frame_first_detected = min(prf1->frame_first_detected,
					 prf2->frame_first_detected);
	prf1->time_first_detected = min(prf1->time_first_detected,
					prf2->time_first_detected);
    }
}

void ActiveShapeTracker::initialise_track(Profile *curr_prf)
{
    realno init_scale = curr_prf->height / active_model->mean->height;
    realno init_theta = 0.0;
    curr_prf->a_x = init_scale * cos(init_theta);
    curr_prf->a_y = init_scale * sin(init_theta);
    active_model->set_profile(curr_prf);
    active_model->setup_shape_filters();
    active_model->b_to_x();
    curr_prf->get_size();
    
    curr_prf->pos_filter = 
	new DynamicKalmanTwoD(SQUARE(noise_vel_sd * curr_prf->height), 
			      SQUARE(noise_vel_sd * curr_prf->height),
			      SQUARE(noise_acc_sd * curr_prf->height),
			      SQUARE(noise_acc_sd * curr_prf->height),
			      curr_prf->origin,
			      Point2(0,0),
			      SQUARE(init_pos_sd * curr_prf->height),
			      SQUARE(init_pos_sd * curr_prf->height),
			      SQUARE(init_vel_sd * curr_prf->height),
			      SQUARE(init_vel_sd * curr_prf->height), 1.0);
    
    
    
    curr_prf->a_filter = new StaticKalmanTwoD(0, 0, 0, curr_prf->a_x, 
					      curr_prf->a_y, 0, 0, 0);
    curr_prf->a_filter->set_polar_noise_cov(SQUARE(init_scale_sd), 
					    SQUARE(init_theta_sd));
    curr_prf->a_filter->update_state();
    curr_prf->a_filter->set_polar_noise_cov(SQUARE(noise_scale_sd),
					    SQUARE(noise_theta_sd));
    
    // remember that filters are initialised
    curr_prf->filters_initialised = true;    
    
    
    
    
    
#ifndef NO_DISPLAY
#ifdef DEBUG
    if (debug_level == 3) 
    {
//	inputs->get_video_image()->draw_in_image();
	active_model->set_profile(curr_prf);
	curr_prf->update_shape_filters();
	active_model->apply_virtual_shape();
	active_model->b_to_x();
	
	if (getplanes() == 8)   // CMap (grey?) image  (fi motion image)
	    color(127);
	else
	    if (getplanes() == 24)          // RGB mode  (fi video image)
		RGBcolor(255,255,255);  // use white for initialisations
	
	curr_prf->draw();
	gflush();
	sleep(1);
    }    
    
#endif
#endif
    
}

// tracking loop for each new image
void ActiveShapeTracker::process_frame(Inputs *inputs,
				       Results *results,
				       unsigned int max_objects)
{
    inputs->get_video_image()->draw_in_image();
    
#ifndef NO_DISPLAY
#ifdef DEBUG
    if (debug_level == 3) 
    {
	inputs->get_video_image()->display();
	ListNode<TrackedObject> *curr_obj;
	
	TrackedObjectSet *objects = results->get_tracked_objects();
	
	// get old profiles from tracked object set, predicting each
	for (curr_obj = objects->first; curr_obj != NULL; curr_obj = curr_obj->next)	
	{
	    if (getplanes() == 8)   // CMap (grey?) image  (fi motion image)
		color(127);
	    else
		if (getplanes() == 24)      // RGB mode  (fi video image)
		    Image::set_colour(curr_obj->dat->id);  // set colour according to id
	    
	    curr_obj->dat->regions->draw_boxes();
	    curr_obj->dat->features->draw();
	}
    }
#endif
#endif
    
    // set images for use in ActiveModel and its EdgeDetector
    active_model->set_video_image(inputs->get_video_image());
    active_model->set_difference_image(results->get_difference_image());  // could be NULL.  OK.
    active_model->set_background_image(results->get_background_image());
    active_model->set_previous_video_image(previous_video_image);  // could be NULL.  OK.
    
    results->get_occlusion_image()->clear(CLEAR_MARK);  // FIXME: should we?
    cdebug << " >>> ActiveShapeTracker::process_frame: "
	   << " results->get_occlusion_image()->clear(CLEAR_MARK); --- (data at "
	   << (void*) results->get_occlusion_image()->get_data() << ") <<< "  // FIXME: should we?
	   << endl;
    
    predict_old_objects(results);
    
    track_old_objects(inputs, results);
    
    detect_new_objects(inputs, results);
    
    // maybe we use a MovingEdgeDetector, which needs the previous_video_image
    //   save current image for that purpose in the next frame
    if (previous_video_image != NULL)
	inputs->get_video_image()->copy(previous_video_image);
    
#ifndef NO_DISPLAY
#ifdef DEBUG
    if ((debug_level == 3) && (results->get_tracked_objects()->no_items > 0))
    {
	sleep(5);
    }
#endif
#endif

}

EdgeDetector *ActiveShapeTracker::setup_edge_detector(Inputs *inputs,
						      MotionDetector *motion_detector,
						      char *detection_method)
{
    bool use_background_image = false;
    bool use_difference_image = false;   // whether to use a pre-calculated diff
    
    // do we need a full size background image and a difference image for edge detection?
    if (strstr(detection_method, "FOREGROUND")) 
	use_background_image = use_difference_image = true;
    
    EdgeDetector *edge_detector = NULL;
    
    if (strstr(detection_method, "COLOUR") && 
	(inputs->get_video_image()->get_image_type() != RGB32))
    {
	cerror << " ActiveShapeTracker::setup_edge_detector(): ignoring colour edge detection "
	       << "for non-colour image" << endl;
	detection_method = "FOREGROUND_EDGE";
    }    
    
    if (strcmp(detection_method, "COLOUR_FOREGROUND_EDGE") == 0)
    {
	edge_detector =
	    new ColourForegroundEdgeDetector
	    (edge_threshold,
	     edge_threshold, this);
    }
    else
	if (strcmp(detection_method,"MOVING_EDGE") == 0)
	{
	    // MovingEdgeDetector needs the previous video image.  provide it...
	    // FIXME: assuming RGB32Image.  Problems would be detected in Grey8Image::copy() later.
	    previous_video_image = new RGB32Image(inputs->get_video_image_source()->get_xdim(),
						  inputs->get_video_image_source()->get_ydim());

	    edge_detector =
		new MovingEdgeDetector(edge_threshold, STATISTICAL_EDGE_SEARCH, this);
	}
	else
	    if (strcmp(detection_method,"SIMPLE_EDGE") == 0)
		edge_detector =
		    new SimpleEdgeDetector(edge_threshold,STATISTICAL_EDGE_SEARCH, this);
	    else
		if (strcmp(detection_method,"SOBEL_EDGE") == 0)
		    edge_detector =
			new SobelEdgeDetector(edge_threshold, STATISTICAL_EDGE_SEARCH, this);
		else
		    if (strcmp(detection_method,"FOREGROUND_EDGE") == 0)
			edge_detector =
			    new ForegroundEdgeDetector(edge_threshold, edge_threshold, 
						       MAXIMUM_EDGE_SEARCH, this);
		    else
			if (strcmp(detection_method,"NORMALISED_COLOUR_FOREGROUND_EDGE") == 0)
			    edge_detector =
				new NormalisedColourForegroundEdgeDetector(edge_threshold,
									   edge_threshold, this);
    
    if (edge_detector == NULL)
    {
	cerror << "ActiveShapeTracker::setup_edge_detector(): no valid edge detection "
	       << "method given. " << endl
	       << " trying a simple ForegroundEdgeDetector " << endl;
	edge_detector =
	    new ForegroundEdgeDetector(edge_threshold, edge_threshold, 
				       MAXIMUM_EDGE_SEARCH, this);
    }
    
    return edge_detector;
}

// use Calibration class to create additional hypotheses about person heights
void ActiveShapeTracker::create_hypotheses_using_calibration(Profile *profile,
							     ProfileSet *results)
{
    // image vectors in NAG vector format, homogeneous:
    NagVector origin(3);    // origin (centre) of the person
    NagVector feet(3);      // lowest image point of the person
    NagVector head(3);      // highest image point of the person

    // determine the three positions
    origin[0] = profile->origin.x;
    origin[1] = profile->origin.y;
    origin[2] = 1;

    feet = origin;
    feet[1] = profile->ylo;
    
    head = origin;
    head[1] = profile->yhi;
    
    assert(fabs(profile->height - (profile->yhi - profile->ylo)) < 1.5);
    
    // <= because in extreme cases, the height might be 0...
    assert((Image::image_addressing_mode == IA_TOP_TO_BOTTOM) ^ (feet[1] <= head[1]));

    // convert min/max/typical person height from world to image coordinates at origin
    realno min_image_height = calibration->get_image_distance_from_height_in_cm
	(origin, MIN_WORLD_HEIGHT, MIN_WORLD_HEIGHT / 2);          // height: centre
    
    realno max_image_height = calibration->get_image_distance_from_height_in_cm
	(origin, MAX_WORLD_HEIGHT, MAX_WORLD_HEIGHT / 2);          // height: centre
    
    realno typical_image_height = calibration->get_image_distance_from_height_in_cm
	(origin, TYPICAL_WORLD_HEIGHT, TYPICAL_WORLD_HEIGHT / 2);  // height: centre
    
    // check given profile for its size
    realno world_height = calibration->get_world_distance_from_height_in_pixels
	(feet, profile->height, 0);
    
    if ((profile->height < min_image_height) || (profile->height > max_image_height))
    {
	// too large or too small.  Modify height
	cdebug << "ActiveShapeTracker::create_hypotheses_using_calibration(): "
	       << "height check: " << profile->height << " pix == "
	       << world_height << " cm -> not a person." << endl;
	
	// create and add hypothesis 1: we see the feet of the person
	Profile *new_profile = new Profile (profile->origin.x,
					    profile->ylo + typical_image_height / 2,
					    profile->width,
					    typical_image_height);
	// adjust status
	new_profile->source = profile->source;      // e.g. MEASUREMENT
	
	// add to results
	results->add(new_profile);
	
	// create and add hypothesis 2: we see the head of the person
	new_profile = new Profile (profile->origin.x,
				   profile->ylo + typical_image_height / 2,
				   profile->width,
				   typical_image_height);
	
	// adjust status
	new_profile->source = profile->source;      // e.g. MEASUREMENT
	
	// add to results
	results->add(new_profile);
	
	// if requested, remove original profile from results
	if (new_hypothesis_replaces_measurement == true)
	    results->destroy(profile);
	
	cdebug << "ActiveShapeTracker::create_hypotheses_using_calibration(): "
	       << "changed to " << typical_image_height << " pix, that's " 
	       << calibration->get_world_distance_from_height_in_pixels
	    (origin, TYPICAL_WORLD_HEIGHT, TYPICAL_WORLD_HEIGHT / 2) << " cm (goal: " 
	       << TYPICAL_WORLD_HEIGHT << " cm) " << endl;
	
    }
    else
	cdebug << "ActiveShapeTracker::create_hypotheses_using_calibration(): "
	       << "height check: " << profile->height << " pix == "
	       << world_height << " cm -> OK." << endl;
    
    // TODO: could to more neat stuff here like making more guesses
}

} // namespace ReadingPeopleTracker
