/***************************************************************
 * C++ source
 *
 * File : 	ActiveModel.cc
 *
 * Module :	ActiveModel
 *
 * Author : 	Adam Baumberg
 *
 * Creation Date : Mon May 16 15:25:51 1994 
 *
 * Comments : 	
 *
 ***************************************************************/

#include "ActiveModel.h"

#include <stdlib.h> // for ISO 9899 rand()
#include <math.h>

#include "tracker_defines_types_and_helpers.h"
#include "ProfileSet.h"
#include "Image.h"
#include "PCAclass.h"
#include "NagMatrix.h"
#include "Profile.h"
#include "OcclusionHandler.h"
#include "EdgeDetector.h"
#include "NagMatrix.h"
#include "KalmanFilterOneD.h"
#include "SplineWeights.h"
#include "EnvParameter.h"
#include "RGB32Image.h"
#include "default_pca_model.h"
#include "MotionDetector.h"
#include "ActiveShapeTracker.h"
#include "OcclusionImage.h"


#ifdef DEBUG
#ifndef NO_DISPLAY
#include "os_specific_things.h"  // for usleep()
#endif

#include "MovieStore.h"
#endif // ifdef DEBUG

namespace ReadingPeopleTracker
{

// definition and initialisation of static member variables

// apart from shape there's ox, oy, ax, ay
const int ActiveModel::NO_OF_NON_SHAPE_PARAMS = 4;

#ifdef DEBUG
MovieStore *ActiveModel::demo_movie = NULL;
EnvStringParameter ActiveModel::demo_movie_name("DEMO_MOVIE_NAME");
RGB32Image *ActiveModel::demo_image = NULL;
#endif // ifdef DEBUG


ActiveModel::ActiveModel(ActiveShapeTracker *the_tracker, ImageType the_video_image_type) :     
    P(), H(), N(), evals(), mean_points(0), mean_normals(0)
{
    assert (the_tracker != NULL);  // we need it
    tracker = the_tracker;
    video_image_type = the_video_image_type;
    
    // indicate that we have no images yet.  These have to be set each frame!
    video_image = NULL;	      
    difference_image = NULL;
    background_image = NULL;
    previous_video_image = NULL;
    
    epsilon = 0.0;  // overweighting parameter: no overweighting for now.
    
    // set up PCA model using configuration parameters from ActiveShapeTracker
    setup_pca_model();
}

void ActiveModel::setup_pca_model()   // called once by constructor, sets up PCA model
{
    if (tracker->pca_model_filename != NULL)
    {
	ifstream tmpfile(tracker->pca_model_filename);
	if (!tmpfile)
	{
	    cerror << " ActiveModel::setup(): cannot open PCA model file "
		   << tracker->pca_model_filename << " --- using default model"
		   << endl;
	    tracker->pca_model_filename = NULL;
	}
    }
    
    model_depth = tracker->default_model_depth;
    no_subdivisions = tracker->default_subdivisions;
    
    read_shape_model();
    db.reconstruct(model_depth);
    measure_error.reconstruct(model_depth);
    
    int old_no_subdivisions = no_subdivisions;
    
    int max_bunch = (model_depth + NO_OF_NON_SHAPE_PARAMS) * tracker->bunch_size;
    if (max_bunch > MAX_BUNCH_SIZE)
    {
	max_bunch = MAX_BUNCH_SIZE;
	cerror << "ActiveModel::setup(): bunch size may be restricted to a maximum of "
	       << MAX_BUNCH_SIZE << endl;
    }
    
    while ((no_sample_points = Profile::NO_CONTROL_POINTS * no_subdivisions) < max_bunch)
	no_subdivisions *=2;
    
    
    if (no_subdivisions != old_no_subdivisions)
    {
	cerror << "ActiveModel::setup(): using " << no_subdivisions << " subdivisions " << endl;
    }
    
    
    H_no_rows = 2 * Profile::NO_CONTROL_POINTS * no_subdivisions;
    
    
    // setup full model to sub_point transformation
    SplineWeights sweights(no_subdivisions);
    
    int k;
    
    NagMatrix G(2 * no_sample_points, 2 * Profile::NO_CONTROL_POINTS,0.0);
    NagMatrix dG(2 * no_sample_points, 2 * Profile::NO_CONTROL_POINTS,0.0);
    for (unsigned int i = 0; i < no_sample_points; i++)
    {
	k = (i / no_subdivisions);
	G.set(2 * i, 2 * ((k-1+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	      sweights.w0[i % no_subdivisions]);
	G.set(2 * i, 2 * ((k+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	      sweights.w1[i % no_subdivisions]);
	G.set(2 * i, 2 * ((k+1+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	      sweights.w2[i % no_subdivisions]);
	G.set(2 * i, 2 * ((k+2+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	      sweights.w3[i % no_subdivisions]);
	
	G.set(1 + 2 * i, 1 + 2 * ((k-1+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	      sweights.w0[i % no_subdivisions]);
	G.set(1 + 2 * i, 1 + 2 * ((k+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	      sweights.w1[i % no_subdivisions]);
	G.set(1 + 2 * i, 1 + 2 * ((k+1+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	      sweights.w2[i % no_subdivisions]);
	G.set(1 + 2 * i, 1 + 2 * ((k+2+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	      sweights.w3[i % no_subdivisions]);
	
	
	dG.set(2 * i, 2 * ((k-1+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	       sweights.dw0[i % no_subdivisions]);
	dG.set(2 * i, 2 * ((k+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	       sweights.dw1[i % no_subdivisions]);
	dG.set(2 * i, 2 * ((k+1+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	       sweights.dw2[i % no_subdivisions]);
	dG.set(2 * i, 2 * ((k+2+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	       sweights.dw3[i % no_subdivisions]);
	
	dG.set(1 + 2 * i, 1 + 2 * ((k-1+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	       sweights.dw0[i % no_subdivisions]);
	dG.set(1 + 2 * i, 1 + 2 * ((k+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	       sweights.dw1[i % no_subdivisions]);
	dG.set(1 + 2 * i, 1 + 2 * ((k+1+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	       sweights.dw2[i % no_subdivisions]);
	dG.set(1 + 2 * i, 1 + 2 * ((k+2+Profile::NO_CONTROL_POINTS) % Profile::NO_CONTROL_POINTS),
	       sweights.dw3[i % no_subdivisions]);
    }
    G.multiply(P,H);
    G.multiply(*mean, mean_points);
    mean_sq_distance = mean_points.length2() / no_sample_points;
    
    dG.multiply(P,N);
    dG.multiply(*mean, mean_normals);
//    Point2* prf2 = mean->spline;
    mean->update_size_variables();
    mean->get_size();
}

void ActiveModel::read_shape_model()
{ 
    if (tracker->pca_model_filename == NULL)
    {
	// use the default 5 mode shape model
	
	Profile::NO_CONTROL_POINTS = default_no_control_points;
	unsigned int n2 = 2 * Profile::NO_CONTROL_POINTS;
	mean = new Profile();
	mean->NagVector::copy_data(default_mean_shape,n2);
	
	if (model_depth > n2)
	{
	    cerror << " In ActiveModel::read_shape_model(): "
		   << " Warning: model depth reduced from "
		   << model_depth << " to " << n2 << endl;
	    model_depth = n2;
	}
	
	if (model_depth > default_model_modes) 
	{
	    cerror << " In ActiveModel::read_shape_model(): "
		   << " Warning: model depth reduced from "
		   << model_depth << " to " << default_model_modes << endl;
	    model_depth = default_model_modes;
	}
	
	unsigned int depth = MIN(model_depth, default_model_modes);
	
	P.reconstruct(n2, model_depth);
	P.clear();
	memcpy(P.get_data(), default_eigenvector_data,
	       depth * n2 * sizeof(realno));
	evals.reconstruct(model_depth);
	evals.clear();
	evals.copy_data(default_eigenvalue_data, depth);
	
	return;
    }
    
    PCAclass *pca_model = new PCAclass(tracker->pca_model_filename);
    Profile::NO_CONTROL_POINTS = pca_model->data_size / 2;
    
    if (Profile::draw_arrow)
	Profile::NO_CONTROL_POINTS--;
    
    if (model_depth > pca_model->data_size)
	model_depth = pca_model->data_size;
    
    P.reconstruct(pca_model->data_size, model_depth);
    evals.reconstruct(model_depth);
    
    mean = new Profile();
    mean->NagVector::copy_data(pca_model->mean.get_data(), 2*Profile::NO_CONTROL_POINTS);
    
    unsigned int i;
    unsigned int mode;
  
    // setup model to spline point transformation
    for (mode = 0; mode < model_depth; mode++)
    {
	realno *evec = pca_model->get_evec(mode+1);
	for (i = 0; i < pca_model->data_size; i++)
	    P.set(i,mode,evec[i]);
	evals[mode] = pca_model->get_eval(mode+1);
    }
    delete pca_model;
}

void ActiveModel::setup_shape_filters()
{
    if (currx->filters == NULL)
	currx->filters = new FilterOneD*[model_depth];
    
    for (int lp = 0; lp < model_depth; lp++)
	currx->filters[lp] = NULL;
    
    currx->no_shape_parameters = model_depth;
    
    for (unsigned int i = 0; i < model_depth; i++)
    {
	if (currx->filters[i] != NULL)
	    delete currx->filters[i];
	
	currx->filters[i] = (FilterOneD*) new StaticKalmanOneD
	    (SQUARE(tracker->noise_scalar) *  evals[i]	// q^2 	
	     ,4			                // r^2  
	     ,0 		         	// b0 	
	     ,SQUARE(tracker->initial_noise) * evals[i]	// p11 	
	     ,epsilon);
    }
}

void ActiveModel::get_estimate(int id, Point2 &p, Point2 &n, Point2 &q)
{
    int id_x = 2 * id;
    int id_y = id_x + 1;
    q = Point2(mean_points[id_x], mean_points[id_y]);
    Point2 tmp(mean_normals[id_x], mean_normals[id_y]);
    int offset = 0;
    
    for (unsigned int i = 0; i < model_depth; i++)
    {
	realno b_i = currx->filters[i]->get_state();
	q.x += H[id_x+offset] * b_i;
	q.y += H[id_y+offset] * b_i;
	tmp.x += N[id_x+offset] * b_i;
	tmp.y += N[id_y+offset] * b_i;
	offset += H_no_rows; 
    }
    
    n = tmp.map(currx->a_x, currx->a_y);
    n = Point2(-n.y, n.x);
    p = q.map(currx->a_x, currx->a_y);
}

void ActiveModel::b_to_x()
{
    NagVector b(model_depth);
    
    for (unsigned int i = 0; i < model_depth; i++)
	b[i] = currx->filters[i]->get_state();
    
    P.multiply(b, *currx);
    currx->add(*mean, *currx);
    currx->map(currx->a_x, currx->a_y, currx);
}



void ActiveModel::quick_observe_point(Point2 &n, Point2 &p, Point2 &q, int id,
				      realno &mxx, realno &mxy, realno &myy,
				      realno &inv_var,
				      realno &m_density, Point2 &dp)
{
//    int delta_id = no_sample_points / current_bunch_size;
    get_estimate(id, p, n, q);  
    m_density = n.length();
    n = n / m_density;  // normalise

    inv_var = 0;
    mxx = 0; mxy = 0; myy = 0;
    Point2 pos = p + currx->origin;
    
    if ((occ_handler->is_occlusion(pos.x, pos.y)) || 
	(occ_handler->is_off_image(pos.x, pos.y)))
    {
	return;
    }
    
    // since there is no occlusion we should now find an edge
    realno cov_xx = 0;
    realno cov_yy = 0;
    realno cov_xy = 0;
    int id_x = id * 2;
    int id_y = id_x + 1;
    int offset = 0;
    realno curr_ax = currx->a_x;
    realno curr_ay = currx->a_y;
    
    for (unsigned int i = 0; i < current_depth; i++)
    {
	realno b_i = currx->filters[i]->get_p11();
	realno tmp_x = curr_ax * H[id_x+offset] - curr_ay * H[id_y+offset];
	realno tmp_y = curr_ay * H[id_x+offset] + curr_ax * H[id_y+offset];
	cov_xx += (tmp_x * tmp_x) * b_i;
	cov_xy += (tmp_x * tmp_y) * b_i;
	cov_yy += (tmp_y * tmp_y) * b_i;
	offset += H_no_rows;
    }
    
    Matrix2by2 point_covariance(cov_xx, cov_xy, cov_yy);
    point_covariance += currx->pos_filter->get_cov();
    
    // add term due to a_x, a_y 
// nts: next 3 were unused --- removed beginning 2002
//    realno py2 = q.y * q.y;
//    realno px2 = q.x * q.x;
//    realno pxy = 2.0 * q.x * q.y;
    
    Matrix2by2 Cov_a;
    
    if (currx->a_filter != NULL)
	Cov_a = currx->a_filter->get_cov();
    else
    {
	realno cth = 1.0;
	realno sth = 0.0;
// nts: next 2 were unused --- removed beginning 2002
//	realno s_hat = currx->s_filter->get_state();
//	realno sq_s_hat = s_hat * s_hat;
	realno var_s = currx->s_filter->get_p11();
	realno var_th = 0.0;
	if (currx->th_filter != NULL)
	{
	    realno theta = currx->th_filter->get_state();
	    var_th = currx->th_filter->get_p11();
	    cth = cos(theta);
	    sth = sin(theta);
	}
	
	Cov_a = Matrix2by2(var_s * cth * cth + var_th * sth * sth,
			   (var_s - var_th) * cth * sth,
			   var_s * sth * sth + var_th * cth * cth);
    }
    
    
    point_covariance += (Matrix2by2(q.x, -q.y, q.y, q.x) * Cov_a) * 
	Matrix2by2(q.x, q.y, -q.y, q.x);
    
    Point2 edge_orientation = n;
    
    if (tracker->do_m_search)
    {
	// use Mahalanobis search direction
	n = point_covariance.multiply(n);
	n.normalise();
    }
    
    mxx = n.x * n.x;
    mxy = n.x * n.y;
    myy = n.y * n.y;
    
    realno scale = 2.0 / sqrt(n * (point_covariance.inverse()).multiply(n));    
    
    realno sq_scale = scale * scale;
    
    if (sq_scale < 1.0)
	return;
    
    Point2 direction = scale * n;
    
    realno measure_var;
    Point2 p_obs;  
    
    measure_var = frame_step;
    edge_status_t edge_status =
	edge_detector->find_edge(((realno)id) / ((realno)no_sample_points),
			       pos, n,
			       edge_orientation, scale,
			       occ_handler, currx, p_obs,
			       measure_var, tracker->get_active_model());
    
    if (edge_status != EDGE_OCCLUDED)
	no_looks++;
    
    if (edge_status == EDGE_GOOD)
    {
	dp = p_obs - pos;
	measure_var +=  (var_fac * scale * scale) + var_const;
	inv_var = (frame_step) / (current_bunch_size * measure_var);
    }
    else
    {
	// edge is occluded or was not found
	dp = Point2(0,0);
	inv_var = 0;
    }   
    
#ifndef NO_DISPLAY
#ifdef DEBUG
    if (tracker->debug_level == 1)
    {
	// draw search line and results into image
	Point2 pos_r = pos + direction;
	Point2 pos_l = pos - direction;
	
	Point2 tmp_n(-n.y, n.x);
	realno unit1 = scale / 5.0;
	
	if (video_image != NULL)
	{
	    // draw into video image
	    video_image->draw_in_image();
	}
	else
	    if (difference_image != NULL)
	    {
		// draw into difference image
		difference_image->draw_in_image();
	    }
	
	linewidth(1);
	setlinestyle(0);
	RGBcolor(0,255,255);

	Image::move2(pos_r + unit1 * tmp_n);
	Image::draw2(pos_r - unit1 * tmp_n);
	Image::move2(pos_r);
	Image::draw2(pos_r - unit1 * n);
	Image::move2(pos_l + unit1 * tmp_n);
	Image::draw2(pos_l - unit1 * tmp_n);
	Image::move2(pos_l);
	Image::draw2(pos_l + unit1 * n);
	
	Image::move2(pos_r);
	Image::dotted_draw2(pos_l);
	
	RGBcolor(0,200,40);
	Image::circ(pos, 2);      // mark expected edge
    }
#endif
#endif   // #ifndef NO_DISPLAY
    if (inv_var  < 1e-20)
	return;
    
#ifndef NO_DISPLAY
#ifdef DEBUG
    if (tracker->debug_level == 1)
    {
	RGBcolor(255,0,0);
	Image::circ(p_obs, 2);    // mark detected edge
    }
#endif
#endif   // #ifndef NO_DISPLAY
    
    no_observed++;
}

void ActiveModel::get_new_depth()
{
//   realno sq_pos_scale = currx->x_filter->get_p11() +
//     currx->y_filter->get_p11();
    
//   realno sq_pos_scale = 
//     MIN(currx->x_filter->get_p11(),currx->y_filter->get_p11());
    
    realno sq_pos_scale =
	currx->pos_filter->get_cov().get_min_real_eigenvalue();
    
    sq_pos_scale = (var_fac * 4.0 * sq_pos_scale) + var_const;

    realno frame_sq_scale =  sq_pos_scale / 
	(SQUARE(currx->a_x) + SQUARE(currx->a_y));
    
    realno m_var = Profile::NO_CONTROL_POINTS * frame_sq_scale / frame_step;
    
    while (current_depth < model_depth)
    {
	register realno shape_var = currx->filters[current_depth]->get_p11();
	register realno gain = shape_var / (shape_var + m_var);
	
	if (gain > 0.01)
	    current_depth++;
	else
	    break;
    }
#ifdef  DEBUG
    if (tracker->debug_level == 1)
    {
	cdebug << "AM : new model depth " << current_depth << endl;
    }
#endif
}


realno ActiveModel::adjust_state()
{  
#ifndef NO_DISPLAY
#ifdef DEBUG
    if (tracker->debug_level == 1)
    {
	b_to_x();
	
	if (video_image != NULL)
	{
	    // draw current shape into video image
	    video_image->display();   // to remove previous drawings
	    video_image->draw_in_image();

	    RGBcolor(255,255,255);
	    linewidth(2);
	    setlinestyle(0);

	    currx->draw();
	}
	
//  	if (difference_image != NULL)
//  	{
//  	    // nts: we are usually looking for edges in this difference image
//  	    //      so draw current shape here as well!    ---  added April 2001
//  	    difference_image->display();	
//  	    currx->draw();
//  	}
    }
#endif
#endif   // #ifndef NO_DISPLAY
    
// nts: next 2 were unused --- removed beginning 2002
//    realno scale = 0;
//    realno size = currx->a_x * currx->a_x + currx->a_y * currx->a_y;    
    
    // added new twist --
    // dynamically vary the model depth 
    // and the bunch size
    // based on the positional search scale !
    
    get_new_depth();
    
    current_bunch_size = (current_depth + NO_OF_NON_SHAPE_PARAMS) 
	* tracker->bunch_size;    
    
    if (current_bunch_size > MAX_BUNCH_SIZE) 
	current_bunch_size = MAX_BUNCH_SIZE;
    
#ifdef DEBUG
     if (tracker->debug_level == 1)
     {
	 cdebug << " ActiveModel::adjust_state: current bunch size " << current_bunch_size << endl;
     }
#endif
     
    int id = rand() % no_sample_points;  // start somewhere along the contour
    
    NagVector pos_weights(current_bunch_size);
    
    int lp;
    
#ifdef DEBUG
    if (tracker->debug_level == 1)
    {
 	cdebug << " ActiveModel::adjust_state: "
	       << "no_sample_points " << no_sample_points
	       << ", current_bunch_size " << current_bunch_size
	       << ", tracker->bunch_size " << tracker->bunch_size
	       << ", Profile::NO_CONTROL_POINTS " << Profile::NO_CONTROL_POINTS
	       << endl;
    }
#endif

    for (lp = 0; lp < current_bunch_size; lp++)
    {
	
	ids[lp] = (id + (lp * no_sample_points) / current_bunch_size) %
	    no_sample_points;
	
	quick_observe_point(n[lp], p[lp], q[lp], ids[lp],
			    mxx[lp], mxy[lp], myy[lp],
			    inv_var[lp], pos_weights[lp], dp[lp]);
    }
    
#ifndef NO_DISPLAY
#ifdef DEBUG
    if (tracker->debug_level == 1)
    {
	gflush();
	sleep(1);   // Show the results for some time
    }
#endif
#endif   // #ifndef NO_DISPLAY
    
    realno ox_err= 0, oy_err = 0, oxy_err = 0, dox = 0, doy = 0;
    
    // weight measurements according to sampling density
    pos_weights.scale(current_bunch_size / (pos_weights.sum()),
		      pos_weights);

    for (unsigned int lp1 = 0; lp1 < current_bunch_size; lp1++)
    {	 
	 inv_var[lp1] *= pos_weights[lp1];
    }
    
    
// commented out as it does not work very well    
//     // if there was occlusion then make the
//     // other obervations more significant  
//      // but make sure we are tracking a significant section of contour
//      realno min_no_looks = 
//  	MAX(MIN_SIGNIFICANCE * current_bunch_size, no_looks);
//    
//      realno look_fac = current_bunch_size / min_no_looks;
    
    realno look_fac = 1.0;
    
    for (lp = 0; lp < current_bunch_size; lp++)
    {
	inv_var[lp] *= look_fac;
	ox_err += inv_var[lp] * mxx[lp];
	dox += inv_var[lp] * (dp[lp].x * mxx[lp] + dp[lp].y * mxy[lp]);
	oy_err += inv_var[lp] * myy[lp];
	oxy_err += inv_var[lp] * mxy[lp];
	doy += inv_var[lp] * (dp[lp].y * myy[lp] + dp[lp].x * mxy[lp]);
    }
    
     realno gain_pos = 
	currx->pos_filter->delta_update(Point2(dox, doy),
					Matrix2by2(ox_err, oxy_err,oy_err));
    
    Point2 origin = currx->pos_filter->get_state();
    currx->origin = origin;    
    
    Point2 delta_origin = currx->pos_filter->state_change();
    
    realno axx_err = 0, axy_err = 0, ayy_err = 0, dax = 0, day = 0;
    
    for (lp = 0; lp < current_bunch_size; lp++)
    {
	dp[lp] -= delta_origin;
	p[lp] += dp[lp];
	realno &qx = q[lp].x; realno &qy = q[lp].y;
	realno qxx = qx * qx; realno qxy = qx * qy; 
	realno qyy = qy * qy;
	
	axx_err +=
	    inv_var[lp] * (qxx * mxx[lp] + 2 * qxy * mxy[lp] + 
			   qyy * myy[lp]);
	axy_err += 
	    inv_var[lp] * ((qxx - qyy) * mxy[lp] + 
			   qxy * (myy[lp] - mxx[lp]));
	
	ayy_err +=
	    inv_var[lp] * (qyy * mxx[lp] - 2 * qxy * mxy[lp] + 
			   qxx * myy[lp]);
	
	realno &px = dp[lp].x; realno &py = dp[lp].y;
	realno pxqy = px * qy; realno pyqx = py * qx;
	realno pxqx = px * qx; realno pyqy = py * qy;
	
	dax += inv_var[lp] *
	    (mxy[lp] * (pxqy + pyqx) + mxx[lp] * pxqx + myy[lp] * pyqy);
	day += inv_var[lp] *
	    (mxy[lp] * (pxqx - pyqy) - mxx[lp] * pxqy + myy[lp] * pyqx);
	
    }
    
    realno gain_s = 0;
    realno gain_th = 0;
    realno gain_a = 0;
    
    realno old_ax = currx->a_x;
    realno old_ay = currx->a_y;
    
    if (currx->a_filter == NULL)
    {
	// use polar (s,th) representation
	realno th = 0.0, sth = 0.0, cth = 1.0;
	if (currx->th_filter != NULL)
	{
	    th = currx->th_filter->get_state();
	    sth = sin(th); cth = cos(th);
	}
	realno s = currx->s_filter->get_state();      
	
	// map measurements to polar representation
	Matrix2by2 H(cth, -s * sth, sth, s * cth);
	Matrix2by2 R_inv(axx_err, axy_err, ayy_err);
	Matrix2by2 tmp_M = H.transpose() * R_inv * H;
	Point2 tmp_P = H.transpose() * Point2(dax, day);
	
	// filter s and th independently
	gain_s = currx->s_filter->delta_update(tmp_P.x, tmp_M.val00);
	s = currx->s_filter->get_state();
	if (currx->th_filter != NULL)
	{
	    gain_th = 
		currx->th_filter->delta_update(tmp_P.y, tmp_M.val11);
	    th = currx->th_filter->get_state();
	    cth = cos(th); sth = sin(th);
	}
	
	currx->a_x = s * cth;
	currx->a_y = s * sth;      
    }
    else
    {
	
	gain_a = 
	    currx->a_filter->delta_update(dax, day, axx_err, axy_err,
					  ayy_err);
	currx->a_x = currx->a_filter->get_xstate();
	currx->a_y = currx->a_filter->get_ystate();
	
    }
    
    measure_error.clear();
    db.clear();
    
    for (lp = 0; lp < current_bunch_size; lp++)
    {
	
	Point2 map_n = n[lp].map(old_ax , -old_ay);
	
	realno nxx = (map_n.x * map_n.x);
	realno nxy = (map_n.x * map_n.y);
	realno nyy = (map_n.y * map_n.y);
	realno mnxx = (map_n.x * n[lp].x);
	realno mnxy = (map_n.x * n[lp].y);
	realno mnyx = (map_n.y * n[lp].x);
	realno mnyy = (map_n.y * n[lp].y);
	
	int id_x = ids[lp] * 2;
	int id_y = id_x + 1;
	int offset = 0;
	
	for (unsigned int i = 0 ; i < current_depth; i++)
	{
	    realno tmp_Hx = H[id_x+offset];
	    realno tmp_Hy = H[id_y+offset];
	    Point2 tmp_h(tmp_Hx, tmp_Hy);
	    tmp_h = tmp_h.map(old_ax, old_ay);
	    
	    offset += H_no_rows;
	    measure_error[i] += inv_var[lp] * 
		( tmp_Hx * tmp_Hx * nxx +
		  2 * tmp_Hy * tmp_Hx * nxy +
		  tmp_Hy * tmp_Hy * nyy);
	    db[i] += inv_var[lp] *
		(dp[lp].x * (tmp_Hx * mnxx +
			     tmp_Hy * mnyx) +
		 dp[lp].y * (tmp_Hx * mnxy +
			     tmp_Hy * mnyy));	  
	}
    }
    
    	
    realno new_max_gain = gain_pos;
    new_max_gain = fmax(new_max_gain, gain_a);
    new_max_gain = fmax(new_max_gain, gain_s);
    new_max_gain = fmax(new_max_gain, gain_th);
    
    for (unsigned int i = 0; i < current_depth; i++)
    {
	realno gain_b_i =
	    currx->filters[i]->delta_update(db[i], measure_error[i]);
	new_max_gain = fmax(new_max_gain, gain_b_i);
    }
    
    return new_max_gain;
}


realno ActiveModel::get_M_distance(FilterOneD **filters, unsigned int depth,
				   NagVector &evals) const
{
    realno dist = 0;
    for (unsigned int i = 0; i < depth; i++)
    {
	realno b = filters[i]->get_state();
	realno sq_b  = b * b;
	dist += sq_b / evals[i];
    }
    return dist;
}

void ActiveModel::constrain_b()
{
    realno blength = get_M_distance(currx->filters, current_depth, evals);

    if (blength >= (tracker->cloud_size * tracker->cloud_size))
    {
	realno mfac = tracker->cloud_size / sqrt(blength);
	for (unsigned int i = 0; i < current_depth; i++)
	    currx->filters[i]->get_state() *= mfac;
    }
}



void ActiveModel::get_initial_frame_step()
{
    realno initial_gain = tracker->minimum_best_gain;
    realno v_pos = currx->pos_filter->get_cov().get_max_real_eigenvalue();
    realno vb0 = currx->filters[0]->get_p11();
    
    realno va;
    
    if (currx->a_filter != NULL)
	va = currx->a_filter->get_uncertainty();
    else
	va = currx->s_filter->get_p11();
    
    // estimate frame_step for desired gain ... assuming
    // istropic measurements with variance proportional
    // to uncertainty
    
    realno fstep_lower_bound = initial_gain * 4.0 * var_fac / 
	(1.0 - initial_gain);    
    
    // estimate frame_step for desired gain ... assuming isotropic
    // measurements with fixed measurement variance
    
    // (gain for position parameter)
    realno fstep_pos = (initial_gain * var_const) / 
	(v_pos * (1.0 - initial_gain));
    
    // (gain for principal shape parameter)
    realno s2 = SQUARE(currx->a_x) + SQUARE(currx->a_y);
    realno fstep_b0 = (initial_gain * (Profile::NO_CONTROL_POINTS * var_const / s2))
	/ (vb0 * (1.0 - initial_gain));
    // gain for a_x and a_y
    realno fstep_a = (initial_gain * (var_const / mean_sq_distance))
	/ (va * (1.0 - initial_gain));
    
    realno min_frame_step = fstep_pos;
    min_frame_step = fmin(min_frame_step, fstep_b0);
    min_frame_step = fmin(min_frame_step, fstep_a);
    min_frame_step = fmax(min_frame_step, fstep_lower_bound);
    
    frame_step = min_frame_step;
}

realno ActiveModel::track_profile(object_id_t object_id)
{
    var_fac = SQUARE(tracker->measure_fac_sd);
    var_const = SQUARE(tracker->measure_const_sd);
    
    if (edge_detector == NULL)
    {
	cerror << "ActiveModel::track_profile(): No edge detector has been set up." << endl
	       << " trying SobelEdgeDetector... " << endl;
	edge_detector = new SobelEdgeDetector(0.05, MAXIMUM_EDGE_SEARCH, tracker);
    }
    
    occ_handler->setup_occlusion_map(object_id);
    
    get_initial_frame_step();
    
    frame_time = 0;
    
    int no_iterations = 0;
    no_observed = no_looks = 0;
    no_sampled = 0;
    
    int total_no_looks = 0;
    
    if (tracker->use_hierarchical_search)
	current_depth = 0;
    else
	current_depth = model_depth;
    
    get_new_depth();
    
    edge_detector->setup_batch(currx);
    
    realno max_gain;

    while (frame_time < 1.0)
    {
	no_looks = 0;
	max_gain = adjust_state();
	no_sampled += current_bunch_size;	
	
#ifdef DEBUG
	if ((tracker->debug_level == 1) && (demo_movie_name.value[0] != '\0'))
	{
	    if (demo_movie == NULL)
#ifdef LINUX
		demo_movie = new MovieStore(MOVIESTORE_JPEG, demo_movie_name.value);
#else
	    demo_movie = new MovieStore(demo_movie_name.value);
#endif
	    demo_image = video_image->rgb_read_image(demo_image);
	    if (!demo_movie->add_a_frame(demo_image))
		demo_movie_name.value[0] = '\0';
	}
#endif
	
	frame_step = fmin(frame_step, 1.0 - frame_time);
	
	    
//	// do not let max_gain go under the minimum  // FIXME: should not happen anyway!
//	max_gain = fmax(max_gain, tracker->minimum_best_gain);  // nts: testing Aug 2002 /// ////

	frame_time += frame_step;
	if (max_gain < tracker->minimum_best_gain) 
	    frame_step *= 2;
	total_no_looks += no_looks;
	no_iterations++;
	
	constrain_b();
	Point2 origin = currx->pos_filter->get_state();
	currx->origin = origin;
	
#ifndef NO_DISPLAY
#ifdef DEBUG
	if (tracker->debug_level == 2) 
	{
	    b_to_x();
	    if (video_image != NULL)
	    {
		// draw into video image
		video_image->display();
		video_image->draw_in_image();
		currx->draw();
	    }
	    
//   	    if (difference_image != NULL)
//   	    {
//   		// nts: draw into difference image, too
//   		difference_image->display();	
//   		currx->draw();
//   	    }
	}
#endif
#endif   // #ifndef NO_DISPLAY
    }
    
    b_to_x();
    
    no_looks = total_no_looks;
    
    return get_fitness();
}


void ActiveModel::apply_virtual_shape()
{
    // add a virtual observation 
    for (unsigned int i = 0; i < model_depth; i++)
	currx->filters[i]->add_observation(0, 1.0 * evals[i]);
}

void ActiveModel::hold_shape()
{
    for (unsigned int i = 0; i < model_depth; i++)
	delete currx->filters[i];
    
    setup_shape_filters();
}  

} // namespace ReadingPeopleTracker
