///////////////////////////////////////////////////////////////////////////////
//  video_play.cc                                                            //
//                                                                           //
//  This program simply plays back video input, storing if desired           //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Thu Nov 16 2000                                              //
//  Revision  : 1.0 of Thu Nov 16 14:29:52 GMT 2000                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
//  Changes:                                                                 //
//   nts: rev 1.0: initial working revision    Thu Nov 16 14:29:52 GMT 2000  //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


#include <cstdio>
#include <getopt.h>

#include "Image.h"
#include "ImageSource.h"
#include "MovieStore.h"

#include "Point2.h"
#include "Point2.cc"
#include "Inputs.h"
#include "Inputs.cc"


#ifdef PROFILING
#include "Profiling.h"
#endif

using namespace ReadingPeopleTracker;

static const char *video_play_Revision = "@(#) video_play.cc, rev 1.0 of Thu Nov 16 14:29:52 GMT 2000, Author Nils T Siebel, Copyright (c) 2000--2002 The University of Reading";

#define USAGE "video_play -i options [-l n] [-o file] -q \n\
-i input_options (-i help for details) \n\
-l last frame number to process \n\
-o output numbered jpeg files, give first name \n"

int main(int argc, char **argv)
{ 
    ImageSource *input_source = NULL;
    bool show_images = true;
    char *output_movie = NULL;
    char *parameter_file = NULL;
    MovieStore *mv_out = NULL; 
    
    frame_id_t last_frame = 0;
    int c;
    
    while ((c = getopt(argc, argv, "i:l:o:q:")) != EOF)
	switch(c)
	{
	case 'i':
	    if (Inputs::setup_image_source_input(optarg, input_source) == false)
	    {
		cerror << argv[0] << ": could not set up input.  Exiting... "
		       << endl;
		exit(1);
	    }
	    break;
	
	case 'o':
	    if (optarg != NULL) output_movie = optarg;
	    break;
	    
	case 'l':
	    last_frame = atoi(optarg);
	    break;
	    
	case 'q':
	    show_images = false;
	    break;
	    
	case '?':
	default:
	    fprintf(stderr, USAGE);
	    exit(1);
	}

    if (input_source == NULL)
    {
	cerror << argv[0] << ": No input specified.  "
	     << "Please use the `-i' option (`-i help' for help)"
	     << endl;
	exit(1);
    }
    
    ImageSource &vidsrc = *input_source;
    
    //Configuration.output_info();
    
    if (show_images)
	vidsrc.display();
    
    if (output_movie != NULL)
#ifdef LINUX
	mv_out = new MovieStore(MOVIESTORE_JPEG,output_movie);
#else
        mv_out = new MovieStore(output_movie, 0, true);
#endif
    
    /********************************************************************/
    /*                                                                  */
    /*                      Main Loop                                   */
    /*                                                                  */
    /********************************************************************/
    
#ifdef PROFILING
    Profiling::initialise_timer();  // initialise timing routine, see Profiling.h
    setusecs(0);
#endif

    while (vidsrc.get_next() != NULL) 
    {
	if ((last_frame != 0) && (vidsrc.get_frame_id() >= last_frame))
	    break;
	
#ifdef PROFILING
	fprintf(stderr,"timer: time spent on getting this frame: %i microsec \n",
	    (int) (getusecs()));
	setusecs(0);
#endif
	
	if (show_images)
	{
	    vidsrc.display();
#ifdef PROFILING
	    fprintf(stderr,"timer: time spent on displaying: %i microsec \n",
		    (int) (getusecs()));
	    setusecs(0);
#endif
	    cinfo << "\r frame " << vidsrc.get_frame_id() << flush;
	}
	else
	    cinfo << " frame " << vidsrc.get_frame_id() << endl;
	
	if (output_movie != NULL)
	{
	     mv_out->add_a_frame(vidsrc.get_current());
#ifdef PROFILING
	fprintf(stderr,"timer: time spent on writing output frame: %i microsec \n",
	    (int) (getusecs()));
	setusecs(0);
#endif
	}
    }
    
    if (output_movie != NULL)
	delete mv_out;

    cinfo << endl;
    
    return 0;
}

// Local Variables:
// compile-command: "make video_play |& fgrep :"
// End:
