/* test for ProfileSequence class */

#include "ProfileSequence.h"
#include <gl/gl.h>
#include <getopt.h>
#include <math.h>
#include <stdlib.h>
#include "utils.h"

#include "icons.h"

using namespace ReadingPeopleTracker;

#define USAGE " process_sequence -a alignfile -t trackfile -o outputfile \
-p pca_file -s show_file \n\
-r reflect_file \n\
-m model_input_file -n look_at_mode_n \n\
-c center_file \n\
-x use_extended_model \n\
-q quiet mode (no GUI) \n"


static bool use_extended_model =
Configuration.register_bool("EXTENDED_MODEL", false,
		      &Profile::draw_arrow, true,
		      "main", 
		      "Use extended model ? The extended model \
includes the tracked displacement of each object as part of the \
shape vector");

static int mode_index = 
Configuration.register_int("MODE_NUMBER", 1,
		     &mode_index, false,
		     "main",
		     "The shape mode to display (minimum is 1)");

static char *inputfile = 
Configuration.register_filename("INPUT_SEQUENCE_FILE", "",
			  &inputfile, 
			  "main",
			  "The input file containing extracted \
silouhette (profile) data");

static char *model_file =
Configuration.register_filename("INPUT_MODEL_FILE", "",
			  &model_file,
			  "main",
			  "The input linear model file");

static char *outfile =
Configuration.register_filename("OUTPUT_FILE", "",
			  &outfile,
			  "main",
			  "Filename for output");

static char *functions[] = { "align", "center", "reflect",		     
			     "track", "show shapes", 
			     "principal component analysis",
			     "display modes",
			     NULL };

static char *function_mode = 
Configuration.register_string("FUNCTION", "display_modes",
			&function_mode, true,
			"main", 
			"Pick a function ... \n\
If no output file has been specified, use the input file with \
an additional extension described below.\
\n\n\
align: Align the shapes in the data file. Output extension is '.align'\
\n\n\
center: Recenter the shapes (i.e. align without rotating). \
Output extension is '.centered' \
\n\n\
reflect: Reflect each input shape. Output extension is '.reflect'\
\n\n\
track: Label and track the input shapes between frames. Output \
extension is '.track' \
\n\n\
show_shapes: Display the input shapes. If a model file has been \
specified project each shape into the model subspace. \
\n\n\
If an output file has been specified then typing 'n' at the terminal will \
throw away the displayed shape. Any other keys accepts the shape\
\n\n\
principal component analysis: Do PCA on input data. Output a model \
file for tracking etc\
\n\n\
display modes: Visualize the shape modes of variation for the input \
model file\n\n", functions);



#include <termio.h>
#include <osfcn.h>
int tty_cbreak (int fd);    // put tty in cbreak (raw, !cooked) mode
int tty_reset (int fd);     // restore terminal's mode
static struct termios save_stdin;

int main(int argc, char** argv)
{
    Profile::draw_in_colour = true;
    Profile::draw_boxes = true;
    Profile::draw_linewidth = 8;
    Profile::draw_box_size = 2.0;
    
    bool no_interface = false;
    char mode = 'd';
    int c;
    bool tmp_sequence = false;
    while ((c = getopt(argc, argv, "qxn:m:a:c:r:t:o:p:s:")) != EOF)
	switch(c)
	{
	case 'n':
	    if (optarg != NULL) mode_index = atoi(optarg);
	    break;
	case 'm':
	    if (optarg != NULL) model_file = optarg;
	    break;
	case 'a':
	case 'c':
	case 'r':
	case 't':
	case 'p':
	case 's':
	    inputfile = optarg;
	    mode = c;
	    break;
	case 'o':
	    outfile = optarg;
	    break;	
	    
	case 'x':
	    Profile::draw_arrow = true;
	    break;
	    
	case 'q':
	    no_interface = true;
	    break;
	    
	case '?':
	default:	
	    fprintf(stderr," usage - %s", USAGE);
	    exit(1);
	}

    for (int i = 0; functions[i] != NULL; i++)
	if (functions[i][0] == mode)     // compare first letter
	    function_mode = functions[i];      
    
    mode = function_mode[0];
    
    if (! no_interface)
    {
  	// set default icon for windows...
  	Configuration.set_icon(process_sequence_icon_16);

	Configuration.create_interface(argc, argv);
    }
    
    char* in_ext = strrchr(inputfile, '.');
    if (in_ext != NULL)
	if ((strncmp(in_ext,".Z",2) == 0) ||
	    (strncmp(in_ext, ".gz", 2) == 0))
	{
	    char sys_command[200];
	    sprintf(sys_command," zcat %s > /tmp/tmp_sequence.txt",
		    inputfile);
	    system(sys_command);
	    inputfile = "/tmp/tmp_sequence.txt";
	    tmp_sequence = true;
	}
    
    if (strlen(outfile) == 0)
	outfile = NULL;

    if (strlen(inputfile) == 0)
	inputfile = NULL;

    if (strlen(model_file) == 0)
	model_file = NULL;
    
    ProfileSequence data(inputfile, model_file);
    
    switch (mode)
    {
    case 'a':
	data.align(outfile);
	break;
    case 'c':
	data.center(outfile);
	break;
    case 'r':
	data.reflect(outfile);
	break;
    case 't':
	data.track(outfile);
	break;
    case 's':
    {
	prefsize(300, 400);

	winopen(" profiles ");
	
#ifdef USE_GL
	RGBmode();
	gconfig();
	RGBcolor(0,0,0);
#else
        color(BLACK);
#endif
	clear();  

	Point2 centre(150, 200);
	if (outfile == NULL)
	    data.display_shapes(centre);
	else 
	{
	    tty_cbreak(STDIN_FILENO);
	    data.edit_shapes(centre, outfile);
	    tty_reset(STDIN_FILENO);
	}
	break;
    }
    
    
    case 'p':
	data.do_analysis();
	if (outfile != NULL)
	{
	    ofstream out_file(outfile);
	    out_file << *(data.pca) << endl;
	    out_file.close();
	    break;
	}
	outfile = NULL;
	
    case 'd':
    {
	if (data.pca  == NULL)
	    break;
	int indx = 0;
	if (outfile != NULL)
	{
	    ofstream data_out(outfile);
	    data_out << *(data.pca);
	    data_out.close();
	}
	data.display_mode(mode_index);
	gexit();
    }
    }    
    if (tmp_sequence)
	system("rm -f /tmp/tmp_sequence.txt");
    
    return 0;
}

// from W Richard Steven's book
// ``Advanced Programming in the Unix Environment''

int tty_cbreak (int fd)             // put terminal into a cbreak mode
{
    struct termios buf;

    if (fd == STDIN_FILENO)
    {
	if (tcgetattr (fd, &save_stdin) < 0)
	    return -1;
    }
    buf = save_stdin;             // structure copy

    buf.c_lflag &= ~(ECHO | ICANON);
    // echo off, canonical mode off

    buf.c_cc[VMIN] = 1;           // Case B: 1 byte at a time, no timer
    buf.c_cc[VTIME] = 0;

    if (tcsetattr (fd, TCSAFLUSH, &buf) < 0)
	return -1;
    return 0;
}


int tty_reset (int fd)              // restore terminal's mode
{
    if (fd == STDIN_FILENO)
    {
	if (tcsetattr (fd, TCSAFLUSH, &save_stdin) < 0)
	    fprintf (stderr, "\n Failed to reset stdin \n");
	return -1;
    }
    return 0;
}

// Local Variables:
// compile-command: "make process_sequence"
// End:
