

// program to extract shapes of moving blobs from video

// major makeover inspired by new features in MotionDetector --- nts Feb 2001

// system includes
#include <cstdio>
#include <gl/gl.h>
#include <getopt.h>
#include <climits>  // for INT_MAX

// the following 3 includes are for reading keyboard input only
#include <sys/ioctl.h>
#include <termios.h>
#include <unistd.h>   // this one also for usleep()

// local includes
#include "Image.h"
#include "miscsrc.h"
#include "RegionSet.h"
#include "ProfileSet.h"

#include "Grey8Image.h"

#include "EnvParameter.h"
#include "utils.h"
#include "imgsrc_utils.h"
#include "tracker_defines_types_and_helpers.h"

#include "icons.h"

using namespace ReadingPeopleTracker;

// when the user presses '+' or '-' in/decrease threshold by this value:
#define THRESHOLD_INCREASE 0.01

bool kbhit (void);          // check whether key was pressed
int tty_cbreak (int fd);    // put tty in cbreak (raw, !cooked) mode
int tty_reset (int fd);     // restore terminal's mode
static struct termios save_stdin;

extern void setup_spline_matrices();
#define USAGE "usage: data_extraction [options] \n\
-l run Length for median filter \n\
-t differencing Threshold \n\
-n No. of control points \n\
-i input options (-i help for details) \n\
-d don't align/ normalise shapes \n\
-g gaps size for merging regions \n\
-v vertical flip images \n\
-m minimum blob size as proportion of image area (e.g. 0.06) \n\
-p parameter_file (use file to specify parameters)\n"

/* fixed globals */

//  bool quiet_mode = 
//  Configuration.register_bool("QUIET_MODE", false,
//  		      &quiet_mode, true,
//  		      "main", 
//  		      "If set ON then don't display any images");

int& default_ncp = 
Configuration.register_int("NO_CONTROL_POINTS", 32,
		     &Profile::NO_CONTROL_POINTS, true,
		     "main", 
		     "The number of spline control points used to \
represent each shape");

static bool do_background_updating =
Configuration.register_bool("UPDATE_BACKGROUND",
		      true, &do_background_updating,
		      true, "main",
		      "Update background, using a median filter ?");

int ...should be frame_id_t ... and rename to background_update_skip ...  time_skip =
Configuration.register_int("BACKGROUND_UPDATE_SKIP", 1,
		     &time_skip, true,
		     "main",
		     "Temporal resampling for background filtering");

int default_window_length=
Configuration.register_int("BACKGROUND_RUN_LENGTH", 50,
		     &default_window_length, true,
		     "main",
		     "Median window length (in frames) \
for background image update");

bool show_background_images = 
Configuration.register_bool("SHOW_BACKGROUND_IMAGES",
		      true, &show_background_images,
		      true, "main",
		      "Show the generated background images ?");

bool show_difference_images = 
Configuration.register_bool("SHOW_DIFFERENCE_IMAGES",
		      false, &show_difference_images,
		      false, "main",
		      "Show the differenced images foreground-background ?");

bool show_thresholded_difference_images = 
Configuration.register_bool("SHOW_THRESHOLDED_DIFF_IMAGES",
		      false, &show_thresholded_difference_images,
		      false, "main",
		      "Show the thresholded difference foreground-background ?");

bool show_motion_images = 
Configuration.register_bool("SHOW_MOTION_IMAGES",
		      true, &show_motion_images,
		      false, "main",
		      "Show the thresholded and filtered motion detection images ?");

bool show_filtered_difference_images = 
Configuration.register_bool("SHOW_FILTERED_DIFFERENCE_IMAGES",
		      true, &show_filtered_difference_images,
		      true, "main",
		      "Show the filtered difference images fore-background?");


bool do_normalisation = 
Configuration.register_bool("NORMALISE",
		      true, &do_normalisation,
		      true, "main",
		      "Normalise profiles ? (i.e. recenter using centroid
and rotate principal axis to vertical");

bool flip_images = 
Configuration.register_bool("FLIP_VERTICALLY",
		      false, &flip_images,
		      true, "main",
		      "Vertically flip input images ?");

bool default_do_blur = 
Configuration.register_bool("BLUR_MOTION",
		      false, &default_do_blur,
		      true, "main",
		      "Blur images in motion detection (slow) ?");

static bool skip_non_existing =
Configuration.register_bool("SKIP_NON_EXISTING",
		      true, &skip_non_existing,
		      true, "main",
		      "Shall we tolerate (skip) non-existing input files ? \
Useful when you have 0000.jpg, 0007.jpg etc");

int max_skip_non_existing =
Configuration.register_int("MAX_SKIP_NON_EXISTING", 1024,
		     &max_skip_non_existing, true,
		     "main",
		     "Maximum number of non-existing input files to \
tolerate (skip) before assuming end of sequence");

static bool new_background_after_skip =
Configuration.register_bool("NEW_BACKGROUND_AFTER_SKIP",
		      true, &new_background_after_skip,
		      true, "main",
		      "After skipping non-existing input files, take 1st valid \
image as new background");


/* variable globals */

bool use_border_regions = 
Configuration.register_bool("USE_BORDER_REGIONS",
		      true, &use_border_regions,
		      false, "main",
		      "Use regions near the image border ?");

bool reduce_noise = 
Configuration.register_bool("REDUCE_NOISE",
		      true, &reduce_noise,
		      false, "main",
		      "Use dilation to reduce noise (slow) ?");

int gap_size = 
Configuration.register_int("GAP_SIZE", 1,
		     &gap_size, false, "main",
		     "The dilation size in pixels when filtering binary motion
image.  The effect is to join blobs that are this many pixels apart.");


//  realno camera_moved_threshold = 
//  Configuration.register_real("CAMERA_MOVED_THRESH", 0.5,
//  		      0.0, 1.0,
//  		      &camera_moved_threshold,
//  		      false, "main", 
//  		      "If the number of pixels where there is movement over
//  the total number of pixels is greater than this threshold, then the camera
//  has moved.");

realno diff_threshold = 
Configuration.register_real("DETECT_DIFF_THRESH", 0.1,
		      0.0, 1.0, &diff_threshold, 
		      false, "main",
		      "Differencing threshold for object detection");


realno max_region_ratio =
Configuration.register_real("MAX_REGION_SIZE", 0.5,
		      &max_region_ratio,
		      false, "main",
		      "Maximum size of a motion blob over size of image");

realno min_region_ratio = 
Configuration.register_real("MIN_REGION_SIZE", 0.001,
		      &min_region_ratio,
		      false, "main",
		      "Minimum size of a motion blob over size of image");


int main (int argc, char** argv)
{
    int c;
    int runlength;  // background run length
    
    ImageSource *input;

    // pipe sources
    PipeSource *foreground;
    PipeSource *blur_foreground;
    PipeSource *background;
    PipeSource *difference;
    PipeSource *thresholded_difference;
   
    // (colour) filtered sources
    PipeSource *filtered_input;
    PipeSource *filtered_background;
    PipeSource *filtered_difference;    
    
    Profile::draw_boxes = true;
    char *parameter_file = NULL;
    
    while ((c = getopt(argc, argv, "p:m:i:dg:vl:t:n:?")) != EOF)
	switch(c)
	{
//	case 'q':
//	    quiet_mode = true;
//	    break;
	    
	case 'p':
	    parameter_file = optarg;
	    break;
	    
	case 'm':
	    if (optarg != NULL)
		min_region_ratio = atof(optarg);
	    break;
	    
	case 'i':
	    input = get_live_or_movie_feed(optarg);
	    break;
	    
	case 'd':
	    do_normalisation = false;
	    break;

	case 'g':
	    reduce_noise = true;
	    if (optarg != NULL)
		gap_size = atoi(optarg);
	    break;

	case 'v':
	    flip_images = true;
	    break;

	case 'l':
	    if (optarg != NULL)
		default_window_length = atoi(optarg);
	    break;

	case 't':
	    if (optarg != NULL)
		diff_threshold = atof(optarg);
	    break;

	case 'n':
	    if (optarg != NULL)
		Profile::NO_CONTROL_POINTS = atoi(optarg);
	    break;

	case '?':
	default:
	    fprintf(stderr, USAGE);
	    exit(0);
	} 

    Configuration.parse_parameter_file(parameter_file);

// nts: we are always interactive   if (!quiet_mode)

    // set default icon for windows...
    Configuration.set_icon(data_extraction_icon_16);
    Configuration.create_interface(argc, argv);
  
    
    setup_spline_matrices();

    Image* reference_image = get_reference_frame();

    if (flip_images)
    {
	ImageSource* old_in = input;
	input = new VFlipSource(old_in);
	if (reference_image != NULL)
	    reference_image->flip_vertically(reference_image);
    }

    //  Set up background, either constant or medium-filtered:
    if (do_background_updating)
    {
	runlength = default_window_length;//    / time_skip;
	background = new MedianFilterSource(input, runlength, reference_image);
    }
    else
    {
	// set up constant background image
	runlength = INT_MAX;
	if (reference_image == NULL)
	{
	    fprintf(stderr, " Cannot use NULL background image and not update it. \n");
	    exit(1);
	}    
	background = new ConstantSource(reference_image);
    }
    
    filtered_background = new ColourFilterSource(background);
    filtered_input = new ColourFilterSource(input);
    difference = new DifferenceSource(input, background);

// 2 alternatives here:
//    filtered_difference = new ColourFilterSource(difference,CF_METHOD_Y_709);
    filtered_difference = new DifferenceSource
	(filtered_input,filtered_background);

    thresholded_difference = new ThresholdSource
	(filtered_difference, 255 * diff_threshold);

    if (default_do_blur)  // blur foreground?
    {
	blur_foreground = new BlurSource(thresholded_difference);
	foreground = new ThresholdSource
	    (blur_foreground, 255.0 * diff_threshold);
    }
    else
    {
	foreground = new NeighbourSource(thresholded_difference, 4);
	// foreground = new PipeSource(thresholded_difference);
    }
    
    foreground->set_title("Detected Foreground");
   
//    if (do_background_updating)
//	background->get_next();  // this calls input->get_next()
//    else
//	input->get_next();      // ... which we need, anyway

     // set up Window titles...
    if (do_background_updating)
	background->set_title("Median-filtered Background");
    else
	background->set_title("Constant Background");
    
    filtered_background->set_title("Filtered Background");
    filtered_input->set_title("Filtered Input");
    difference->set_title("Difference Fore-Background");
    filtered_difference->set_title("Filtered Difference");
    thresholded_difference->set_title("Thresholded Difference");
    input->set_title("Video Input");
    blur_foreground->set_title("Blurred Foreground");
    foreground->set_title("Foreground");
    
    RegionSet curr_regions;
    ProfileSet curr_profiles;

    unsigned int vwidth = input->get_xdim();
    unsigned int vheight = input->get_ydim();
    unsigned int vsize = vwidth * vheight;

    fprintf(stderr, " and ACTION! \n");
    

/////////////////////////////////////////////////////////////////////////
//                                                                     //
//       MAIN LOOP                                                     //   
//                                                                     //
/////////////////////////////////////////////////////////////////////////    

//    bool found_some = false;
    bool finished = false;         // whether user wishes to exit program
    bool use_profiles = false;     // user says extracted profiles are to be used
    bool frame_processed = false;  // whether current frame is processed
    Image* check_image;            // for checking availability of next image
// unused    Point2 object_center;
// unused    bool flag_center = false;
    frame_id_t frame_id;
    frame_id_t number_of_tries;  // counting #tries to get next frame
    frame_id_t iteration_count;  // how often user asks to re-process frame
    unsigned int min_region_size;
    unsigned int max_region_size;

    int ch;  // input character read from the terminal

    if (skip_non_existing == false)    // we should not tolerate (skip) non-existing files
	max_skip_non_existing = 0;

    tty_cbreak (STDIN_FILENO);  // put terminal into cbreak (raw) mode

    cout << endl << Profile::NO_CONTROL_POINTS << " control points" << endl;
    
    for (frame_id = input->get_frame_id(); finished == false;
	 frame_id = input->get_frame_id())
    {
	//
	//  New interactive approach to improve detection quality:
	//
	//  For each frame, we do an inner loop until we have found
	//  something and detected it properly (user presses Enter)
	//  or we or the user are sure that there is nothing.
	//
	//  Frames may be skipped; more precisely: we allow for numbered
	//  input files to be named 0000.jpg, 0005.jpeg, 0100.jpeg etc.
	//  The maximum difference is determined by the MAX_SKIP_NON_EXISTING variable.
	//  This skipping can be dis/enabled using the SKIP_NON_EXISTING variable.
	//
	//  User interaction output is written to stderr so stdout only
	//  contains detection data.
	//

	//////////////////////////////////////////////////////
	//                                                  //
	//       step 1 - get next input frame              //
	//                                                  //
	//////////////////////////////////////////////////////

	number_of_tries = 0;
	do
	{
//	    if ((do_background_updating) && (frame_id % time_skip) == 0)
//		check_image = background->get_next();  // calls input->get_next()
//	    else
//		check_image = input->get_next();       // ... which we need

	    check_image = foreground->get_next();

	    if ((number_of_tries++) > 1)
	    {
		fprintf(stderr,"\r Scanning input for next image ... %5i",
		       number_of_tries);
		fflush(stderr);
	    }
	    
	    if (kbhit ())
		do
		{
		    switch (ch=getchar())
		    {
		    case 'q':  // let's exit on these characters
		    case 'Q':
//		    case 'x':
//		    case 'X':
		    case 0x1b: //  (Esc)
			finished = true;
			break;
		    default:
			// silently ignore all other input
			break;
		    }
		}
		while (kbhit ());  // process all key strokes

	    frame_id++;
	}
	while ((check_image == NULL) &&          // while we don't get an image
	       (number_of_tries <= max_skip_non_existing) &&  // and we don't skip too many
	       (! finished));                    // and user does not want to exit

	if (number_of_tries > 2)
	    fprintf(stderr,"\r                                         \r");

	if (check_image == NULL) // we have an exit condition
	{
	    if (number_of_tries > max_skip_non_existing)
	    {
		fprintf(stderr,"\n %i non-existing files in a row while looking\
 for next input image. \n Assuming end of sequence. \n",number_of_tries);
		exit(0);
	    }
	    else  //  <=> (finished) ie user pressed Esc or 'q' 
	    {
		fprintf(stderr,"\n Exit on user request \n");
		exit(2);
	    }
	}

	// now we have a valid current input image.

	// if we skipped non-existing images we might want to restart bg
	if ((number_of_tries > 1) && (new_background_after_skip))
	{
	    //if (do_background_updating)
	    input->get_current()->copy(background->get_current());
#ifdef DEBUG
	    fprintf(stderr," Images skipped, therefore background restarted in frame %i \n",frame_id);
#endif
	}


	//////////////////////////////////////////////////////
	//                                                  //
	//       step 2 - process current frame             //
	//                                                  //
	//////////////////////////////////////////////////////

	frame_processed = false;	
	for (iteration_count = 0;
	     ((finished == false) && (frame_processed == false));
	     iteration_count++)
	{
	    use_profiles = false;     // profiles were not accepted, yet.

	    if (iteration_count > 0)  // we were asked to re-process frame
	    {
		curr_profiles.destroy_all();
		curr_regions.destroy_all();
	    }	    

	    min_region_size = (unsigned int) (vsize * min_region_ratio);
	    max_region_size = (unsigned int) (vsize * max_region_ratio); 
	    
	    // re-set threshold as it might have been changed online
	    ((ThresholdSource*) thresholded_difference)
		-> threshold = (unsigned char) (255.0 * diff_threshold);
	    
	    filtered_input->refresh();
	    filtered_background->refresh();	
	    
//	((ThresholdSource*) foreground)
//	    -> threshold = (unsigned char) (255.0 * diff_threshold);
	    
	    difference->refresh();
	    filtered_difference->refresh();
	    thresholded_difference->refresh();
	    
//	foreground->get_next();
	    
	    if (reduce_noise == true)   // do dilation?
	    {
		if (gap_size > 0)
		{
		    Grey8Image* filled_foreground = (Grey8Image*)
			foreground->get_current();
		    
		    filled_foreground->fix_holes(gap_size, filled_foreground);
		}
	    }
	    
	    if (show_background_images)
		background->display();
	    
	    if (show_difference_images)
		difference->display();
	    
	    if (show_filtered_difference_images)
		filtered_difference->display();
	    
	    if (show_thresholded_difference_images)
		thresholded_difference->display();
	    
	    if (show_motion_images)
		foreground->display();
	    
	    input->display();
	    input->get_current()->draw_in_image();
	    
	    curr_regions.grab_regions(foreground->get_current(),
				      min_region_size, max_region_size,
				      use_border_regions);
	    
	    curr_regions.trace_regions();
	    curr_profiles = curr_regions.to_profiles();
	    
	    curr_profiles.draw();

	    gflush();
	    
	    // displayed current profile.  Now ask user what to do.

// FIXME: maybe more than one profile in image

	    fprintf(stderr,"\r `+' and `-' to adjust threshold.  Enter: accept. Space: ignore.  Q: quit.");
	    fflush(stderr);

	    // we don't have an event handler for our windows :-(
	    while (!(kbhit ())) // make sure displays are updated
	    {
		unsigned int delay;
		for (delay = 0; delay < 25; delay++)
		{
		    usleep(20*1000);  // wait 20 msec
		    if (kbhit ())
			break;
		}
		if (delay > 24) // we have waited 500 msec for input
		{
		    // refresh windows:
		    if (show_background_images)
			background->display();
		    
		    if (show_difference_images)
			difference->display();
		    
		    if (show_filtered_difference_images)
			filtered_difference->display();
		    
		    if (show_thresholded_difference_images)
			thresholded_difference->display();
		    
		    thresholded_difference->refresh();
		    foreground->refresh();

		    if (reduce_noise == true)   // do dilation?
		    {
			if (gap_size > 0)
			{
			    Grey8Image* filled_foreground = (Grey8Image*)
				foreground->get_current();
			    
			    filled_foreground->fix_holes(gap_size, filled_foreground);
			}
		    }

		    if (show_motion_images)
			foreground->display();
		    
		    input->display();
		    input->get_current()->draw_in_image();
		    curr_profiles.draw();
		    
		    gflush();
		}
	    }
		
	    switch (ch=getchar())
	    {
	    case '\n':  // Enter: profiles accepted as displayed.
		frame_processed = true;
		use_profiles = true;
		break;
		
	    case ' ':   // Space: Nothing usable in this frame
		frame_processed = true;
		use_profiles = false;
		break;

	    case '+':   // increase detection threshold
		diff_threshold += THRESHOLD_INCREASE;
		if (diff_threshold > 1.0)
		    diff_threshold = 1.0;
		break;

	    case '-':   // decrease detection threshold
		diff_threshold -= THRESHOLD_INCREASE;
		if (diff_threshold < 0.0)
		    diff_threshold = 0.0;
		break;
		
	    case 'q':   // let's exit on these characters
	    case 'Q':
//	    case 'x':
//	    case 'X':
//	    case 0x1b:  //  (Esc)
		finished = true;
		break;
	    default:
		// silently ignore all other input
		break;
	    }
	    fprintf(stderr,"\r                                                                                     \r");
	    fflush(stderr);
	}
	
	if (finished == true)  //  user wishes to exit program
	{
	    fprintf(stderr,"\n Exit on user request \n");
	    exit(2);
	}

	if ((curr_profiles.no_items > 0) && (use_profiles))
	{
	    if (do_normalisation)
		curr_profiles.normalise();
	    else
		curr_profiles.add_origins();

	    cout << "***  frame " << frame_id << "  ***" << endl
		 << curr_profiles << "********************" << endl;
	}
	curr_profiles.destroy_all();
	curr_regions.destroy_all();
    }
    return 0;
}


// poll the keyboard for a character
// idea from The Linux Journal article of Sept 95 issue 17
bool kbhit (void)
{
    struct timeval tv;
    fd_set read_fd;

    // do not wait at all, not even a microsecond
    tv.tv_sec = 0;
    tv.tv_usec = 0;

    // must be done first to initilize read_fd
    FD_ZERO (&read_fd);

    // makes select() ask if input is ready: 0 is the file descriptor stdin
    FD_SET (0, &read_fd);

    // the first parameter is the number of the largest file descriptor to check+1

    if (select (1, &read_fd,
		NULL,             // NO writes
		NULL,             // NO exceptions
		&tv)
	== -1)
	return false;                   // An error occurred

    // read_fd now holds a bitmap of files that are readable.
    // We test the entry for the standardinput (file 0).

    if (FD_ISSET (0, &read_fd)) // character pending on stdin
	return true;

    // no charcaters were pending

    return false;
}


// from W Richard Steven's book
// ``Advanced Programming in the Unix Environment''

int tty_cbreak (int fd)             // put terminal into a cbreak mode
{
    struct termios buf;

    if (fd == STDIN_FILENO)
    {
	if (tcgetattr (fd, &save_stdin) < 0)
	    return -1;
    }
    buf = save_stdin;             // structure copy

    buf.c_lflag &= ~(ECHO | ICANON);
    // echo off, canonical mode off

    buf.c_cc[VMIN] = 1;           // Case B: 1 byte at a time, no timer
    buf.c_cc[VTIME] = 0;

    if (tcsetattr (fd, TCSAFLUSH, &buf) < 0)
	return -1;
    return 0;
}


int tty_reset (int fd)              // restore terminal's mode
{
    if (fd == STDIN_FILENO)
    {
	if (tcsetattr (fd, TCSAFLUSH, &save_stdin) < 0)
	    fprintf (stderr, "\n Failed to reset stdin \n");
	return -1;
    }
    return 0;
}

// Local Variables:
// compile-command: "cd ..; make; cd progs; make data_extraction"
// End:
