// Program to convert a binary PCA model file between platforms which
//   have different byte order (LSB_FIRST vs MSB_FIRST)
// created by nts in 2001

#include <iostream>
#include <fstream>
#include <sys/types.h>  // the for ISO C u_int32_t type (32-bit unsigned int)
#include <getopt.h>
// #include <stdlib.h>

#include "realno.h"
#include "text_output.h"

using namespace ReadingPeopleTracker;

#define USAGE " [ -m MSB_first_pca_file | -l LSB_first_pca_file ] -o ascii_output_file \n\n\
Please give either MSB first (-m, for non-x86) or LSB first (-l, for x86) input file. \n"

union realno_or_eightbytes
{
    u_int32_t four_bytes[2];  // either interpret as 2 x four byte words
    realno realno_type;       // or as a realno which is an 8 byte double
};

// function to read four bytes, convert (swap) if necessary and write them out
void read_convert_write(const unsigned int number_of_realnos);

// function to print USAGE and exit(1)
void usage_and_exit();

// function to call if something unexpected comes up in the file, exit with a message
void panic();


bool swap_bytes;     // whether to swap bytes of machine words on input...
ifstream data_in;
ofstream data_out;
char *program_name;  // our name

int main(int argc, char** argv)
{
    program_name = argv[0];
    char *input_file = NULL;
    char *output_file = NULL;
    bool MSB_first_input = false;
    signed char c;
    
    while ((c = getopt(argc, argv, "m:l:o:?")) != EOF)
	switch(c)
	{
	case 'm':
	    if ((optarg != NULL) && (strlen(optarg) > 0))
	    {
		input_file = optarg;
		MSB_first_input = true;
		break;
	    }
	    else
		usage_and_exit();
	case 'l':
	    if ((optarg != NULL) && (strlen(optarg) > 0))
	    {
		input_file = optarg;
		MSB_first_input = false;
		break;
	    }
	    else
		usage_and_exit();
	case 'o':
	    if ((optarg != NULL) && (strlen(optarg) > 0))
		output_file = optarg;
	    break;
	case '?':
	default:	
	    usage_and_exit();
	}    

    if ((input_file == NULL) || (output_file == NULL))
    {
	usage_and_exit();
    }
    
    data_in.open(input_file, ios::in | ios::binary);
    data_out.open(output_file, ios::out);

    //
    //  technical stuff ...
    //

    // whether to swap bytes of machine words on input...
#ifdef LSB_FIRST
    swap_bytes = MSB_first_input;
#else
#ifdef MSB_FIRST
    if (MSB_first_input)
	swap_bytes = false;
    else
	swap_bytes = true;
#else
#error "Please define LSB_FIRST or MSB_FIRST in generik.mk"
#endif  // ifdef LSB_FIRST
#endif  // ifdef MSB_FIRST

    if (swap_bytes == false)
	cdebug << " Not";
    
    cdebug << " swapping bytes... " << endl;


    // check whether the realno type uses 8 bytes...
    if (sizeof(realno) != 8)
    {
	cerror << " Error in type: expecting 8 byte `realno's (not "
	       << sizeof(realno) << " byte). " << endl;
	exit(1);
    }

    //
    //  variables used to read and write files
    //
    unsigned int matrix_dim_x;
    unsigned int matrix_dim_y;
    char text[4096];

    unsigned int no_processed;
    unsigned int data_size;
    unsigned int no_elements;
    unsigned int count;

    
    /////////////////////////////////////////////////////////////////////
    //                                                                 //
    //   now we can read the file, reading `realno's into four_bytes   //
    //   first to swap bytes, if necessary                             //
    //                                                                 //
    /////////////////////////////////////////////////////////////////////

    //
    //  1 -- header.  It is plain text so no conversion here...
    //
    data_in >> text;
    if (strcmp(text,"PCA_MODEL") != 0)
	panic();
    data_out << "PCA_MODEL" << endl;
    
    data_in >> text >> no_processed;
    if (strcmp(text,"no_processed") != 0)
	panic();
    data_out << " no_processed " << no_processed << endl;

    data_in >> text >> data_size;
    if ((strcmp(text,"data_size") != 0) ||
	((data_size & 1) != 0))
	// only accept even number because they are 2-D coordinates
	panic();
    data_out << " data_size " << data_size << endl;

    //
    // 2 -- data.  These go like "Eigenvectors 64 by 64 matrix\n"
    //             "4096 d[\n" DATA "]\n"
    //             with DATA being 4096 binary `realno's
    //

    // Covariance matrix
    data_in >> text;
    if (strcmp(text,"Covariance_Matrix") != 0)
	panic();

    data_in >> matrix_dim_y >> text >> matrix_dim_x;
    if ((strcmp(text,"by") != 0) ||
	(matrix_dim_x != data_size) ||
	(matrix_dim_y != data_size))
	panic();
    
    data_in >> text;
    if (strcmp(text,"matrix") != 0)
	panic();
    data_out << " Covariance_Matrix " << data_size
	     << " by " << data_size << " matrix" << endl;

    data_in >> no_elements >> text;
    if ((no_elements != (data_size * data_size)) ||
	(strcmp(text,"d[") != 0))   // NAG's way to introduce matrix data
	panic();
    
    data_in.read(&text,1);  // clean newline character from input stream
    if (text[0] != '\n')
	panic();

    data_out << no_elements << " [" << endl;    // NOTE: no "d" before "[" here

    //  process binary matrix data.  convert if necessary.
    read_convert_write(no_elements);

    data_in >> text;
    if (strcmp(text,"]") != 0)   // NAG's way to end matrix data
	panic();

    data_out << "]" << endl;
    
    // Mean vector
    data_in >> text;
    if (strcmp(text,"Mean") != 0)
	panic();

    data_in >> no_elements >> text;
    if ((no_elements != data_size) ||
	(strcmp(text,"d[") != 0))   // NAG's way to introduce matrix data
	panic();

    data_in.read(&text,1);  // clean newline character from input stream
    if (text[0] != '\n')
	panic();

    data_out << " Mean " << no_elements << " [" << endl;  // no "d" here

    //  process binary matrix data.  convert if necessary.
    read_convert_write(no_elements);

    data_in >> text;
    if (strcmp(text,"]") != 0)   // NAG's way to end matrix data
	panic();

    data_out << "]" << endl;

    // Weights vector
    data_in >> text;
    if (strcmp(text,"Weights") != 0)
	panic();

    data_in >> no_elements >> text;
    if ((no_elements != (data_size / 2)) ||
	(strcmp(text,"d[") != 0))   // NAG's way to introduce matrix data
	panic();

    data_in.read(&text,1);  // clean newline character from input stream
    if (text[0] != '\n')
	panic();

    data_out << " Weights " << no_elements << " [" << endl;  // no "d" here

    //  process binary matrix data.  convert if necessary.
    read_convert_write(no_elements);

    data_in >> text;
    if (strcmp(text,"]") != 0)   // NAG's way to end matrix data
	panic();

    data_out << "]" << endl;

    // Eigenvalues vector
    data_in >> text;
    if (strcmp(text,"Eigenvalues") != 0)
	panic();

    data_in >> no_elements >> text;
    if ((no_elements != data_size) ||
	(strcmp(text,"d[") != 0))   // NAG's way to introduce matrix data
	panic();

    data_in.read(&text,1);  // clean newline character from input stream
    if (text[0] != '\n')
	panic();

    data_out << " Eigenvalues " << no_elements << " [" << endl;  // no "d" here

    //  process binary matrix data.  convert if necessary.
    read_convert_write(no_elements);

    data_in >> text;
    if (strcmp(text,"]") != 0)   // NAG's way to end matrix data
	panic();

    data_out << "]" << endl;

    // Eigenvectors matrix
    data_in >> text;
    if (strcmp(text,"Eigenvectors") != 0)
	panic();

    data_in >> matrix_dim_y >> text >> matrix_dim_x;
    if ((strcmp(text,"by") != 0) ||
	(matrix_dim_x != data_size) ||
	(matrix_dim_y != data_size))
	panic();
    
    data_in >> text;
    if (strcmp(text,"matrix") != 0)
	panic();

    data_in >> no_elements >> text;
 
    if ((no_elements != (data_size * data_size)) ||
	(strcmp(text,"d[") != 0))   // NAG's way to introduce matrix data
	panic();
  
    data_in.read(&text,1);  // clean newline character from input stream
    if (text[0] != '\n')
	panic();

    data_out << " Eigenvectors "
	     << data_size << " by " << data_size << " matrix" << endl
	     << no_elements << " [" << endl;

    //  process binary matrix data.  convert if necessary.
    read_convert_write(no_elements);

    data_in >> text;
    if (strcmp(text,"]") != 0)   // NAG's way to end matrix data
	panic();

    data_out << "]" << endl;
  
    // OPTIONAL Noise_Model vector
    if (data_in.peek() != EOF)
	data_in >> text;  // try to read introductory text...

    if (data_in.peek() != EOF)  // test again to ignore whitespace just before EOF
    {
	if (strcmp(text,"Noise_Model") != 0)
	    panic();

	data_in >> no_elements >> text;
	if ((no_elements != data_size) ||
	    (strcmp(text,"d[") != 0))   // NAG's way to introduce matrix data
	    panic();

	data_in.read(&text,1);  // clean newline character from input stream
	if (text[0] != '\n')
	    panic();
	
	
	data_out << " Noise_Model " << no_elements << " [" << endl;  // no "d"
	
	//  process binary matrix data.  convert if necessary.
	read_convert_write(no_elements);
	
	data_in >> text;
	if (strcmp(text,"]") != 0)   // NAG's way to end matrix data
	    panic();
	
	data_out << "]" << endl << endl;
    }
    
    data_in.close();
    data_out.close();

    return 0;
}

void panic()  // something unexpected in the file, exit
{
    cerror << program_name << " Unexpected data format in input file.  I'd better exit. "
	   << endl;
    exit(1);
}

void usage_and_exit()
{
    cerror << program_name << ": Wrong parameters.  Usage: " << endl
	   << program_name << USAGE
	   << endl;
    exit(1);
}

inline void read_convert_write(const unsigned int number_of_realnos)
{
    union realno_or_eightbytes result;
    unsigned int count;
    
    for (count = 0; count < number_of_realnos; count++)
    {
	//
	// we read data in 32-bit chunks as we probably have a 32-bit CPU
	//
	// first package of four bytes...
	if (swap_bytes)
	{
	    data_in.read((char *) &(result.four_bytes[1]),4);
	    
	    register u_int32_t AX;
	    register u_int32_t DX;
	    AX = result.four_bytes[1];
	    DX = AX << 8;
	    AX = AX >> 8;
	    DX = DX | (AX & 0x000000ff);  // FIXME: OK on MSB?
	    AX = AX >> 8;
	    DX = DX << 8;
	    DX = DX | (AX & 0x000000ff);
	    AX = AX >> 8;
	    DX = DX << 8;
	    DX = DX | (AX & 0x000000ff);
	    //four_bytes0 = DX;
	    result.four_bytes[1] = DX;
	}
	else
	    data_in.read((char *) &(result.four_bytes[0]),4);

	// second package of four bytes...
	if (swap_bytes)
	{
	    data_in.read((char *) &(result.four_bytes[0]),4);
	    
	    register u_int32_t AX;
	    register u_int32_t DX;
	    AX = result.four_bytes[0];
	    DX = AX << 8;
	    AX = AX >> 8;
	    DX = DX | (AX & 0x000000ff);  // FIXME: OK on MSB?
	    AX = AX >> 8;
	    DX = DX << 8;
	    DX = DX | (AX & 0x000000ff);
	    AX = AX >> 8;
	    DX = DX << 8;
	    DX = DX | (AX & 0x000000ff);
	    //four_bytes1 = DX;
	    result.four_bytes[0] = DX;
	}
	else
	    data_in.read((char *) &(result.four_bytes[1]),4);
	
	data_out << result.realno_type << endl;
    }    
}

// Local Variables:
// compile-command: "make convert_model |& fgrep :"
// End:
