///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  FILENAME : IEEE1394_Capture.cc                                           //
//  AUTHOR   : Nils T Siebel                                                 //
//  CREATED  : Mon Sep 30 16:03:37 BST 2002                                  //
//  COPYRIGHT: Nils T Siebel                                                 //
//                                                                           //
//  PURPOSE :  capture and store images using the IEEE1394Source class       //     
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <cstdio>    // for cin, sprintf
#include <cstring>   // for strcasecmp
#include <cctype>    // for toupper
#include <pthread.h> // for IEEE POSIX 1003.1c --- 1995 threads

// #undef DEBUG
#include "IEEE1394Source.h"
#include "text_output.h"
#include "Image.h"         // for drawing functions

using namespace ReadingPeopleTracker;

// POSIX threading stuff
bool enable_video_capture;
pthread_mutex_t enable_video_capture_modification = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t enable_video_capture_has_changed = PTHREAD_COND_INITIALIZER;

IEEE1394Source *source;
Image *image;

// clear input buffer of cin
void clear_cin()
{
    int buffer_size = cin.rdbuf()->in_avail();

    for (buffer_size = cin.rdbuf()->in_avail(); buffer_size > 0; buffer_size--)
	cin.get();
}


void *capture_video(void *unused)
{
    for ( ; ; )
    {
	if (enable_video_capture == false)
	{
            pthread_mutex_lock(&enable_video_capture_modification);
            //
            while (enable_video_capture == false)
            {
                pthread_cond_wait(&enable_video_capture_has_changed,
                                  &enable_video_capture_modification);
            }
            //
            pthread_mutex_unlock(&enable_video_capture_modification);
	}

	image = source->get_next();
	image->display();

  	// draw vertical lines where cropping will occur (for portrait images)
	//   we are aiming at 432x576 images, cropping from 720x576
	image->draw_in_image();

	linewidth(4);
	RGBcolor(255,0,40);
	move2i((720-432) / 2 - 2, 9);
	draw2((720-432) / 2 - 2, 576 - 10);
	
	move2i(432 + (720-432) / 2 + 1, 9);
	draw2(432 + (720-432) / 2 + 1, 576 - 10);
	
	gflush();
   }
}


int main()
{
    cinfo << " IEEE1394_Capture Version 1.4, part of the Reading People Tracker" << endl
	  << " Program for capturing people's images using an IEEE1394 digital camera" << endl
	  << " Author: Nils T Siebel. " << endl
	  << " Copyright: (c) 2003-2005 Nils T Siebel. " << endl
	  << " Please see the file COPYRIGHT for details. " << endl
	  << endl;
    
    char person_name[1024];
    unsigned long frame_count;
    unsigned long sequence_number;
    char dummy_string[1024];
    char user_response;
    char image_filename[1024 + 512];
    
    source = new IEEE1394Source(DV_QUALITY_BEST);

    pthread_t thread_id;
    int result = pthread_create(&thread_id, NULL, &capture_video, NULL);

    if (result != 0)
    {
        // an error occurred
        cerror << " Error: could not start thread for video capturing, "
	       << " error code " << result << ". " << endl;
	exit(1);
    }

    cinfo << " Please enter the sequence number.  This should start at 1, unless " << endl
	  << " you have used the program before.  In that case please enter the  " << endl
	  << " last valid sequence number plus 1 : ";
    cin >> sequence_number;

    enable_video_capture = true;
    pthread_cond_signal(&enable_video_capture_has_changed);

    // loop more or less forever: get person data and frame, write it to disk.
    for (frame_count = 1; frame_count > 0; frame_count++)
    {
	bool name_entered_correctly = false;
	
	while (name_entered_correctly == false)
	{
	    cinfo << " Please enter the person's name or exit : ";
	    
	    clear_cin();
	    cin.getline(person_name, 1023);
	    
	    if (strcasecmp(person_name,"exit") == 0)
		break;
	    
	    cinfo << " Sequence number " << sequence_number << ", "
		  << " person's name " << person_name << " " << endl;
	    
	    user_response = '\0';
	    
	    while ((toupper(user_response) != 'N') &&
		   (toupper(user_response) != 'Y'))
	    {
		cinfo << " Are the above details correct? (Y/N) : ";
		cin >> user_response;
	    }
	    
	    name_entered_correctly = (toupper(user_response) == 'Y');
	}

	// check for exit request
	if (strcasecmp(person_name,"exit") == 0)
	    break;

	// now we have valid sequence number and name.
	// capture a frame:

	bool image_ok = false;

	while (image_ok == false)
	{
	    cinfo << endl
		  << " Please press enter to capture " << person_name << "'s image. ";

	    clear_cin();
	    
	    // get person's image
	    
	    // the following will be read by our display thread
	    enable_video_capture = true;
	    pthread_cond_signal(&enable_video_capture_has_changed);

	    // wait for enter
	    cin.getline(dummy_string, 1023);
	    
	    // stop capturing
	    enable_video_capture = false;
	    pthread_cond_signal(&enable_video_capture_has_changed);

	    // ask whether image is ok with everyone
	    user_response = '\0';
	    
	    while ((toupper(user_response) != 'N') &&
		   (toupper(user_response) != 'Y'))
	    {
		cinfo << " Is everybody happy with the image ? (Y/N) : ";
		cin >> user_response;
	    }
	    
	    image_ok = (toupper(user_response) == 'Y');
	}
	
	// image ok, write it to disk

	// create filename
	sprintf(image_filename,"%04lu_%s.ppm",sequence_number,person_name);
	
	// fix invalid characters
	unsigned int index;
	for (index = 0; index < strlen(image_filename); index++)
	    if ((image_filename[index] == '\\') ||
		(image_filename[index] == ':') ||
		(image_filename[index] == ';') ||
		(image_filename[index] == '`') ||
		(image_filename[index] == '"') ||
		(image_filename[index] < 32))
		image_filename[index] = '_';
	    else
		if (image_filename[index] == '\'')
		    image_filename[index] = (unsigned char) '';  // ascii 0xb4 looks very similar and does not quote
		else
		    if (image_filename[index] == '/')
			image_filename[index] = ',';  // avoid directory slash, use comma
	
	
	// save file as pnm
	image->save_pnm(image_filename);
	
	cinfo << endl
	      << " Next person please... " << endl
	      << endl;
	
	sequence_number++;
    }


    // stop capturing
    enable_video_capture = false;
    pthread_cond_signal(&enable_video_capture_has_changed);
    
    // give thread some time to finish
    // FIXME: should put another condition variable
    usleep(500*1000);
    
    delete source;
    
    return 0;  
}

// Local Variables:
// compile-command: "make IEEE1394_Capture |& fgrep :"
// End:

