////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//  Calibration_test.cc                                                       //
//                                                                            //
//  Test program for the Calibration class, using GL/Ygl                      //
//                                                                            //
//  Author    : Rmi Belin (rb)  supervised by  Nils T Siebel (nts)           //
//  Created   : Wed Jul 18 13:50:21 BST 2001                                  //
//  Revision  : 1.0 of Wed Aug  8 15:08:33 BST 2001                           //
//  Copyright : The University of Reading                                     //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include "Calibration.h"

#include <gl/gl.h>
#include <gl/device.h>

#include "ImageSource.h"
#include "JPEGSource.h"
#include "Calibration.h"
#include "os_specific_things.h"   // for usleep()

#include "text_output.h"

#include "name_mangle.h"
// FIXME: It is not nice to #include the following .cc file.  However, this is
// not part of The Code so it will not be linked.
#include "name_mangle.cc"

using namespace ReadingPeopleTracker;

static const char *Calibration_test_Revision = "@(#) Calibration_test.cc, rev 1.0 of Wed Aug  8 15:08:33 BST 2001, Author Rmi Belin, Copyright (c) 2000--2002 The University of Reading";

Calibration *calibration;
ImageSource *input_source = NULL;

bool done;


void get_image_point(NagVector &img_pt, bool update_image = true)
{
    Int32 minx;
    Int32 miny;
    Int32 x;
    Int32 y;
    Int32 width;
    Int32 height;
    long dev;
    short val;
    bool have_values = false;
    done = false;
    qdevice(LEFTMOUSE);
    qdevice(QKEY);
    qdevice(SPACEKEY);
    
    qreset();
    
    while (have_values == false)
    {
	dev = qread(&val);
	
        if (update_image)
	{
	    // make sure the image display is updated until someone clicks the mouse or so

	    while (qtest() == 0)  // GL event queue empty?
            {
		unsigned int delay;
                for (delay = 0; delay < 25; delay++)
                {
                    usleep(20*1000);  // wait 20 msec
                    if (qtest ())
                        break;
                }
                if (delay > 24) // we have waited 500 msec for input
                {
                    // refresh window:
                    input_source->display();
                    input_source->get_current()->draw_in_image();
                    gflush();
                }
            }
	}
	
	
        switch (dev)
	{
	case LEFTMOUSE:
	{
	    if (val != 1)	// button press
		break;
	    
	    // Retrieve coord of mouse click
	    
	    // get position of our window in absolute screen coordinates (upper left corner with GL)
	    x = getvaluator( MOUSEX );
	    y = getvaluator( MOUSEY );

	    // get origin of window (upper left corner with GL)
	    getorigin(&minx, &miny);
	    
	    cdebug << " window pos (" << minx << ", " << miny << ") " << endl;
	    cdebug << " mouse pos (" << x << ", " << y << ") " << endl;
	    
	    getsize(&width, &height);

	    // if the mouse click was not inside our window, ignore it:
	    if ((x < minx) || ((minx+width) < x) ||
		(y < miny) || ((miny+height) < y))
		break;

	    cdebug << " difference (" << x-minx << ", " << y-miny << ") " << endl;

	    // draw point in red
	    RGBcolor(255,0,0);
	    circf(x-minx, y-miny, 2);
	    
            // set img_pt to the relative position of the mouse cursor within our window
	    
	    img_pt[0] = x - minx;

	    if (Image::image_addressing_mode == IA_TOP_TO_BOTTOM)
                img_pt[1] = height - (y - miny);   // convert from GL's IA_BOTTOM_TO_TOP
            else
		img_pt[1] = y - miny;

            img_pt[2] = 1;  // make homogeneous coordinates: scale = 1

            have_values = true;
	    break;
	}
        case QKEY:
        {
	    exit(0);
	    break;
        }
	case SPACEKEY:
	{
	    // request to go to next image
            do
	    {
		// nothing
	    }
	    while (qread(&val) != dev);
		
	    input_source->get_next();
	    input_source->display();
	    input_source->get_current()->draw_in_image();
	    
	    done = false;
            have_values = false;

	    break;
	}
	case REDRAW:
	{
	    reshapeviewport();
	    input_source->display();
	    input_source->get_current()->draw_in_image();
	}
	}
    };
}

void check_ground_plane_pos()
{
    done = false;
    NagVector img_pt(3);
    NagVector ground_plane_pos(4);
    
    
    cout << "Let's check the ground plane position..." << endl;
    cout << "Please click an image point" << endl;

    while (done == false)
    {
	get_image_point(img_pt,false);

	cout << "******************************************" << endl
	     << endl
	     << "Image point : " << endl
	     << img_pt << endl;

	ground_plane_pos = calibration->get_world_from_ip(img_pt, 0);   // height = 0

	cout << "Ground plane position :" << endl
	     << ground_plane_pos << endl;
    }
}

void check_image_distance()
{
    
    done = false;
    NagVector img_pt(3);
    NagVector world_position(4);
    
    
    cout << endl << "Let's check the image distance..." << endl << endl;
    
    while (true)
    {
	cout << "******************************************" << endl << endl;
	cout << "Please click the upper or left image point" << endl;
	get_image_point(img_pt,true);
	if (done) return;
	qreset();
	cout << "Image point : " << endl;
	cout << img_pt << endl;
	realno height;
	cout << "Please enter the height of this point in cm: ";
	cin >> height;
	realno world_distance;
	cout << "Please enter the world distance in cm (height or width) : ";
	cin >> world_distance;
	
	realno distance_v =
	    calibration->get_image_distance_from_height_in_cm(img_pt,world_distance,height);
	realno distance_u =
	    calibration->get_image_distance_from_width_in_cm(img_pt,world_distance,height);
	
	
	cout << "Distance_u (width) in pixels : " << endl;
	cout << distance_u << endl;
	
	cout << "Distance_v (height) in pixels : " << endl;
	cout << distance_v << endl;
	
    }
}

void check_distance_to_camera()
{
    
    
    done = false;
    NagVector img_pt(3);
    NagVector world_position(4);
    
    
    cout << endl << "Let's check distance to camera from world position..." << endl << endl;
    cout << "Camera position :" << endl;
    cout << calibration->get_camera_position() << endl << endl;
    
    while (true)
    {
	cout << "******************************************" << endl << endl;
	cout << "Please click an image point" << endl;
	get_image_point(img_pt, true);
	if (done) return;
	qreset();
	realno height;
	cout << "Please enter the height of this point in cm: ";
	cin >> height;
	world_position = calibration->get_world_from_ip(img_pt,height);
	realno distance = calibration->get_distance_to_camera_from_worldpos(world_position);
	
	cout << "Image point : " << endl;
	cout << img_pt << endl;
	cout << "World position :" << endl;
	cout << world_position << endl ;
	cout << "Distance to camera :" << endl ;
	cout << distance << endl << endl;
    }
}




void check_world_distance()
{
    
    done = false;
    cout << "Let's check the world distance ..." << endl;
    NagVector img_pt_1(3);
    NagVector img_pt_2(3);
    realno distanceX;
    realno distanceZ;
    
    while(true)
    {
	cout << "please click two image points" << endl;
	
	get_image_point(img_pt_1, true);
	
	if (done)
	    return;
	
	cout << "Image point 1: " << endl;
	cout << img_pt_1 << endl;
	
	RGBcolor(255,0,0);
	circf(img_pt_1[0],img_pt_1[1],2);
	
	get_image_point(img_pt_2, false);
	
	if (done)
	    return;
	
	cout << "Image point 2: " << endl;
	cout << img_pt_2 << endl;
	
	RGBcolor(0,127,255);
	circf (img_pt_2[0], img_pt_2[1],2);
	
	qreset();
	realno height;
	cout << "Please enter the height of the upper point in cm: ";
	cin >> height;
	
	// check which point is higher up before querying Calibration for the distance
	if ((img_pt_2[1] > img_pt_1[1]) ^
	    (Image::image_addressing_mode == IA_BOTTOM_TO_TOP))
	{
	    distanceX = calibration->
		get_world_distance_from_width_in_pixels(img_pt_1,    // this must be the upper image point
							fabs(img_pt_2[0] - img_pt_1[0]), height);
	    distanceZ = calibration->
		get_world_distance_from_height_in_pixels(img_pt_1,   // this must be the upper image point
							 fabs(img_pt_2[1] - img_pt_1[1]), height);
	}
	else
	{
	    distanceX = calibration->
		get_world_distance_from_width_in_pixels(img_pt_2,    // this must be the upper image point
							fabs(img_pt_1[0] - img_pt_2[0]), height);
	    distanceZ = calibration->
		get_world_distance_from_height_in_pixels(img_pt_2,   // this must be the upper image point
							 fabs(img_pt_1[1] - img_pt_2[1]), height);
	}
	cout << "DistanceX (width) in cm : "
	     << distanceX << endl;
	
	cout << "DistanceZ (height) in cm : "
	     << distanceZ << endl << endl;
    }
    return;
    
}


int main(int argc, char **argv)
{
    // simply use JPEGSource now
    
    if ((argc < 2) || (argc > 3))
    {
	cerror << "Usage: " << argv[0] << " /path/to/image00001.jpg  [/path2/calibration-file] " << endl
	       << "  By default, the calibration file is derived from the image files, eg /path/to/image.calibration " << endl;
	exit(1);
    }

    input_source = new JPEGSource (argv[1], false, 0);
    
    if (input_source == NULL)
    {
	cerror << argv[0] << "No valid video source given." << endl;
	exit(1);
    }
    
    char *calibration_filename;

    if (argc > 2)
	calibration_filename = argv[2];
    else
    {
	// automatically derive calibration data file name from input...
	calibration_filename = name_mangle(argv[1], "calibration", true);  // true: remove number
    }
    
    calibration = new Calibration(calibration_filename,
				  input_source->get_ydim());
    

    if (input_source->get_current() == NULL)
    {
	cerror << argv[0] << ": No image found in video source." << endl;
	exit(1);
    }

    if (Image::image_addressing_mode == IA_TOP_TO_BOTTOM)
	cinfo << " Using image addressing mode TOP_TO_BOTTOM ..." << endl;
    else
	cinfo << " Using image addressing mode BOTTOM_TO_TOP ..." << endl;
    
    input_source->get_current()->display();
    

    cout << endl
	 << "Which function would you like to test ?" << endl;
    cout << "       1. Check ground plane position" << endl;
    cout << "       2. Check world distance" << endl;
    cout << "       3. Check image distance" << endl;
    cout << "       4. Check distance to camera" << endl << endl;
    
    int option;
    
    cin >> option;
    
    switch(option)
    {
    case 1:
	check_ground_plane_pos();
	break;
    case 2:
	check_world_distance();
	break;
    case 3:
	check_image_distance();
	break;
    case 4:
	check_distance_to_camera();
	break;
    default:
	cerror << "suboptimal choice of option, try again" << endl;
    }
    
}

// Local Variables:
// compile-command: "make Calibration_test |& fgrep :"
// End:
