// 
// NagVector.h
// 
// A class for manipulating realno n-vectors
// (uses nag routines)

#ifndef __NAG_VECTOR_H__
#define __NAG_VECTOR_H__

#include <cstdlib>
#include <iostream>
#include <cmath>

#include "realno.h"
#include "EnvParameter.h"

namespace ReadingPeopleTracker
{

class NagMatrix;

class NagVector
{
    friend istream &operator>>(istream&, NagVector &v);
    friend ostream &operator<<(ostream&, const NagVector &v);
//    friend class NagMatrix;

protected:
    
    unsigned int alloc_size;	// size of allocated data (number of elements)
    unsigned int size;		// size of vector, used (number of elements)
    realno *data;		// array of values
    bool own_memory; 	        // if true then delete data upon destruction
    
    static EnvParameter output_binary_mode;

    void vector_error(const char *message) const;
    
public:

    // constructors, destructors;
    // this constructor sets own_memory to false
    NagVector(unsigned int n, realno *dt, unsigned int asize = 0);
    
    // all other constructors create NagVectors 
    // that dynamically allocate memory
    NagVector(unsigned int n, unsigned int the_alloc_size = 0);
    NagVector(unsigned int n, realno (*func) (unsigned int), unsigned int asize = 0);
    NagVector()
	{
	    size = alloc_size = 0;
	    data = NULL;
	    own_memory = true;
	}
    NagVector(const NagVector &v);
    NagVector(const NagMatrix &m);
    ~NagVector();
    
    NagVector &operator=(const NagVector &vec2);
    void reconstruct(unsigned int n, unsigned int new_alloc_size = 0);    
    
#ifdef _SVID_SOURCE
    // the following uses SVID 48-bit random numbers...

    // random noise with normal distribution.  FIXME: where should this method go?
    static unsigned int iset;
    static bool seed_set;
    static realno gset;
    static void check_seed();
    static realno normal_random();

    // set each element randomly
    void set_unif_random(realno min = -1, realno max = 1.0);
    void set_gauss_random(realno mean = 0, realno sd = 1.0);
#endif // ifdef _SVID_SOURCE

    // data access
    unsigned int get_size() const { return size; }
    realno *get_data() { return data; }
    const realno *get_data_const() const { return data; }
//  operator realno*() { return data; }
    
    // reset to null, don't delete data
    void reset();
    
    inline void set(unsigned int i, realno value)
	{
	    data[i] = value;
	}

    realno read(unsigned int i) const 
	{
	    return data[i];
	}
//   realno *get(unsigned int i)
//     { return &(data[i]); }
    
    realno &operator[](unsigned int i) const
	{ return data[i]; }
    
    void output();
    
    // utility functions
    
    void clear(const realno value = ((realno) 0.0));
    void add(const realno itm);
    void copy_data(float *cdata, unsigned int n = 0);
    void copy_data(double *cdata, unsigned int n = 0);
    
    void add (const NagVector &vec2, NagVector &res) const;
    void subtract(const NagVector &vec2, NagVector &res) const;
    void copy(NagVector &res) const;
    void scale(const realno lmda, NagVector &res) const;
    void sub_vector(const unsigned int i, const unsigned int j, NagVector &res) const;
    void convolve(const NagVector &kern, NagVector &res) const;
    void concat(const NagVector &vec2, NagVector &res) const;
    void flip(NagVector &res) const;
    
    // x.mult_trans(y, M) sets M = x y^T
    void mult_trans(const NagVector &vec2, NagMatrix &res) const;

    
    // select(x)
    // pick the (n * x)'th item from the sorted data
    // (see Numerical Recipies)
    realno select(realno x = 0.5) const;

    realno length2() const;
    realno length() const { return sqrt(length2()); }
    realno sum() const;
    
    realno dot (const NagVector &vec2) const;
    
    NagVector &operator+= (const NagVector &v2);
    NagVector &operator-= (const NagVector &v2);
    NagVector &operator*= (const realno a);
    NagVector &operator/= (const realno a);
    
private:

    // private helpers
    realno nr_select(unsigned long k, unsigned long n, realno *arr) const;

};

istream &operator>> (istream &in, NagVector &v);
ostream &operator<< (ostream &out, const NagVector &v);

inline NagVector operator+ (const NagVector &v1,  const NagVector &v2)
{
    NagVector res;
    v1.add(v2,res);
    return res;
}

inline NagVector operator- (const NagVector &v1,  const NagVector &v2)
{
    NagVector res;
    v1.subtract(v2,res);
    return res;
}

inline NagVector operator* (const realno k,  const NagVector &v)
{
    NagVector res;
    v.scale(k,res);
    return res;
}

inline NagVector operator* (const NagVector &v, const realno k)
{
    NagVector res;
    v.scale(k,res);
    return res;
}

inline NagVector operator/ (const NagVector &v, const realno k)
{
    NagVector res;
    v.scale(1.0 / k,res);
    return res;
}

} // namespace ReadingPeopleTracker

#endif

