/***************************************************************
 * C - C++ Header
 *
 * File : 	Matrix2by2.h
 *
 * Module :	Matrix2by2
 *
 * Author : 	A M Baumberg (CoMIR)
 *
 * Creation Date : Tue Jun  4 11:47:24 1996 
 *
 * Comments : 	basic 2 by 2 matrix
 *
 ***************************************************************/

#ifndef __MATRIX_2_BY_2__
#define __MATRIX_2_BY_2__

#include <math.h>
#include <iostream>
#include <stdlib.h>

#include "Point2.h"
#include "text_output.h"

namespace ReadingPeopleTracker
{

class Matrix2by2
{
public:
    
    realno val00;
    realno val01;
    realno val10;
    realno val11;
    
    
    // constructors
    
    // null
    Matrix2by2()
	{
	    val00 = val01 = val10 = val11 = 0;
	}
    
    // scalar multiple of the identity
    Matrix2by2(realno b) 
	{
	    val00 = val11 = b; val01 = val10 = 0;
	}
    
    // diagonal
    Matrix2by2(realno b00, realno b11) 
	{
	    val00 = b00; val11 = b11; val01 = val10 = 0;
	}
    
    // symmetric
    Matrix2by2(realno b00, realno b01, realno b11)
	{
	    val00 = b00; val01 = val10 = b01; val11 = b11;
	}
    
    // full
    Matrix2by2(realno b00, realno b01, realno b10, realno b11)
	{
	    val00 = b00; val01 = b01; val10 = b10; val11 = b11;
	}
    
    Matrix2by2(const Matrix2by2 &b)
	{
	    val00 = b.val00; val01 = b.val01; 
	    val10 = b.val10; val11 =  b.val11;
	}
    
    // functions
    
    inline Matrix2by2 add(const Matrix2by2 &b)
	{
	    return Matrix2by2(val00 + b.val00, val01 + b.val01, 
			      val10 + b.val10, val11 + b.val11);
	}
    
    inline Matrix2by2 subtract(const Matrix2by2 &b)
	{
	    return Matrix2by2(val00 - b.val00, val01 - b.val01,
			      val10 - b.val10, val11 - b.val11);
	}
    
    inline Matrix2by2 multiply(const Matrix2by2 &b)
	{
	    return Matrix2by2(val00 * b.val00 + val01 * b.val10,
			      val00 * b.val01 + val01 * b.val11,
			      val10 * b.val00 + val11 * b.val10,
			      val10 * b.val01 + val11 * b.val11);
	}
    
    
    inline Matrix2by2 transpose()
	{
	    return Matrix2by2(val00, val10, val01, val11);
	}
    
    inline void operator+=(const Matrix2by2 &b)
	{ 
	    val00 += b.val00; val01 += b.val01;
	    val10 += b.val10; val11 += b.val11;
	}
    
    inline void operator-=(const Matrix2by2 &b)
	{ 
	    val00 -= b.val00; val01 -= b.val01;
	    val10 -= b.val10; val11 -= b.val11;
	}
    
    inline Point2 multiply(Point2 v)
	{
	    return Point2(val00 * v.x + val01 * v.y,
			  val10 * v.x + val11 * v.y);
	}
    
    inline realno determinant()
	{
	    return val00 * val11 - val01 * val10;
	}
    
    inline realno check_determinant()
	{
	    realno det = determinant();
	    if (fabs(det) < 1e-30)
	    {
		cerror << " Matrix2by2::check_determinant(): det < 1e-30, aborting..."
		       << endl;
		abort();
	    }
	    return det;
	}
    
    inline Matrix2by2 inverse()
	{
	    realno det = check_determinant();
	    return Matrix2by2(val11 / det, -val01 / det,
			      -val10 / det, val00 / det);
	}
    
    inline bool inverse(Matrix2by2 &res)
	{
	    realno det = determinant();
	    if (fabs(det) < 1e-30)
		return false;
	    res = Matrix2by2(val11 / det, -val01 / det,
			     -val10 / det, val00 / det);
	    return true;
	} 
    
    inline Matrix2by2 scale(realno b)
	{
	    return Matrix2by2(val00 * b, val01 * b, val10 * b, val11 * b);
	}
    
    inline void operator*=(realno b)
	{
	    val00 *= b; val01 *= b; val10 *= b; val11 *= b;
	}
    
    inline Matrix2by2 divide(realno b)
	{
	    return Matrix2by2(val00 / b, val01 / b, val10 / b, val11 / b);
	}
    
    
    inline void get_real_eigenvalues(realno &e1, realno &e2)
	{
	    realno tmp1 = val00 + val11;
	    realno tmp2 = val00 - val11;
	    tmp2 = sqrt(tmp1 * tmp2 + (4 * val01 * val10));
	    e1 = tmp1 + tmp2;
	    e2 = tmp1 - tmp2;
	}
    
    inline realno get_max_real_eigenvalue()
	{
	    realno tmp = val00 - val11;
	    return val00 + val11 + sqrt(tmp * tmp + (4 * val01 * val10));
	}
    
    inline realno get_min_real_eigenvalue()
	{
	    realno tmp = val00 - val11;
	    return val00 + val11 - sqrt(tmp * tmp + (4 * val01 * val10));
	}  
    
    
};

inline ostream &operator<<(ostream &strm_out, const Matrix2by2 &m)
{
    strm_out << "[ " << m.val00 << "  " << m.val01 << " ]" << endl;
    strm_out << "[ " << m.val10 << "  " << m.val11 << " ]" << endl;
    return strm_out;
}

inline Matrix2by2 operator+(Matrix2by2 a, Matrix2by2 b)
{
    return a.add(b);
}
 
inline Matrix2by2 operator-(Matrix2by2 a, Matrix2by2 b)
{ 
    return a.subtract(b);
}

inline Matrix2by2 operator*(Matrix2by2 a, Matrix2by2 b)
{
    return a.multiply(b); 
}

inline Point2 operator*(Matrix2by2 a, Point2 b)
{
    return a.multiply(b); 
}

} // namespace ReadingPeopleTracker

#endif
