/*************************************************************************************
* FILENAME:	timestamp_c.h							     *
*										     *
* DESCRIPTION:									     *
* This header file details the TIMESTAMP (and related) structs.			     *
* For a detailed description and specification of each field, see ADVISOR DOCUMENT   *
* 009 (JPEG Header Time Stamp Specification).					     *
*										     *
**************************************************************************************/

#ifndef __TIMESTAMP_C_H__
#define __TIMESTAMP_C_H__

namespace ReadingPeopleTracker
{

/*	Identifies contrast/brightness setting as undefined	*/
#define TS_CONTRAST_UNSPECIFIED -1
#define TS_BRIGHTNESS_UNSPECIFIED -1

/*==================================================================================
 			         Enumerations used in TS_FORMAT
  ==================================================================================*/

/*	Refer to Table 1, Col 1 (Pg 8 of document)	*/
typedef enum
{
	TS_COLOUR,
	TS_MONOCHROME

} TS_CAMERA_OUTPUT;

/*	Refer to Table 1, Col 2 (Pg 8 of document)	*/
typedef enum
{
	TS_LIVE,
	TS_DV,
	TS_SVHS,
	TS_VHS

} TS_RECORDED_FORMAT;

/*	Refer to Table 1, Col 3 (Pg 8 of document)	*/
typedef enum  
{
	TS_COMPOSITE,
	TS_S_VIDEO

} TS_DIGITISER_INPUT;

/*	Refer to Table 2, Col 1 (Pg 9 of document)	*/
typedef enum
{
	TS_CORECO,
	TS_SOLLATEK,
	TS_FAST_ELECTRONICS

} TS_DIGITISER;

/*	Code to denote whether format information is specified in the timestamp	*/
typedef enum
{
	TS_FORMAT_SPECIFIED,
	TS_FORMAT_UNSPECIFIED

} TS_FORMAT_STATUS;


/*==================================================================================
                                         TS_FORMAT struct 
  ==================================================================================*/

/*	Refer to Table 1 (Pg 8 of document)	*/
typedef struct
{
	TS_FORMAT_STATUS format_status;
	TS_CAMERA_OUTPUT camera_output;
	TS_DIGITISER_INPUT digitiser_input;
	TS_RECORDED_FORMAT recorded_format;

} TS_FORMAT;

/*------------------------ Description of the TS_FORMAT struct --------------------
 *
 *	The TS_FORMAT struct relates to the format field of the timestamp.  The
 *	format status is set to either TS_FORMAT_SPECIFIED or
 *	TS_FORMAT_UNSPECIFIED depending on whether details of the format have
 *	been supplied in the timestamp.  The other struct members hold the
 *	components of the format described in Table 1 (Pg 8 of document).
 *
 *---------------------------------------------------------------------------------*/


/*	Code used to denote result of timestamp conversion function	*/
typedef enum
{
	TS_OK,
	TS_ERROR

} TS_RESULT;


/*==================================================================================
					TS_TIMESTAMP struct
  ==================================================================================*/

typedef struct
{
	unsigned int version;
	unsigned int year;
	unsigned int month;
	unsigned int day;
	unsigned int hour;
	unsigned int minute;
	unsigned int second;
	unsigned int millisecond;
	unsigned int frame_count;
	unsigned int pc_board_num;
	unsigned int pc_port_num;
	char camera_id[4];
	char pc_name[7];
	char complex[3];
	char zone[7];
	char sector[5];
	char area[7];
	TS_FORMAT format; 
	int contrast;
	int brightness;
	TS_DIGITISER digitiser; 
	
} TS_TIMESTAMP;

/*-------------------- Description of the TS_TIMESTAMP struct ------------------
 *
 *	The TS_TIMESTAMP C struct represents the JPEG Header Time Stamp.  Each
 *	field of the Time Stamp (found in section 3.1, Pg 6-7 of document)
 *	corresponds to one of the struct members.
 * 
 *	Numeric Fields:
 *
 *	year, month, day, hour, minute, second, millisecond, frame_count,
 *	pc_board_num, pc_port_num, version.
 *
 *	These straightforward fields are all unsigned int.
 *
 *
 *	Textual Fields:
 *
 *	camera_id, pc_name, complex, zone, sector, area.
 *
 *	The above fields vary from purely alphabetic to alphanumeric, with or
 *	without leading spaces.  These fields are stored as C-strings.
 *
 *
 *	Other Fields:
 *
 * 	format          This member expands the Time Stamp field's single letter code,
 *                      as per Table 1 (Pg 8 of document).  The members of the TS_FORMAT
 *                      struct store this expanded data.
 *
 *	contrast	These fields store integers between 0-255.
 *      & brightness    If a setting is unknown the value is set to
 *                      TS_CONTRAST_UNSPECIFIED / TS_BRIGHTNESS_UNSPECIFIED respectively.
 *
 *	digitiser	This member expands the single letter code from Table 2
 *                      (Pg 9 of document).
 *
 *
 *---------------------------------------------------------------------------------*/

/*--------------------------------- Conversion function ---------------------------*/


extern TS_RESULT ts_convert_timestamp(char *timestamp_ptr,
				      TS_TIMESTAMP *struct_ptr);



/*==================================================================================
					END OF FILE
  ==================================================================================*/

} // namespace ReadingPeopleTracker

#endif
