/*
 * RGB32Image.h
 * 24-bit (plus alpha) colour image class derived from generic
 * Image class in Image.h/Image.cc
 *
 * author : A M Baumberg
 */

#ifndef __RGB32_IMAGE_H__
#define __RGB32_IMAGE_H__

#include <math.h>
#include <ostream>
#include <cassert>
#include <string.h> // for memcpy()

#include "Image.h"
#include "GreyMap.h"
#include "tracker_defines_types_and_helpers.h"

#ifndef NO_DISPLAY
#ifndef USE_GL
#include <X11/Xutil.h>
#endif
#endif   // #ifndef NO_DISPLAY

namespace ReadingPeopleTracker
{

class Grey8Image;

// write out an RGB32pixel as (r,g,b) triplet
ostream &operator<< (ostream &target, const RGB32pixel &ColourPixel);

class RGB32Image : public Image
{
private:
    
    static const unsigned int BOX_SIZE;
    RGB32pixel RGB32_draw_colour;

#ifndef NO_DISPLAY
#ifndef USE_GL
    int rshift, gshift, bshift;
    XVisualInfo xv_info;
    int matched_depth;
#endif
#endif   // #ifndef NO_DISPLAY
    
public:
    
    RGB32Image(unsigned int the_width, unsigned int the_height,
	       frame_id_t the_frame_id = 0, frame_time_t the_frame_time = 0,
	       unsigned char *the_data = NULL) : Image(the_width, the_height,
						       the_frame_id, the_frame_time,
						       4, the_data)
	{ 
	    image_type = RGB32;

	    RGB32_draw_colour.alpha = 0x00;
	    RGB32_draw_colour.red
		= RGB32_draw_colour.green
		= RGB32_draw_colour.blue
		= 0xff;    
	}

    Image *copy (Image *res = NULL) const;

    inline Image *copy_type(unsigned int w = 0, unsigned int h = 0) 
	{ 
	    if (w == 0)
		w = width;
	    if (h == 0)
		h = height;
	    return new RGB32Image(w, h);
	}
    
    inline Image *virtual_copy(Image *res = NULL) const
	{
	    if (res == NULL)
		return new RGB32Image(width, height, frame_id, frame_time_in_ms, data);
	    else 
	    {
		// just copy over our variables
		memcpy(res, this, sizeof(*this));   // fast copy, including pointers
		res->own_data = false;
		return res;
	    }
	}
    
    long display(long glw = NULLWIN);
    RGB32Image *rgb_read_image(RGB32Image *res);
    void save_pnm(char *filename);
    Image *difference(Image *image2, Image *res = NULL);
    Image *difference(Grey8Image *image2, Image *res = NULL);
    // differencing plus thresholding:
    Image *difference(Image *image2, realno threshold, Image *res = NULL);
    Image *threshold(unsigned int threshold, Image *res = NULL,
		     unsigned int *no_marked = NULL);
    Image *simple_difference(Image *image2, realno thresh, 
			     unsigned int *no_marked, Image *res =  NULL);
    Image *resample(unsigned int xstep, unsigned int ystep, Image *resampled = NULL);
    unsigned int measure_contrast(void *p1, void *p2);
    inline Image *to_rgb(Image *res = NULL)
	{
	    return this;
	}
    Image *convolve(Kernel k, Image *convolved = NULL);
    Image *image_blend(Image *i2, unsigned int a, unsigned int n, Image *res = NULL);
    Image *map_intensities(PntGreyMap gmap, Image *res = NULL);
    
    void median_img_array(Image **img_array, unsigned int asize);
    
    Image *sobel(unsigned int threshold, Image *res = NULL);
    Image *sobel(Image *hsobel = NULL);
    Image *half_blur(Image *res = NULL);
    Image *transform(unsigned int centre_x,
		     unsigned int centre_y,
		     realno angle,
		     realno scale, Image *res = NULL);
    
    // greyscale_using_lookup added by Tony Heap (ajh)
    Grey8Image *greyscale_using_lookup(unsigned char *lookup,
				       Grey8Image *dest = NULL);
    
    // 32-bit versions of mask, neighbour_order, clear and clear_border
    Image *mask(Image *mask, Image *res);
    Image *neighbour_order(unsigned int n, Image *ordered = NULL);
 
    // Filtering to grey based on colour
    Image *ColourFilter(Image *res = NULL,
			ColourFilteringMethod method = CF_METHOD_Y_709);

#ifdef HSV
    // converting to HSV (after D Travis) for HSV32Image
    Image *to_HSV32(Image *res = NULL);
#endif

    void clear(const RGB32pixel fill_colour);
    void clear_border();
   
    // drawing routines
    void set_draw_colour(unsigned char red, unsigned char green,
			 unsigned char blue);
    void set_draw_colour(unsigned char grey_value);
    
    void draw_horizontal(unsigned int y, unsigned int xl, unsigned int xr, bool dotted = false);
    void draw_vertical(unsigned int x, unsigned int yl, unsigned int yu, bool dotted = false);
    
    void plot_pixel(unsigned int x, unsigned int y);

    // 32-bit version of draw_rectangle
    void draw_rectangle(int xmin, int xmax, int ymin, int ymax, unsigned int val);    
    

    RGB32pixel to_RGB32(unsigned long &colour);
    RGB32pixel to_RGB32(const unsigned char colour);
    unsigned long to_ulong(RGB32pixel &colour);

    static inline unsigned char colour_distance (
	const RGB32pixel pixel1, const RGB32pixel pixel2)
	{
	    // NB not a good approach and unintuitive.  Use HSV32Image::colour_distance(...).
	    return (ABS(pixel1.red - pixel2.red) +
		    ABS(pixel1.green - pixel2.green) +
		    ABS(pixel1.blue - pixel2.blue)) / 3;
	}
    
private:
    // private helpers and static members:
    static const unsigned int HIST_SHIFT;
    static const unsigned int HIST_RANGE;
    static const unsigned int HIST_GROUP;
    static const unsigned int HIST_SHIFT_RED;
    static const unsigned int HIST_SHIFT_GREEN;
    static const unsigned int HIST_SHIFT_BLUE;
    static const unsigned long HIST_MASK_RED;
    static const unsigned long HIST_MASK_GREEN;
    static const unsigned long HIST_MASK_BLUE;

    unsigned long lookup_index(unsigned long l);
    unsigned char rgb_diff(RGB32pixel &pix1, RGB32pixel &pix2);
    void get_detail(RGB32pixel *g1, RGB32pixel *g2,
		    unsigned char *r, unsigned int &width);
    static int compar_char(const void *c1, const void *c2);

};

} // namespace ReadingPeopleTracker

#endif
