/*
 * RGB32Image.cc
 * 24-bit (plus alpha) colour image class derived from generic
 * Image class in Image.h/Image.cc
 *
 * author : A M Baumberg
 */

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <fstream>
#include <cstring>  // for memcpy()

#include "RGB32Image.h"

#include "Grey8Image.h"
#include "Point2.h"
#include "text_output.h"

#ifdef HSV
#include "HSV32Image.h"
#endif

#ifndef NO_DISPLAY
#ifdef USE_GL
#include <gl/gl.h>
#else
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#endif
#endif   // #ifndef NO_DISPLAY

#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

// definition and initialisation of static member variables
const unsigned int RGB32Image::BOX_SIZE = 8;
const unsigned int RGB32Image::HIST_SHIFT = 2;
const unsigned int RGB32Image::HIST_RANGE = 256 >> HIST_SHIFT;
const unsigned int RGB32Image::HIST_GROUP = 1 << HIST_SHIFT;
const unsigned int RGB32Image::HIST_SHIFT_RED = HIST_SHIFT;
const unsigned int RGB32Image::HIST_SHIFT_GREEN = HIST_SHIFT*2;
const unsigned int RGB32Image::HIST_SHIFT_BLUE = HIST_SHIFT*3;
const unsigned long RGB32Image::HIST_MASK_RED = (HIST_RANGE-1) << HIST_SHIFT;
const unsigned long RGB32Image::HIST_MASK_GREEN = (HIST_RANGE-1) << 8+HIST_SHIFT;
const unsigned long RGB32Image::HIST_MASK_BLUE = (HIST_RANGE-1) << 16+HIST_SHIFT;

// Often, we would like to address the whole struct as an unsigned long
// so here is a not-so-nice casting function:
inline RGB32pixel RGB32Image::to_RGB32(unsigned long &colour)
{
    return * ((RGB32pixel*) &colour);
}

inline RGB32pixel RGB32Image::to_RGB32(const unsigned char colour)
{
    unsigned long colour_all_over = ((colour << 8 + colour) << 8 + colour) << 8 + colour;
    return to_RGB32(colour_all_over);
}

// and the other way around:
inline unsigned long RGB32Image::to_ulong(RGB32pixel &colour)
{
    return * ((unsigned long*) &colour);
}

// write out an RGB32pixel as (r,g,b) triplet
ostream &operator<< (ostream &target, const RGB32pixel &ColourPixel)
{
    target << "(";
    u_int32_t AX = 0;
    *((unsigned char*) &AX) = ColourPixel.red;
    target << AX << ",";
    AX = 0;
    *((unsigned char*) &AX) = ColourPixel.green;
    target << AX << ",";
    AX = 0;
    *((unsigned char*) &AX) = ColourPixel.blue;
    target << AX << ")";
    return target;
}

// copy image data, dimensions, time.
Image *RGB32Image::copy(Image *res) const
{
    if (res == NULL)
	res = new RGB32Image(width, height, frame_id, frame_time_in_ms);
    else
    {
	// check destination image format and dimensions
	if ((res->get_image_type() != RGB32) ||
	    (res->get_width() != width) ||
	    (res->get_height() != height))
	{
	    cerror << " RGB32Image::copy: cannot copy to resulting image of different type/dimensions "
		   << endl;
	    exit(1);
	}
	
	// use same frame id and time since it is the same image
	res->set_frame_id(frame_id);
	res->set_frame_time_in_ms(frame_time_in_ms);
    }
    res->set_timestamp(&timestamp);

    // NB window handle (in GL, variable glwin) is not copied over!
    
    void *src = data;
    void *dest = res->get_data();
    size_t n = (width * height) << 2;
    
    memcpy(dest, src, n);
    
    return res;
}

#ifdef USE_GL
long RGB32Image::display(long awindow)
{
#ifdef NO_DISPLAY
    return NULLWIN;
#else
    if (awindow != NULLWIN)
	glwin = awindow;
    
    if (glwin == NULLWIN)
    {
	unsigned int zoom = Image::display_zoom->value;
	if (window_title[0] == 0)  // no title given?  default to this:
	    sprintf(window_title," RGB32 image %d by %d ", width, height);
	prefsize(width * zoom, height * zoom);
	
// nts: this gives an error in Ygl version 4.x
//	foreground();
	
	glwin = winopen(window_title);
	
	winset(glwin);
	reshapeviewport();
	prefsize(width * zoom, height * zoom);
	if (zoom != 1)
	{
	    rectzoom((float) zoom, (float) zoom);
	    viewport(0, width * zoom - 1, 0, height * zoom - 1);
	    ortho2(-0.5, width + 0.5, -0.5, height + 0.5);
	}
	winconstraints();
	RGBmode();
	
	gconfig();  // we have to do this after RGBmode and before cpack
	
	cpack(0x00ffffff);
	gflush();
	gconfig();
    }
    winset(glwin);
    lrectwrite(0,0,width-1,height-1,(Int32 *) data);
    return glwin;
#endif   // #ifdef NO_DISPLAY  #else
}


#else
// X interface

long RGB32Image::display(long dummy)
{
#ifndef NO_DISPLAY
    if (mydisplay == NULL)
    {
	if (window_title[0] == 0)  // no title given?  default to this:
	    sprintf(window_title," RGB image %d by %d ", width, height);      
	XSizeHints myhints;
	myhints.width = width;
	myhints.height = height;
	myhints.flags = PSize;
	mydisplay = XOpenDisplay("");
	int screen = DefaultScreen(mydisplay);
	rshift = bshift = gshift = 0;
	Status res = XMatchVisualInfo(mydisplay, screen, 24, TrueColor,
				      &xv_info);
	matched_depth = 32;
	if (res == 0) // no matching visual found
	{
	    Status res = XMatchVisualInfo(mydisplay, screen, 8, TrueColor,
					  &xv_info);
	    if (res == 0) // no matching visual found
	    {
		cerror << "RGB32Image::display: XMatchVisualInfo did not return usable visual" << endl;
		exit(1);
	    }
	    rshift = bshift = gshift = 0;
	    register int i;
	    for (i = xv_info.red_mask; i < 128; i *= 2)
		rshift++;
	    for (i = xv_info.green_mask; i < 128; i *=2)
		gshift++;
	    for (i = xv_info.blue_mask; i < 128; i *=2 )
		bshift++;
	    matched_depth = 8;
	}
	else
	{
	    register long i;
	    for (i = xv_info.red_mask; i > 256; i /= 256)
		rshift++;
	    for (i = xv_info.green_mask; i > 256; i /= 256)
		gshift++;
	    for (i = xv_info.blue_mask; i > 256; i /= 256)
		bshift++;
	    rshift = 3 - rshift;
	    gshift = 3 - gshift;
	    bshift = 3 - bshift;
	}
	
	RGB32pixel white = WhitePixel(mydisplay, screen);
	RGB32pixel black = BlackPixel(mydisplay, screen);
	Colormap cmap = XCreateColormap(mydisplay,
					XRootWindow(mydisplay, screen),
					xv_info.visual, AllocNone);
	
	mypixmap = XCreatePixmap(mydisplay,
				 XRootWindow(mydisplay, screen), width,
				 height, xv_info.depth);
	
	XSetWindowAttributes swa;
	swa.background_pixel = BlackPixel(mydisplay, screen);
	swa.border_pixel = WhitePixel(mydisplay, screen);
	swa.colormap = cmap;
	swa.background_pixmap = mypixmap;
	
	mywindow = XCreateWindow (mydisplay, RootWindow(mydisplay, screen),
				  100,100,width,height,1,
				  xv_info.depth, InputOutput,
				  xv_info.visual,
				  CWBackPixmap | CWColormap | CWBorderPixel, &swa);
	
	XSetStandardProperties(mydisplay,
			       mywindow, &window_title,
			       "image", None,
			       NULL, 0, &myhints);
	
	XMapRaised(mydisplay, mywindow);
	
	myimage = XCreateImage( mydisplay, xv_info.visual,
				xv_info.depth, ZPixmap, 0, None, width, height,
				matched_depth, 0);
#ifdef LSB_FIRST
	myimage->byte_order = LSBFirst;
	myimage->bitmap_bit_order = LSBFirst;
#else
	myimage->byte_order = MSBFirst;
	myimage->bitmap_bit_order = MSBFirst;
#endif
	
	//mygc = XCreateGC (mydisplay, mywindow, 0, 0);
	mygc = XCreateGC (mydisplay, mypixmap, 0, 0);

	// check whether the data format is compatible with ours
	if ((matched_depth == 32) && (Image::image_storage_mode == IA_TOP_TO_BOTTOM)
	    && (rshift == 3) && (gshift == 2) && (bshift == 1))
	    myimage->data = (char *) data;
	else
	    myimage->data = new char[width * height * matched_depth / 8];
    }
    if (xv_info.depth == 8)
    {
	cerror << "RGB32Image::display: Warning: No TrueColour visual found.  Using 8-bit visual." << endl;
	
	unsigned int x;
	unsigned int y;
	RGB32pixel *idat;
	unsigned char *odat = (unsigned char*) myimage->data;
	unsigned char red_mask = xv_info.red_mask;
	unsigned char green_mask = xv_info.green_mask;
	unsigned char blue_mask = xv_info.blue_mask;
	
	for (y = height; y > 0; y--)
	{
	    idat = (RGB32pixel *)get_pixel(0, y-1);
	    for (x = 0; x < width; x++)    
	    {
		*odat++ = (((idat->red) >> rshift) & red_mask)
		    + (((idat->green) >> gshift) & green_mask)
		    + (((idat->blue) >> bshift) & blue_mask);
		idat++;
	    }
	}
    }
    else  // that is, xv_info.depth != 8 so it must be 24
    {
	assert (xv_info.depth == 24);

	if (myimage->data != (char *)(data))
	{
	    // xv_info.depth == 24
	    unsigned int x,y;
	    RGB32pixel *idat;
	    unsigned char *odat = (unsigned char*) myimage->data;
	    for (y = height; y > 0; y--)
	    {
		idat = (RGB32pixel *)get_pixel(0, (y-1));
		for (x = 0; x < width; x++)    
		{
		    odat[rshift] = idat->red;
		    odat[gshift] = idat->green;
		    odat[bshift] = idat->blue;
		    idat++;
		    odat += 4;
		}
	    }
	}
    }
    
    //  XPutImage(mydisplay, mywindow, mygc, myimage, 0,0,0,0,width,height);
    XPutImage(mydisplay, mypixmap, mygc, myimage, 0,0,0,0,width, height);
    XClearArea(mydisplay, mywindow, 0,0,0,0, True);
    
#endif   // #ifndef NO_DISPLAY
    return 0;    
}

#endif

void RGB32Image::save_pnm(char *filename)  // save PPM file
{
    // open output file for PPM output
    ostream *ostream_ptr = NULL;
    
    if (filename != NULL)
	ostream_ptr = (ostream*)
	    new ofstream(filename, ios::out);
    
    if ((ostream_ptr == NULL) || (ostream_ptr->bad()))
        {
            cerror << "RGB32Image::save_pnm: could save image file "
                   << "" << filename << " . "
                   << endl;

	    if (ostream_ptr != NULL)
		delete ostream_ptr;

	    return;
        }

    ostream &data_out = *ostream_ptr;
    
    // write PPM raw header: type P6
    data_out << "P6" << endl
	     << width << " " << height << endl
	     << "255" << endl;

    // write pixel values, depending on image_storage_mode
    // The PNM formats are storing the image IA_TOP_TO_BOTTOM
    if (Image::image_storage_mode == Image::image_addressing_mode)
    {
	// no flipping necessary
	for (unsigned int y = 0; y < height; y++)
	{
	    // (NB get_pixel depends on Image::image_storage_mode)
	    RGB32pixel *dat1 = (RGB32pixel *) get_pixel(0,y);
	    for (unsigned int x = 0; x < width; x++)
	    {
		data_out << dat1->red << dat1->green << dat1->blue;
		dat1++;
	    }
	}
    }
    else
    {
	// flip image (y lines) as we write the file
	for (unsigned int y = height; y > 0; y--)
	{
	    // (NB get_pixel depends on Image::image_storage_mode)
	    RGB32pixel *dat1 = (RGB32pixel *) get_pixel(0, y - 1);
	    for (unsigned int x = 0; x < width; x++)
	    {
		data_out << dat1->red << dat1->green << dat1->blue;
		dat1++;
	    }
	}
    }
    
    delete (ofstream*) ostream_ptr;
}


Image *RGB32Image::simple_difference(Image *image2, realno thresh, 
				     unsigned int *no_marked, Image *res)
{
    no_marked = 0;
    
    // rescale threshold for colour images
    unsigned int colour_threshold = (unsigned int) (thresh * 3.0);
    
    if (thresh < 0)
    {
	no_marked = 0;
	return difference(image2,0,res);
    }
    
    if (image2->get_width() != width || image2->get_height() != height)
    {
	cerror << " cannot difference different sized images !" << endl;
	exit(1);
    }
    if (res == NULL)
	res = new Grey8Image(width, height, frame_id, frame_time_in_ms);
    
    RGB32pixel *idat1;
    RGB32pixel *idat2 = (RGB32pixel*) image2->get_data();
    RGB32pixel *enddat = (RGB32pixel*) get_end_data();
    unsigned char *odat = res->get_data();
    
    for (idat1 = (RGB32pixel*) data ; idat1 < enddat; idat1++)
    {
	if ((abs(idat1->red - idat2->red) + 
	     abs(idat1->green - idat2->green) +
	     abs(idat1->blue - idat2->blue)) > colour_threshold ) 
	{
	    *odat = MARK;
	    no_marked++;
	}
	else
	    *odat = CLEAR_MARK;
	idat2++;odat++;
    }
    return res;
}

// quick differencing 
inline unsigned char RGB32Image::rgb_diff(RGB32pixel &pix1, RGB32pixel &pix2)
{
    register unsigned int ires = (abs(pix1.red - pix2.red)
				  + abs(pix1.green - pix2.green)
				  + abs(pix1.blue - pix2.blue)) / 3;
    return (unsigned char) ires;
}

inline void RGB32Image::get_detail(RGB32pixel *g1, RGB32pixel *g2,
				   unsigned char *r, unsigned int &detail_width)
{
    unsigned int lpx,lpy;
    RGB32pixel *tg1, *tg2;
    unsigned char *tr;
    for (lpy = 0; lpy < BOX_SIZE; lpy++)
    {
	tg1 = g1;
	tg2 = g2;
	tr = r;

	for (lpx = 0; lpx < BOX_SIZE; lpx++)
	    *r++ = rgb_diff(*g1++, *g2++);

	g1 = tg1 + detail_width;
	g2 = tg2 + detail_width;
	r = tr + detail_width;
    }
}

////  Threshold to grey (values MARK and CLEAR_MARK)
Image *RGB32Image::threshold(unsigned int threshold, Image *res,
			     unsigned int *no_marked)
{
    if (res == NULL)
	res = new Grey8Image(width, height, frame_id, frame_time_in_ms);
    
    // row and column counters...
    unsigned int row;
    unsigned int col;
    
    // pointers to data: columns of image and result
    RGB32pixel *image_col;
    unsigned char *result_col;
    
    register unsigned int sum;
    unsigned int colour_threshold = 3 * threshold;
    unsigned int mark_count = 0;
    
    for (row = 0; row < height; row++)
    {
	image_col  = (RGB32pixel*) get_pixel(0,row);
	result_col = res -> get_pixel(0,row);
	
	for (col = 0; col < width; col++)
	{
	    sum = (unsigned int) image_col->red
		+ (unsigned int) image_col->green
		+ (unsigned int) image_col->blue;
	    
	    image_col++;
	    
	    if (sum <= colour_threshold)
		*result_col++ = CLEAR_MARK;
	    else
	    {
		*result_col++ = MARK;
		mark_count++;
	    }
	}
    }
    
    if (no_marked != NULL)
	*no_marked = mark_count;
    
    return res;
}

////  "ColourFilter" to grey  ////  for experiments
Image *RGB32Image::ColourFilter (Image *res, ColourFilteringMethod method)
{
    if (res == NULL)
	res = new Grey8Image(width, height, frame_id, frame_time_in_ms);
    
    // row and column counters...
    unsigned int row;
    unsigned int col;
    
    // pointers to data: columns of image and result
    RGB32pixel *image_col;
    unsigned char *result_col;
    
    register signed int sgrey;
    
    float R;        float G;        float B;
    float Rdash;    float Gdash;    float Bdash;
    float H;        float S;        float V;
    float min_RGB;  float max_RGB;
    
    // FIXME: should put switch out of loop or use several inline
    //        functions (pointed to) or something.  No time.
    for (row = 0; row < height; row++)
    {
	image_col  = (RGB32pixel*) get_pixel(0,row);
	result_col = res -> get_pixel(0,row);
	
	for (col = 0; col < width; col++)
	{
	    switch (method)
	    {
// Simple norm methods
	    case CF_METHOD_1_NORM:   //  1-norm of (R,G,B)
	    {
		sgrey = ((unsigned int) image_col->red
			 + (unsigned int) image_col->green
			 + (unsigned int) image_col->blue
			 + 1)
		    / 3;
		break;
	    }
	    
	    case CF_METHOD_2_NORM:   //  2-norm of (R,G,B)
	    {
		sgrey = (unsigned int)
		    (sqrt( ((float)
			    ((SQUARE ((unsigned int) image_col->red))
			     +(SQUARE ((unsigned int) image_col->green))
			     +(SQUARE ((unsigned int) image_col->blue))))
			   / 3));
		break;
	    }
	    
	    case CF_METHOD_MAX_NORM: //  max-norm of (R,G,B)
	    {
		sgrey = MAX(MAX((unsigned int) image_col->red,
				(unsigned int) image_col->green),
			    (unsigned int) image_col->blue);	
		break;
	    }
	    
// HSV after D Travis
	    case CF_METHOD_H:        //  H of HSV after D Travis
	    case CF_METHOD_S:        //  S of HSV after D Travis
	    case CF_METHOD_V:        //  V of HSV after D Travis
	    {
		// calculate HSV (after D Travis)
		
		R = ((float) (unsigned int) image_col->red) / 255;
		G = ((float) (unsigned int) image_col->green) / 255;
		B = ((float) (unsigned int) image_col->blue) / 255;
		
		min_RGB = MIN(MIN(R,G),B);
		max_RGB = MAX(MAX(R,G),B);
		
		S = (max_RGB == 0) ? 1 : (max_RGB - min_RGB) / max_RGB;
		V = max_RGB;
		
		if (S == 0)
		    H = 0;  // in fact, undefined (colour has no hue)
		else
		{
		    Rdash = (max_RGB - R) / (max_RGB - min_RGB);
		    Gdash = (max_RGB - G) / (max_RGB - min_RGB);
		    Bdash = (max_RGB - B) / (max_RGB - min_RGB);
		    
		    if ((R == max_RGB) && (G == min_RGB))
			H = 5 + Bdash;
		    else
			if (R == max_RGB)
			    H = 1 - Gdash;
			else
			    if ((G == max_RGB) && (B == min_RGB))
				H = 1 + Rdash;
			    else
				if (G == max_RGB)
				    H = 3 - Bdash;
				else
				    if (R == max_RGB)
					H = 3 + Gdash;
				    else
					H = 5 - Rdash;
		} // gives H \in [0..6]
		H = H / 6;
		
		if (method == CF_METHOD_H)
		    sgrey = (int) (255 * H);
		else
		    if (method == CF_METHOD_S)
			sgrey = (int) (255 * S);
		    else
			sgrey = (int) (255 * V);
		break;
	    }
	    
// Chrominance and luminance according to CIE recommendations 601 and 709
	    case CF_METHOD_Y_601:    // Y  from YCbCr after CIE recommendation 601
	    {
		sgrey = 16 +
		    (262 * (unsigned int) image_col->red
		     + 514 * (unsigned int) image_col->green
		     + 100 * (unsigned int) image_col->blue)
			 >> 10;
		break;
	    }
	    case CF_METHOD_CB_601:   // Cb from YCbCr after CIE recommendation 601
	    {	
		sgrey = 128 +
		    (-151 * (unsigned int) image_col->red
		     + 297 * (unsigned int) image_col->green
		     + 448 * (unsigned int) image_col->blue)
			 >> 10;
		break;
	    }
	    case CF_METHOD_CR_601:   // Cr from YCbCr after CIE recommendation 601
	    {
		sgrey = 128 +
		    (448 * (unsigned int) image_col->red
		     - 375 * (unsigned int) image_col->green
		     - 73 * (unsigned int) image_col->blue)
			 >> 10;
		break;
	    }
	    case CF_METHOD_Y_709:     // Y  from YCbCr after CIE recommendation 709
	    {
		//  source: C Poynton's Colour FAQ page 6
		sgrey = (218 * (unsigned int) image_col->red
			 + 732 * (unsigned int) image_col->green
			 + 74 * (unsigned int) image_col->blue)
			     >> 10;	
		break;
	    }
	    
	    default:
	    {
		cerror << " RGB32Image::ColourFilter: unknown filtering method requested " << endl;
		exit(1);
	    }
	    }
	    
	    
	    if (sgrey > 255)
		*result_col++ = 255;
	    else
		if (sgrey < 0)
		    *result_col++ = 0;
		else
		    *result_col++ = (unsigned char) sgrey;
	    
	    image_col++;
	}
    }
    
    return res;
}   

// These are more algorithms for the colour filtering methods above...
// See my report (or my thesis) for a review of the most interesting methods --- nts.

//  // 4th root ( sum ( differences R-G, R-B and G-B each ** 4 ) )
//         	    grey = (unsigned int)
//        		sqrt(sqrt( (float)
//        		      ((SQUARE (SQUARE ((unsigned int) image_col->red
//        					- (unsigned int) image_col->green))
//        			+ SQUARE (SQUARE ((unsigned int) image_col->red
//        					  - (unsigned int) image_col->blue))
//        			+ SQUARE (SQUARE ((unsigned int) image_col->green
//        					  - (unsigned int) image_col->blue)))
//        		       / 3)));

//  // root squared sum differences R-G, R-B and G-B
//           	    grey = (unsigned int)
//          		sqrt( (float)
//          		      ((SQUARE ((unsigned int) image_col->red
//          				- (unsigned int) image_col->green)
//          			+ SQUARE ((unsigned int) image_col->red
//          				  - (unsigned int) image_col->blue)
//          			+ SQUARE ((unsigned int) image_col->green
//          				  - (unsigned int) image_col->blue))
//          		       / 3));

//  // Y with inverted (then normalised) weigths (original Y after Rec. 709)
//       	    grey = (241 * (unsigned int) image_col->red
//        		+ 72 * (unsigned int) image_col->green
//        		+ 711 * (unsigned int) image_col->blue)
//        		    >> 10;

//  // Y with inverted (then normalised) weigths (original Y after Rec. 601)
//        	    sgrey = 16 +
//        		(262 * (unsigned int) image_col->red
//     not finished   		 + 514 * (unsigned int) image_col->green
//        		 + 100 * (unsigned int) image_col->blue)
//  		     >> 10;

//  // this is Y (after Rec. 601)
//          	    sgrey = 16 +
//          		(262 * (unsigned int) image_col->red
//          		 + 514 * (unsigned int) image_col->green
//          		 + 100 * (unsigned int) image_col->blue)
//    		     >> 10;


//  // this is C_B (after Rec. 601)
//          	    sgrey = 128 +
//          		(-151 * (unsigned int) image_col->red
//          		 + 297 * (unsigned int) image_col->green
//          		 + 448 * (unsigned int) image_col->blue)
//          		    >> 10;

//  // this is C_R (after Rec. 601)
//          	    sgrey = 128 +
//          		(448 * (unsigned int) image_col->red
//          		 - 375 * (unsigned int) image_col->green
//          		 - 73 * (unsigned int) image_col->blue)
//          		    >> 10;




////  colour differencing (no filtering or thresholding), absolute
Image *RGB32Image::difference(Image *image2, Image *res)
{
    if ((this == NULL) || (image2 == NULL))
	return NULL;
    
    if (image2->get_image_type() != RGB32)
    {
	if (image2->get_image_type() == GREY8)
	    return RGB32Image::difference((Grey8Image*) image2, res);
	
	cerror << "RGB32Image::difference(): cannot difference different image types. " << endl;
	exit(1);
    }
    
    if (res == NULL)
	res = new RGB32Image(width, height, frame_id, frame_time_in_ms);
    
    // row and column counters...
    unsigned int row;
    unsigned int col;
    
    // pointers to data: columns in both images, and the result
    RGB32pixel *image1_col;
    RGB32pixel *image2_col;
    RGB32pixel *result_col;
    
    RGB32pixel temp_result;
    temp_result.alpha = 0x00;
    
    for (row = 0; row < height; row++)
    {
	image1_col = (RGB32pixel*)           get_pixel(0,row);
	image2_col = (RGB32pixel*) image2 -> get_pixel(0,row);
	result_col = (RGB32pixel*) res    -> get_pixel(0,row);
	
	for (col = 0; col < width; col++)
	{
	    temp_result.red  = (image1_col->red > image2_col->red) ?
		(image1_col->red - image2_col->red) :
		(image2_col->red - image1_col->red);
	    
	    temp_result.green  = (image1_col->green > image2_col->green) ?
		(image1_col->green - image2_col->green) :
		(image2_col->green - image1_col->green);
	    
	    temp_result.blue = (image1_col->blue > image2_col->blue) ?
		(image1_col->blue - image2_col->blue) :
		(image2_col->blue - image1_col->blue);
	    
	    *result_col++ = temp_result;
	    image1_col++;
	    image2_col++;
	}
    }
    
    return res;
}

////  colour differencing --- version for RGB32-GREY8
Image *RGB32Image::difference(Grey8Image *image2, Image *res)
{
    if ((this == NULL) || (image2 == NULL))
	return NULL;
    
    if (image2->get_image_type() != GREY8)
    {
	cerror << "RGB32Image::difference(): cannot difference these image types. " << endl;
	exit(1);
    }
    
    if (res == NULL)
	res = new RGB32Image(width, height, frame_id, frame_time_in_ms);
    
    // row and column counters...
    unsigned int row;
    unsigned int col;
    
    // pointers to data: columns in both images, and the result
    RGB32pixel *image1_col;
    unsigned char *image2_col;
    RGB32pixel *result_col;
    
    RGB32pixel temp_result;
    temp_result.alpha = 0x00;

    for (row = 0; row < height; row++)
    {
	image1_col = (RGB32pixel*)           get_pixel(0,row);
	image2_col =               image2 -> get_pixel(0,row);
	result_col = (RGB32pixel*) res    -> get_pixel(0,row);
	
	for (col = 0; col < width; col++)
	{
	    temp_result.red = (image1_col->red > *image2_col) ?
		(image1_col->red - *image2_col) :
		(*image2_col - image1_col->red);
	    
	    temp_result.green = (image1_col->green > *image2_col) ?
		(image1_col->green - *image2_col) :
		(*image2_col - image1_col->green);
	    
	    temp_result.blue = (image1_col->blue > *image2_col) ?
		(image1_col->blue - *image2_col) :
		(*image2_col - image1_col->blue);
	    
	    *result_col++ = temp_result;
	    image1_col++;
	    image2_col++;	    
	}
    }
    
    return res;
}


/////   differencing and thresholding and some filtering (check BOX_SIZE) /////
Image *RGB32Image::difference(Image *image2, realno diff_threshold, Image *res)
{
    if ((this == NULL) || (image2 == NULL))
	return NULL;
    
    if (image2->get_image_type() != RGB32)
    {
	cerror << " can't difference different image types! " << endl;
	exit(1);
    }
    
    if (res == NULL)
	res = new Grey8Image(width, height, frame_id, frame_time_in_ms);
    
    res->clear(CLEAR_MARK);
    
    RGB32pixel *g1 = (RGB32pixel*)  data;
    RGB32pixel *g2 = (RGB32pixel*) image2->get_data();
    
    RGB32pixel *tg1, *tg2;
    
    unsigned char *rdat = res->get_data();
    unsigned char *trdat;
    
    int up = BOX_SIZE * width;
    int left = BOX_SIZE;
    
    int upleft  = left + up;
    int upleft2 = upleft / 2;
    int upright = up - left;
    
    g1   += (upleft2) * 3;
    g2   += (upleft2) * 3;
    rdat += (upleft2) * 3;
    
    int lx;  int ly;
    int lymax = height / BOX_SIZE;
    int lxmax = width / BOX_SIZE; 
    
    int interest_threshold = (int) diff_threshold;
    
    int adj1 = upleft2;
    int adj2 = left+adj1;
    int adj3 = up+adj1;
    int adj4 = upleft+adj1;
    int adj5 = adj1-left;
    int adj6 = adj1-up;
    int adj7 = adj1-upleft;
    int	adj8 = adj1-upright;
    int adj9 = adj1+upright;
    
    //lxmax -=1 ;lymax -=1;
    Image *flag_img = new Grey8Image(lxmax, lymax, frame_id, frame_time_in_ms);
    flag_img->clear(CLEAR_MARK);
    for (ly = 1; ly < lymax; ly++)
    {
	tg1 = g1; tg2 = g2; 
	trdat = rdat;
	for (lx = 1; lx < lxmax; lx++)
	{
	    *rdat = rgb_diff(*g1,*g2);
	    if (*rdat >= interest_threshold)
	    {
		register unsigned char *c;
		// this square
		if (flag_img->check_coords(lx,ly))
		{
		    c = flag_img->get_pixel(lx,ly);
		    if (*c == CLEAR_MARK)
		    {
			get_detail(g1-adj1,g2-adj1, rdat-adj1, width);
			*c = 0xff;
		    }
		}
		// left
		if (flag_img->check_coords(lx-1,ly))
		{
		    c = flag_img->get_pixel(lx-1,ly);
		    if (*c == CLEAR_MARK)
		    {
			get_detail(g1-adj2,g2-adj2, rdat-adj2, width);
			*c = 0xff;
		    }
		}
		// up
		if (flag_img->check_coords(lx,ly-1))
		{
		    c = flag_img->get_pixel(lx,ly-1);
		    if (*c == CLEAR_MARK)
		    {
			get_detail(g1-adj3,g2-adj3, rdat-adj3, width);
			*c = 0xff;
		    }
		}
		// up-left
		if (flag_img->check_coords(lx-1,ly-1))
		{
		    c = flag_img->get_pixel(lx-1,ly-1);
		    if (*c == CLEAR_MARK)
		    {
			get_detail(g1-adj4,g2-adj4, rdat-adj4, width);
			*c = 0xff;
		    }
		}
		// right
		if (flag_img->check_coords(lx+1,ly))
		{
		    c = flag_img->get_pixel(lx+1,ly);
		    if (*c == CLEAR_MARK)
		    {
			get_detail(g1-adj5,g2-adj5, rdat-adj5, width);
			*c = 0xff;
		    }
		}
		// down
		if (flag_img->check_coords(lx,ly+1))
		{
		    c = flag_img->get_pixel(lx,ly+1);
		    if (*c == CLEAR_MARK)
		    {
			get_detail(g1-adj6,g2-adj6, rdat-adj6, width);
			*c = 0xff;
		    }
		}
		// down-right
		if (flag_img->check_coords(lx+1,ly+1))
		{
		    c = flag_img->get_pixel(lx+1,ly+1);
		    if (*c == CLEAR_MARK)
		    {
			get_detail(g1-adj7,g2-adj7, rdat-adj7, width);
			*c = 0xff;
		    }
		}
		// down-left
		if (flag_img->check_coords(lx-1,ly+1))
		{
		    c = flag_img->get_pixel(lx-1,ly+1);
		    if (*c ==CLEAR_MARK)
		    {
			get_detail(g1-adj8,g2-adj8, rdat-adj8, width);
			*c = 0xff;
		    }
		}
		// up-right
		if (flag_img->check_coords(lx+1,ly-1))
		{
		    c = flag_img->get_pixel(lx+1,ly-1);
		    if (*c == CLEAR_MARK)
		    {
			get_detail(g1-adj9,g2-adj9, rdat-adj9, width);
			*c = 0xff;
		    }
		}
	    } 
	    g1 += left; g2 += left;
	    rdat += left;
	}
	g1 = tg1 + up;
	g2 = tg2 + up;
	rdat = trdat + up;	
    }
    
    delete flag_img;
    return res;
}

#ifdef HSV
Image *RGB32Image::to_HSV32(Image *res)
{
    if (res == NULL)
	res = new HSV32Image(width, height, frame_id, frame_time_in_ms);
    else
	if (res->get_image_type() != HSV32)
	{
	    cerror << " RGB32Image::to_HSV32() conversion: result type must be HSV32 "
		   << endl;
	    exit(1);
	}
    
    RGB32pixel *idat;
    HSV32pixel *odat;
    
    for (unsigned int y = 0; y < height; y++)
    {
	// initialise line pointers
	idat = (RGB32pixel*) get_pixel(0,y);
	odat = (HSV32pixel*) res->get_pixel(0,y);
	
	for (unsigned int x = 0; x < width; x++)
	    *odat++ = HSV32Image::to_HSV32_using_lookup(idat++);
    }
    
    return res;
}
#endif // ifdef HSV

unsigned int RGB32Image::measure_contrast(void *p1, void *p2)
{
    RGB32pixel *pix1 = (RGB32pixel*) p1;
    RGB32pixel *pix2 = (RGB32pixel*) p2;
    if ((pix1 == NULL) || (pix2 == NULL))
	return 0;
    unsigned int r1 = (unsigned int) pix1->red;   unsigned int r2 = (unsigned int) pix2->red;
    unsigned int g1 = (unsigned int) pix1->green; unsigned int g2 = (unsigned int) pix2->green;
    unsigned int b1 = (unsigned int) pix1->blue;  unsigned int b2 = (unsigned int) pix2->blue;
    
    return abs(r1 - r2) + abs(g1 - g2) + abs(b1 - b2);
}

Image *RGB32Image::resample(unsigned int xstep, unsigned int ystep, Image *resampled)
{
    unsigned int new_width = (width / xstep);
    if (resampled == NULL)
	resampled = (Image *) new RGB32Image(new_width, height / ystep,
					     frame_id, frame_time_in_ms);
    RGB32pixel *idat;
    RGB32pixel *odat;
    unsigned int outy = 0;
    for (unsigned int y = 0; y < height; y += ystep)
    {
	idat = (RGB32pixel*) get_pixel(0,y);
	odat = (RGB32pixel*) resampled->get_pixel(0, outy);
	for (unsigned int x = 0; x < new_width; x ++)
	{
	    *odat = *idat;
	    odat++;
	    idat = idat + xstep;
	}
	outy++;
    }
    return resampled;
}

Image *RGB32Image::convolve(Kernel k, Image *convolved)
{
    RGB32pixel *upnt, *pnt, *dpnt;
    RGB32pixel *out;
    int k11 = k.coeff[0][0], k12 = k.coeff[0][1], k13 = k.coeff[0][2]; 
    int k21 = k.coeff[1][0], k22 = k.coeff[1][1], k23 = k.coeff[1][2]; 
    int k31 = k.coeff[2][0], k32 = k.coeff[2][1], k33 = k.coeff[2][2];
    int ksum = abs(k11)+abs(k12)+abs(k13)
	+abs(k21)+abs(k22)+abs(k23)
	+abs(k31)+abs(k32)+abs(k33);
    if (convolved == NULL)
	convolved = new RGB32Image(width, height, frame_id, frame_time_in_ms);
    //convolved->clear_border();
    for (int y = 1; y < (height-1); y++)
    {
	upnt = (RGB32pixel*) get_pixel(0,y-1);
	pnt = (RGB32pixel*) get_pixel(0,y);
	dpnt = (RGB32pixel*) get_pixel(0,y+1);
	out = (RGB32pixel*) convolved->get_pixel(1,y);
	for (int x = 1; x < (width-1); x++)
	{
	    out->red = abs
		(k11 * upnt->red + k12 * upnt[1].red + k13 * upnt[2].red
		 + k21 * pnt->red + k22 * pnt[1].red + k23 * pnt[2].red
		 + k31 * dpnt->red + k32 * dpnt[1].red + k33 *
		 dpnt[2].red) / ksum;
	    out->green = abs
		(k11 * upnt->green + k12 * upnt[1].green + k13 * upnt[2].green
		 + k21 * pnt->green + k22 * pnt[1].green + k23 * pnt[2].green
		 + k31 * dpnt->green + k32 * dpnt[1].green + k33 *
		 dpnt[2].green) / ksum;  
	    out->blue = abs
		(k11 * upnt->blue + k12 * upnt[1].blue + k13 * upnt[2].blue
		 + k21 * pnt->blue + k22 * pnt[1].blue + k23 * pnt[2].blue
		 + k31 * dpnt->blue + k32 * dpnt[1].blue + k33 *
		 dpnt[2].blue) / ksum;
	    
	    upnt++; pnt++; dpnt++; out++;
	}
    }
    return convolved;
}

Image *RGB32Image::sobel(unsigned int sobel_threshold, Image *res)
{
    unsigned int sqthresh3 = 3 * sobel_threshold * sobel_threshold;
    
    if (res == NULL)
	res = new Grey8Image(width, height, frame_id, frame_time_in_ms);
    
    Image *hsobel = convolve(SOBELH_KER, NULL);
    Image *vsobel = convolve(SOBELV_KER, NULL);
    unsigned char *odat, *enddat;
    RGB32pixel *hdat, *vdat;
    hdat = (RGB32pixel*) hsobel->get_data();
    vdat = (RGB32pixel*) vsobel->get_data();
    enddat = res->get_end_data();
    for (odat = res->get_data(); odat < enddat; odat++)
    {
	RGB32pixel &hpix = *hdat;
	RGB32pixel &vpix = *vdat;
	int r1 = hpix.red * hpix.red + vpix.red * vpix.red;
	int g1 = hpix.green * hpix.green + vpix.green * vpix.green;
	int b1 = hpix.blue * hpix.blue + vpix.blue * vpix.blue;
	
	if ((r1 + g1 + b1) > sqthresh3)
	    *odat = MARK;
	else
	    *odat = CLEAR_MARK;
	
	hdat++; vdat++;
    }
    delete vsobel;
    delete hsobel;
    return res;
}

Image *RGB32Image::sobel(Image *res)
{
    if (res == NULL)
	res = new Grey8Image(width, height, frame_id, frame_time_in_ms);
    
    Image *hsobel = convolve(SOBELH_KER, NULL);
    Image *vsobel = convolve(SOBELV_KER, NULL);
    unsigned char *odat, *enddat;
    RGB32pixel *hdat, *vdat;
    hdat = (RGB32pixel*) hsobel->get_data();
    vdat = (RGB32pixel*) vsobel->get_data();
    enddat = res->get_end_data();
    for (odat = res->get_data(); odat < enddat; odat++)
    {
	RGB32pixel &hpix = *hdat;
	RGB32pixel &vpix = *vdat;
	int r1 = hpix.red * hpix.red + vpix.red * vpix.red;
	int g1 = hpix.green * hpix.green + vpix.green * vpix.green;
	int b1 = hpix.blue * hpix.blue + vpix.blue * vpix.blue;
	
	int pix = (int) sqrt((float) (r1+g1+b1) / 6);
	*odat = pix;
	
	hdat++; vdat++;
    }
    delete vsobel;
    delete hsobel;
    return res;
}

inline int RGB32Image::compar_char(const void *c1, const void *c2)
{
    return (int) (*((unsigned char *) c1) - *((unsigned char *) c2));
}


void RGB32Image::median_img_array(Image **img_array, unsigned int asize)
{
    if (img_array[0]->get_image_type() != RGB32)
    {
	cerror << " bad call to RGB32Image::median_img_array " << endl;
	abort();
    }
    
    unsigned char *rtable = new unsigned char[asize];
    unsigned char *gtable = new unsigned char[asize];
    unsigned char *btable = new unsigned char[asize];
    
    unsigned int cpos = (asize + 1) / 2;
    unsigned int i,j,k;
    for (i = 0; i < width; i++)
	for (j = 0; j < height; j++)
	{
	    for (k = 0; k < asize; k++)
	    {
		RGB32pixel *rgb_pix = (RGB32pixel*)
		    img_array[k]->get_pixel(i,j);
		rtable[k] = rgb_pix->red;
		gtable[k] = rgb_pix->green;
		btable[k] = rgb_pix->blue;
	    }
	    
	    qsort(rtable, asize, sizeof(unsigned char), &compar_char);
	    qsort(gtable, asize, sizeof(unsigned char), &compar_char);
	    qsort(btable, asize, sizeof(unsigned char), &compar_char);
	    
	    RGB32pixel *rgb_pix = (RGB32pixel*) get_pixel(i,j);
	    rgb_pix->red = rtable[cpos];
	    rgb_pix->green = gtable[cpos];
	    rgb_pix->blue = btable[cpos];
	}
    
    delete [] rtable;
    delete [] gtable;
    delete [] btable;
}


Image *RGB32Image::half_blur(Image *res)
{
    unsigned int new_width = width  / 2;
    unsigned int new_height = height / 2;
    
    if (res == NULL)
	res = new RGB32Image(new_width, new_height, frame_id, frame_time_in_ms);
    
    RGB32pixel *odat = (RGB32pixel*) res->get_data();
    for (unsigned int y = 0; y < new_height; y++)
    {
	RGB32pixel *in_even = (RGB32pixel*) get_pixel(0,y * 2);
	RGB32pixel *in_odd = (RGB32pixel*) get_pixel(0,y * 2 + 1);
	for (unsigned int x = 0; x < new_width; x++)
	{
	    unsigned int r1 = in_even->red + in_odd->red;
	    unsigned int g1 = in_even->green + in_odd->green;
	    unsigned int b1 = in_even->blue + in_odd->blue;
	    in_even++; in_odd++;
	    
	    r1 += in_even->red + in_odd->red;
	    g1 += in_even->green + in_odd->green;
	    b1 += in_even->blue + in_odd->blue;
	    
	    odat->red = r1 >> 2;
	    odat->green = g1 >> 2;
	    odat->blue = b1 >> 2;
	    
	    in_even++; in_odd++;
	    odat++;
	}
    }
    return res;
}

Image *RGB32Image::image_blend(Image *i2, unsigned int a, unsigned int b, Image *res)
{
    if (res == NULL)
	res = new RGB32Image(width, height, frame_id, frame_time_in_ms);
    
    RGB32pixel *idat1  = (RGB32pixel*) data;
    RGB32pixel *idat2  = (RGB32pixel*) i2->get_data();
    RGB32pixel *odat   = (RGB32pixel*) res->get_data();
    RGB32pixel *enddat = (RGB32pixel*) get_end_data();
    
    unsigned int pixr, pixg, pixb;
    unsigned int aplusb = a + b;
    if (i2->get_image_type() == RGB32)
    {
	for ( ; idat1 < enddat; idat1++)
	{
	    pixr = (a * idat1->red) + (b * idat2->red);
	    pixg = (a * idat1->green) + (b * idat2->green);
	    pixb = (a * idat1->blue) + (b * idat2->blue);
	    odat->red = (pixr / aplusb);
	    odat->green = (pixg / aplusb);
	    odat->blue = (pixb / aplusb);
	    
	    idat2++; odat++;
	    
	}
    }
    else
	if (i2->get_image_type() == GREY8)
	{
	    unsigned char *greydat = i2->get_data();
	    for ( ; idat1 < enddat; idat1++)
	    {
		pixr = (a * idat1->red) + (b * *greydat);
		pixg = (a * idat1->green) + (b * *greydat);
		pixb = (a * idat1->blue) + (b * *greydat);
		odat->red = (pixr / aplusb);
		odat->green = (pixg / aplusb);
		odat->blue = (pixb / aplusb);
		
		greydat++; odat++;
		
	    }
	}
	else 
	{
	    cerror << "bad call to RGB32Image::image_blend " << endl;
	    abort();
	}
    return res;
}

Image *RGB32Image::transform(unsigned int centre_x, unsigned int centre_y,
			     realno angle, realno scale, Image *res)
{
    if (res == NULL)
	res = copy_type();
    
    RGB32pixel *src, *dest;
    unsigned int x_from, y_from;
    realno cosa = cos(angle) / scale;
    realno sina = sin(angle) / scale;
    const unsigned int &dw = res->get_width();
    const unsigned int &dh = res->get_height();
    unsigned int dw2 = dw / 2;
    unsigned int dh2 = dh / 2;
    for (unsigned int y = 0; y < dh; ++y) 
    {
	dest = (RGB32pixel*) res->get_pixel(0,y);
	for (unsigned int x = 0; x < dw; ++x) 
	{
	    unsigned int x1 = x - dw2;
	    unsigned int y1 = y - dh2;
	    
	    x_from = (unsigned int) (centre_x + x1 * cosa + y1 * sina);
	    y_from = (unsigned int) (centre_y + y1 * cosa - x1 * sina);
	    
	    if ((x_from < 0) || (y_from < 0) || (x_from >= width) 
		|| (y_from >= height))
		dest->red = dest->green = dest->blue = 0;
	    else
	    {
		src = (RGB32pixel*) get_pixel(x_from, y_from);
		dest->red = src->red;
		dest->green = src->green;
		dest->blue = src->blue;
	    }
	    dest++;
	}
    }
    return res;
}


Image *RGB32Image::map_intensities(PntGreyMap gmap, Image *res)
{
    if (res == NULL)
	res = new RGB32Image(width, height, frame_id, frame_time_in_ms);
    
    RGB32pixel *idat = (RGB32pixel*) data;
    RGB32pixel *odat = (RGB32pixel*) res->get_data();
    RGB32pixel *enddat = (RGB32pixel*) get_end_data();
    
    for ( ; idat < enddat; idat++)
    {
	odat->red = gmap[idat->red];
	odat->green = gmap[idat->green];
	odat->blue = gmap[idat->blue];
	odat++;
    }
    return res;
}

void RGB32Image::clear(const RGB32pixel fill_colour)
{
    RGB32pixel *current;
    RGB32pixel *end = (RGB32pixel *) end_data;

    for (current = (RGB32pixel *) data; current < end; *current++ = fill_colour) ;
}

void RGB32Image::clear_border()
{
    register unsigned int w1 = width-1;
    for (register unsigned int y = 0; y < height; y++)
    {
	*((RGB32pixel*) get_pixel( 0,y)) = to_RGB32(CLEAR_MARK);
	*((RGB32pixel*) get_pixel(w1,y)) = to_RGB32(CLEAR_MARK);
    }
    register unsigned int h1 = height-1;
    for (register unsigned int x = 0; x < width; x++)
    {
	*((RGB32pixel*) get_pixel(x, 0)) = to_RGB32(CLEAR_MARK);
	*((RGB32pixel*) get_pixel(x,h1)) = to_RGB32(CLEAR_MARK);
    }
}

Image *RGB32Image::mask(Image *mask, Image *res)
{
    if (mask == NULL)
	return NULL;
    
    if (res == NULL)
	res = new RGB32Image(width, height, frame_id, frame_time_in_ms);
    
    if ((mask->get_width() != width) || (mask->get_height() != height))
    {
	register unsigned long *mask_pix;
	register unsigned int x;
	unsigned int y;
	for (y = 0; y < height; y++)
	    for (x = 0; x < width; x++)
	    {
		// FIXME: Should use RGB32pixel here, as well.
		if (mask->check_coords(x,y))
		{
		    mask_pix = (unsigned long*) mask->get_pixel(x,y);
		    if (*mask_pix != 0x00000000)
			*((unsigned long*) res->get_pixel(x,y))
			    = *((unsigned long*) get_pixel(x,y));
		}
	    }
    }
    else 
    {
	// FIXME: Should use RGB32pixel here, as well.
	unsigned long *dat1 = (unsigned long*) data;
	unsigned long *dat2 = (unsigned long*) mask->get_data();
	unsigned long *edat = (unsigned long*) get_end_data();
	register unsigned long *rdat = (unsigned long*) res->get_data();
	while (dat1 < edat)
	    *rdat++ = *dat1++ & *dat2++;
    }
    return res;
}

Image *RGB32Image::neighbour_order(unsigned int n, Image *ordered)
// FIXME: are we using the right colour ordering/comparison?
{
    if (ordered == NULL)
	ordered = new RGB32Image (width, height, frame_id, frame_time_in_ms);
    
    ordered->clear_border();
    unsigned char rtable[9];
    unsigned char gtable[9];
    unsigned char btable[9];
    RGB32pixel *upnt;
    RGB32pixel  *pnt;
    RGB32pixel *dpnt;
    RGB32pixel  *out;
    unsigned int ntake1 = n - 1;
    for (unsigned int y = 1; y < height-1; y++)
    {
	upnt = (RGB32pixel*) get_pixel(0,y-1);
	pnt  = (RGB32pixel*) get_pixel(0,y);
	dpnt = (RGB32pixel*) get_pixel(0,y+1);  
	out  = (RGB32pixel*) ordered->get_pixel(1,y);
	for (unsigned int x = 1; x < width-1; x++)
	{
	    rtable[0] = upnt[0].red; rtable[1] = upnt[1].red; rtable[2] = upnt[2].red;
	    rtable[3] =  pnt[0].red; rtable[4] =  pnt[1].red; rtable[5] =  pnt[2].red;
	    rtable[6] = dpnt[0].red; rtable[7] = dpnt[1].red; rtable[8] = dpnt[2].red;
	    
	    gtable[0] = upnt[0].green; gtable[1] = upnt[1].green; gtable[2] = upnt[2].green;
	    gtable[3] =  pnt[0].green; gtable[4] =  pnt[1].green; gtable[5] =  pnt[2].green;
	    gtable[6] = dpnt[0].green; gtable[7] = dpnt[1].green; gtable[8] = dpnt[2].green;
	    
	    btable[0] = upnt[0].blue; btable[1] = upnt[1].blue; btable[2] = upnt[2].blue;
	    btable[3] =  pnt[0].blue; btable[4] =  pnt[1].blue; btable[5] =  pnt[2].blue;
	    btable[6] = dpnt[0].blue; btable[7] = dpnt[1].blue; btable[8] = dpnt[2].blue;
	    
	    qsort(rtable, 9, sizeof(unsigned char), &compar_char);
	    qsort(gtable, 9, sizeof(unsigned char), &compar_char);
	    qsort(btable, 9, sizeof(unsigned char), &compar_char);
	    
	    out->red   = rtable[ntake1];
	    out->green = gtable[ntake1];
	    out->blue  = btable[ntake1];
	    out->alpha = 0x00;  // FIXME: alpha might get lost here
	    upnt++; pnt++; dpnt++; out++;
	}
    }
    return ordered;
}


RGB32Image *RGB32Image::rgb_read_image(RGB32Image *res)
{    
#ifdef NO_DISPLAY
    // mission impossible
    cerror << "RGB32Image::rgb_read_image not possible without display " << endl;
    abort();
    return NULL;
#else
    
#ifdef USE_GL
    Int32 xsize;
    Int32 ysize;
    winset(glwin);
    getsize(&xsize, &ysize);
    
    if (res == NULL)
	res = new RGB32Image(xsize, ysize, frame_id, frame_time_in_ms);
    
    lrectread(0,0,xsize-1,ysize-1, (Int32*) res->get_data());
    
    return res;
#else
    // to be implemented !
    cerror << "RGB32Image::rgb_read_image not implemented on X11 " << endl;
    abort();
    return NULL;
    
#endif
    
#endif   // #ifdef NO_DISPLAY  #else
}

inline unsigned long RGB32Image::lookup_index(unsigned long l)
{
/*
  // Easy-to-read version:
  int r = (l & HIST_MASK_RED) >> HIST_SHIFT_RED;
  int g = (l & HIST_MASK_GREEN) >> HIST_SHIFT_GREEN;
  int b = (l & HIST_MASK_BLUE) >> HIST_SHIFT_BLUE;
  unsigned long ind = r | g | b;
  return ind;
*/
    // Optimised version:
    return ((l & HIST_MASK_RED) >> HIST_SHIFT_RED) |
	((l & HIST_MASK_GREEN) >> HIST_SHIFT_GREEN) |
	((l & HIST_MASK_BLUE) >> HIST_SHIFT_BLUE);
}

// RGB32Image::greyscale_using_lookup
Grey8Image *RGB32Image::greyscale_using_lookup(unsigned char *lookup,
					       Grey8Image *dest)
{
    if (dest == NULL)
	dest = new Grey8Image(width, height, frame_id, frame_time_in_ms);
    
    unsigned long *idat = (unsigned long *) data;
    unsigned char *odat = (unsigned char*) dest->get_data();
    
    for (unsigned int i = 0; i < width*height; ++i)
    {
	unsigned long index = lookup_index(idat[i]);
	unsigned char o = lookup[index];
	odat[i] = o;
    }
    return dest;
}

void RGB32Image::draw_rectangle(int xmin, int xmax, int ymin, int ymax, unsigned int val)
{
    // consistency check
    if ((xmin > xmax) || (ymin > ymax))
    {
	cerror << " RGB32Image::draw_rectangle("
	       << xmin << "," << xmax << "," << ymin << "," << ymax
	       << ", " << val << ") : cannot draw! " << endl;
	return;
    }
    
    // graciously accept some rectangles which are partly out of image...
    if (xmin < 0)
	xmin = 0;
    if (ymin < 0)
	ymin = 0;
    if (xmax >= width)
	xmax = (width-1);
    if (ymax >= height)
	ymax = (height-1);
    
    unsigned int nbytes = (xmax - xmin + 1) << 2; 
    for (int y = ymin; y <= ymax; y++)
	memset(get_pixel(xmin,y), val, nbytes);
}


inline void RGB32Image::set_draw_colour(unsigned char red, unsigned char green,
					unsigned char blue)
{
    RGB32_draw_colour.red = red;
    RGB32_draw_colour.green = green;
    RGB32_draw_colour.blue = blue;
}

inline void RGB32Image::set_draw_colour(unsigned char grey_value)
{
    RGB32_draw_colour.red = RGB32_draw_colour.green =
	RGB32_draw_colour.blue = grey_value;
}

void RGB32Image::draw_horizontal(unsigned int y, unsigned int xl, unsigned int xr,
				 bool dotted)
{
    if ((check_coords(xl,y) == false) ||
	(check_coords(xr,y) == false))
	return;
    
    RGB32pixel *pixleft = (RGB32pixel*) get_pixel(xl, y);
    RGB32pixel *pixright = (RGB32pixel*) get_pixel(xr,y);
    
    while (pixleft <= pixright)
    {
	*pixleft++ = RGB32_draw_colour;
	
	if (dotted)
	    pixleft++;  // skip another pixel
    }
}

void RGB32Image::draw_vertical(unsigned int x, unsigned int yl, unsigned int yu,
			       bool dotted)
{
    if ((check_coords(x,yl) == false) ||
	(check_coords(x,yu) == false))
	return;
    
    if (yl > yu)
    {
	// exchange values if coordinates are upside down
	unsigned int tmp = yl;
	yl = yu;
	yu = tmp;
    }
	
    for (unsigned int y = yl; y <= yu; y++)
    {
	if ((dotted) && ((y & 1) == 1))
	    continue;
	* (RGB32pixel*) get_pixel(x,y) = RGB32_draw_colour;
    }
}

void RGB32Image::plot_pixel(unsigned int x, unsigned int y)
{
    if (check_coords(x,y) == true)
	* (RGB32pixel*) get_pixel(x,y) = RGB32_draw_colour;
}

} // namespace ReadingPeopleTracker
