/***************************************************************
 * C++ source
 *
 * File : 	PnmSource.cc  // used to be called PgmSource in the past
 *
 * Module :	PnmSource
 *
 * Author : 	A M Baumberg (CoMIR)
 *
 * Creation Date : Wed Jun 29 20:25:25 1994 
 *
 ***************************************************************/


#include <math.h>
#include <ctype.h>  // for isspace() etc

#include "PnmSource.h"

#include "Grey8Image.h"
#include "RGB32Image.h"
#include "text_output.h"

namespace ReadingPeopleTracker
{

// utility function to read a pnm file header; returns 5 for P5 etc, 0 for error.
// NOTE: does not open or close file.  Leaves file pointer at first data byte.
PnmFileType PnmSource::read_pnm_header(FILE *infile,
				       unsigned int *width, unsigned int *height)
{
    unsigned char c1;
    unsigned char c2;
    unsigned char aline[255];
    
    fscanf(infile,"%c%c", &c1, &c2);
    if ((c1 != 'P') || ((c2 != '5') && (c2 != '6')))
	return PNM_INVALID;
    
    // nts: new parser which is compliant with the pnm standard (see p{g,p}m(5))
    // FIXME: we assume that there is no end of file at the wrong place etc...
    unsigned char byte;
    
    // skip whitespace and comments until first dimension parameter, width...
    for (byte = (unsigned char) fgetc(infile); (byte == '#') || (isspace(byte));  
	 byte = (unsigned char) fgetc(infile))
	if (byte == '#')   // ignore comments (to end of line)
	    while ((byte = (unsigned char) fgetc(infile)) != '\n') ; // skip chars
    
    unsigned int length = 0; // length of character line we are reading into "aline" buffer
    
    for ( ; (isdigit(byte) && (length < 255)); byte = (unsigned char) fgetc(infile))
	aline[length++] = byte;  // append character to aline
    
    if (length == 0)
	return PNM_INVALID;
    
    aline [length] = '\0';
    sscanf((char *) aline,"%ui",width);
    
    // skip whitespace and comments until second dimension parameter, height...
    for ( ; (byte == '#') || (isspace(byte)); 
	  byte = (unsigned char) fgetc(infile))
	if (byte == '#')   // ignore comments (to end of line)
	    while ((byte = (unsigned char) fgetc(infile)) != '\n') ; // skip chars to eol   
    
    length = 0;
    
    for ( ; (isdigit(byte) && (length < 255)); byte = (unsigned char) fgetc(infile))
	aline[length++] = byte;  // append character to aline
    
    if (length == 0)
	return PNM_INVALID;
    
    aline [length] = '\0';
    sscanf((char *) aline,"%u", height);
    
    // skip whitespace and comments until 3rd and last size parameter, maxval...
    for ( ; (byte == '#') || (isspace(byte)); byte = (unsigned char) fgetc(infile))
	if (byte == '#')   // ignore comments (to end of line)
	    while ((byte = (unsigned char) fgetc(infile)) != '\n') ; // skip chars to eol
    
    unsigned int maxval;  // maxval (maximum value allowed in data fields, expect 255 or less)
    length = 0;
    
    for ( ; (isdigit(byte) && (length < 255)); byte = (unsigned char) fgetc(infile))
	aline[length++] = byte;  // append character to aline
    
    if (length == 0)
	return PNM_INVALID;
    
    //  We assume RAW PPM here: only one whitespace (usually newline) is allowed at
    //  this point in file (see ppm(5))
    if (!(isspace(byte)))
	return PNM_INVALID;    
    
    aline [length] = '\0';
    sscanf((char *) aline,"%u", &maxval);
    
    if (maxval > 255)       // more than 255 possible values for input components?
	return PNM_INVALID; // these cannot be no single bytes per colour channel!
    
    switch (c2)
    {
    case '5': return PNM_GREY_COMPRESSED;
    case '6': return PNM_COLOUR_COMPRESSED;
	/* NOTREACHED */
    default : return PNM_INVALID;
    }
    /* NOTREACHED */
}


Image *PnmSource::read_pnm(FILE *infile, Image *res)
{
    unsigned int width;
    unsigned int height;
    
    PnmFileType file_type = read_pnm_header(infile,
					   &width, &height);
    
    switch (file_type)
    {
    case PNM_COLOUR_COMPRESSED:  // marker "P6" ---  PPM (24-bit RGB) raw (RAWBITS)
    {
	bool created_new_image = false;
	
	if (res == NULL)
	{
	    res = new RGB32Image(width,height);
	    created_new_image = true;
	}
	
	if ((res->get_image_type() != RGB32) || (res->get_width() != width)
	    || (res->get_height() != height))
	{
	    cerror << " PnmSource::read_pnm(): Unexpected image dimensions/format (expected "
		   << res->get_width() << "x" << res->get_height();

	    if (res->get_image_type() == GREY8)
		cerror << " PGM). " << endl;
	    else
		if (res->get_image_type() == RGB32)
		    cerror << " PPM)." << endl;
		else
		    cerror << " PNM)." << endl;

	    if (created_new_image)
		delete res;
	    return NULL;
	}
	
	RGB32pixel *dat1;
	
	unsigned int x;
	unsigned int y;
	
	for (y = height; y > 0; y--)
	{
	    dat1 = (RGB32pixel *) res->get_pixel(0, y-1);
	    for (x = 0; x < width; x++)
	    {
		fscanf(infile, "%c%c%c",
		       &(dat1->red),&(dat1->green),&(dat1->blue));
		dat1->alpha = 0;
		dat1++;
	    }
	}
	
	// fread does not distinguish between end-of-file and error,
	// we use use ferror(3) to determine which one occurred.
	
	if (ferror(infile) != 0)
	{
	    cerror << " PnmSource::read_pnm: error reading PPM image file (file truncated?) "
		   << endl;

	    // ignore error, hoping for the best
	}
	break;
    }
    
    case PNM_GREY_COMPRESSED:  // marker "P5" ---  PGM (256 grey levels) raw (RAWBITS)
    {
	bool created_new_image = false;
	
	if (res == NULL)
	{
	    res = new Grey8Image(width, height);
	    created_new_image = true;
	}
	
	if ((res->get_image_type() != GREY8) || (res->get_width() != width) ||
	    (res->get_height() != height))
	{
	    cerror << " PnmSource::read_pnm(): Unexpected image dimensions/format (expected "
		   << res->get_width() << "x" << res->get_height();

	    if (res->get_image_type() == GREY8)
		cerror << " PGM). " << endl;
	    else
		if (res->get_image_type() == RGB32)
		    cerror << " PPM)." << endl;
		else
		    cerror << " PNM)." << endl;

	    if (created_new_image)
		delete res;

	    return NULL;
	}

	// PNM images are stored top to bottom.
	if (Image::image_storage_mode == IA_TOP_TO_BOTTOM)
	{
	    // no flipping of the image while reading: read all data in one go
	    fread(res->get_data(), 1, width*height, infile);
	}
	else
	{
	    // flip the image, but we use get_pixel which in turn depends on
	    // both Image::image_storage_mode and Image::image_addressing_mode
	    if (Image::image_addressing_mode == IA_TOP_TO_BOTTOM)
	    {
		// no conversion of the address get_pixel gives us
		unsigned int y;
		for (y = 0; y < height; y++)
		    fread(res->get_pixel(0,y), 1, width, infile);
	    }
	    else
	    {
		// convert get_pixel addresses
	    	unsigned int y;
		for (y = height; y > 0; y--)
		    fread(res->get_pixel(0, y-1), 1, width, infile);
	    }
	}
	
	// fread does not distinguish between end-of-file and error,
	// we use use ferror(3) to determine which one occurred.
	
	if (ferror(infile) != 0)
	{
	    cerror << " PnmSource::read_pnm: error reading PGM image file (file truncated?) "
		   << endl;

	    // ignore error, hoping for the best
	}
	break;
    }
    PNM_INVALID:
    default:
    {
	cerror << " PnmSource::read_pnm: Sorry --- Error in or unsupported pgm/ppm image format. " << endl;
	return NULL;
    }
    }
    
    return res;
}

Image *PnmSource::read_pnm(char *filename, Image *res)
{
    FILE *fhandle = fopen(filename, "rb");  // `b' may be important on non-UNIX
    if (fhandle == NULL)
    {
	cerror << "PnmSource::read_pnm: cannot open image file "
	       << filename << " " << endl;
	return NULL;
    }
    Image *PNM_image = read_pnm(fhandle, res);

    if (PNM_image == NULL)
	cerror << " PnmSource::read_pnm: could not read image file "
	       << filename << " " << endl;
    
    fclose(fhandle);

    return PNM_image;
}


//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  PnmSource::get_next: Decode next PNM file to get next input frame       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
Image *PnmSource::get_next()
{
    // get next file, creating new filename using NumberedFiles method

    // count tries as we might be asked to skip non-existing files
    unsigned int number_of_tries = 1;

    while ((current_file = fopen(get_next_filename(),"rb")) == NULL)  // `b' may be important on non-UNIX
    {
	// could not open requested file.
	if (get_file_count() == 0)   //  we have never read a file
	{
	    // We should not get here unless the file has been removed after
	    // our object was instantiated.  Well, better safe than sorry...
	    cerror << "JPEGSource::get_next: Could not open first numbered input file "
		   << get_current_filename() << " any more " << endl;
	    exit(1);
	}

	if ((skip_non_existing == false) ||
	    (number_of_tries > max_skip_non_existing))
        {
	    // no more files, signal end of sequence
	    current = NULL;
	    return current;
        }
	else
	    number_of_tries++;  // count number of tries
    }

#ifdef HAVE_POSIX_PIPES
    // if the input is compressed we have to uncompress it on the fly
    if (compressed_input)
    {
	// close file and uncompress it to a pipe using zcat
	fclose(current_file);
	sprintf(sys_command,"zcat '%s'", get_current_filename());
	current_file = popen(sys_command,"r");
	
	if (current_file == NULL)
	{
	    current = NULL;
	    return current;      //  indicate end of input sequence
	}
    }
#endif
      
    // The file exists and is open.  Read it into current image
    current = read_pnm(current_file, current);
    
    if (current == NULL)
    {
	cerror << " PnmSource::get_next: error while reading the file "
	       << get_current_filename() << " " << endl;

	return NULL;
    }
    
    // check closing method (current_file may be a file or a pipe)
#ifdef HAVE_POSIX_PIPES
    if (compressed_input)
	pclose(current_file);   // we are operating on a pipe
    else
#endif
	fclose(current_file);   // we are operating on a file
    
    // We have not defined a Time Stamp Header for PNM files so just
    // assume starting time 0 at frame 0
    current->set_frame_id(NumberedFiles::get_current_frame_number());
    current->set_frame_time_in_ms((frame_time_t)
				  (NumberedFiles::get_current_frame_number() * default_frame_time_in_ms));
    
    frame_count++;
    return current;
}

PnmSource::PnmSource(char *initial_frame,
		     bool the_skip_non_existing,
		     unsigned int the_max_skip_non_existing) : ImageSource(NULL),
							       NumberedFiles(initial_frame)
{
    skip_non_existing = the_skip_non_existing;
    max_skip_non_existing = the_max_skip_non_existing;
    
    // check suffix: compressed file?
    char *file_ext = strrchr(initial_frame, '.');
    
    if ((file_ext != NULL)
	&& ((strcasecmp(file_ext,".gz") == 0) ||
	    (strcasecmp(file_ext,".z") == 0)))
    {
	// compressed file, we assume following files are compressed, too
	compressed_input = true;
    }
    else
	compressed_input = false;
    
    // check initial file: read header for format check and width/height variables
    
#ifdef HAVE_POSIX_PIPES
    if (compressed_input)
    {
	// uncompress input using zcat
	sprintf(sys_command,"zcat \"%s\" 2> /dev/null", get_current_filename());
	current_file = popen(sys_command,"r");
    }
    else
#endif
    {
	// open file to check existence and dimensions
	current_file = fopen(get_current_filename(),"rb");  // `b' may be important on non-UNIX
    }
    
    if (current_file == NULL)
    {
	cerror << "PnmSource::PnmSource: cannot read input file "
	       << get_current_filename() << " " << endl;
	exit(1);
    }

    PnmFileType file_type = read_pnm_header(current_file, &xdim, &ydim);

    if (file_type == PNM_INVALID)
    {
	cerror << "PnmSource::PnmSource: error in input file "
	       << get_current_filename() << "  --- file format not recognised. " << endl;
	exit(1);
    }

    // make sure we know the ImageType of the images
    switch (file_type)
    {
    case PNM_COLOUR_COMPRESSED:  // marker "P6" ---  PPM (24-bit RGB) raw (RAWBITS)
    {
	image_type = RGB32;  // image will be decoded into an RGB32Image

	break;
    }
    
    case PNM_GREY_COMPRESSED:  // marker "P5" ---  PGM (256 grey levels) raw (RAWBITS)
    {
	image_type = GREY8;  // image will be decoded into a Grey8Image
	
	break;
    }
    default:
    {
	cerror << " PnmSource::PnmSource: Cannot determine image type for image in file "
	       << get_current_filename() << " " << endl;
	exit(1);
    }
    }
    
    
#ifdef HAVE_POSIX_PIPES
    if (compressed_input)  // we have to finish the poor zcat program
    {
	if (pclose(current_file) == -1)
	{
	    cerror << "PnmSource::PnmSource: cannot find or decompress input file "
		   << get_current_filename() << " " << endl;
	    exit(1);
	}
    }
    else
#endif
    {
	// close current input file
	fclose(current_file);
    }
    
    ImageSource::source_type = IST_NUMBERED_FILES;
}

} // namespace ReadingPeopleTracker
