// This is file PipeSourceFilters.h, created by nts on Mon Feb 25 16:47:13 GMT 2002

////
////  NB do not include this file directly.  We are included by PipeSource.h
////

#ifndef __PIPE_SOURCE_FILTERS_H__
#define __PIPE_SOURCE_FILTERS_H__

namespace ReadingPeopleTracker
{

/////   constant   //////       ---  added by nts, end 2000
class ConstantSource : public ImageSource
{
protected:
    Image *ConstantImage;
    
    
public:
    Image *get_next()
	{
	    // do not change the image (constant...) but the frame id and time...
	    frame_id_t new_frame_id = current->get_frame_id() + 1; // FIXME: should be + frame_id_delta
	
#if (MAX_FRAME_ID - FRAME_ID_T_MAX) > 0    // the following makes no sense if MAX_FRAME_ID == FRAME_ID_T_MAX
	    if (new_frame_id > MAX_FRAME_ID)
		new_frame_id %= (MAX_FRAME_ID + 1);
#endif
	    
	    current->set_frame_id(new_frame_id);
	    
	    // use our default frame rate
	    current->set_frame_time_in_ms(current->get_frame_time_in_ms() +
					  default_frame_time_in_ms);
    
	    frame_count++;
	    return current = ConstantImage;
	}
    
    virtual unsigned int get_xdim() const
	{
	    return ConstantImage->get_width();
	}
    
    virtual unsigned int get_ydim() const
	{
	    return ConstantImage->get_height();
	}
    
    virtual ImageType get_image_type() const
	{
	    return ConstantImage->get_image_type();
	}
    
    ConstantSource(Image *const_image)
	{
	    assert (const_image != NULL);  // who would want a constant NULL source?
	    
	    // this creates a new image
	    current = ConstantImage = const_image->copy();
	}
};

    
/****** resample *****/
class ResampleSource : public PipeSource
{
protected:
    unsigned int xstep;
    unsigned int ystep;
    
    inline Image *recalc()
	{
	    return in->get_current()->resample(xstep, ystep, current);
	}
public:
    ResampleSource(ImageSource *source, unsigned int the_xstep,
		   unsigned int the_ystep) : PipeSource(source)
	{
	    xstep = the_xstep;
	    ystep = the_ystep;
	    in = source;
//	    refresh();
	}
};

/****** rotate *****/
class RotateSource : public PipeSource
{
protected:
    int xcentre, ycentre;
    realno angle, scale;
    
    Image *recalc()
	{ 
	    if ((xcentre == -1) && (in->get_current() != NULL))
		xcentre = in->get_current()->get_width() / 2;
	    
	    if ((ycentre == -1) && (in->get_current() != NULL))
		ycentre = in->get_current()->get_height() / 2;
	    
	    return in->get_current()->transform(xcentre, ycentre, angle,
						scale, current);
	}
    
public:
    RotateSource(ImageSource *source, int xc = -1, int yc =-1,
		 realno ang = 3.14159265358979 / 2, realno sc = 1.0)
	: PipeSource(source)
	{
	    xcentre = xc;
	    ycentre = yc;
	    angle = ang;
	    scale = sc;
	    in = source;
//	    refresh();
	}
    
    RotateSource(ImageSource *source, realno ang, realno sc = 1.0)
	: PipeSource(source)
	{
	    xcentre = ycentre = -1;
	    angle = ang;
	    scale = sc;
	    in = source;
//	    refresh();
	}
    inline realno get_angle()
	{
	    return angle;
	}
    inline void set_angle(realno theta)
	{
	    angle = theta;
	}
};


/****** convolve *****/
class ConvolveSource : public PipeSource
{
protected:
    inline Image *recalc() 
	{
	    return in->get_current()->convolve(kern, current);
	}
    
    Kernel kern;

public:
    ConvolveSource(ImageSource *source, Kernel k) : PipeSource(source) 
	{
	    kern = k;
	    in = source;
//	    refresh();
	}
};

/***** blur *****/
class BlurSource : public PipeSource
{
protected:
    inline Image *recalc()
	{
	    return in->get_current()->blur(current);
	}
    
public:
    BlurSource(ImageSource *source) : PipeSource(source)
	{
	    in = source;
//	    refresh();
	}
};

/***** half_blur *****/
class HalfBlurSource : public PipeSource
{
protected:
    inline Image *recalc()
	{
	    return in->get_current()->half_blur(current);
	}
    
public:
    HalfBlurSource(ImageSource *source) : PipeSource(source)
	{
	    in = source;
//	    refresh();
	}
};

/***** middle *****/
class MiddleSource : public PipeSource
{
protected:
    inline Image *recalc() 
	{
	    Image *original = in->get_current();
	    
	    if (original == NULL)
		current = NULL;
	    else
		current = original->middle(current);
	    
	    return current;
	}
    
public:
    MiddleSource(ImageSource *source) : PipeSource(source)
	{
	    in = source;
//	    refresh();
	}
};

/***** neighbour ordered *****/
// sorts each pixel and all 8 neighbours and gives n'th of these 9 values
class NeighbourSource : public PipeSource
{
    unsigned int position;   // position 1 to 9 in the 9 values to return
    
protected:
    inline Image *recalc() 
	{
	    Image *original = in->get_current();
	    
	    if (original == NULL)
		current = NULL;
	    else
		current = original->neighbour_order(position, current);

	    return current;
	}
    
public:
    NeighbourSource(ImageSource *source, unsigned int the_position) : PipeSource(source)
	{
	    assert (the_position > 0);   // this should be an ordinal number
	    assert (the_position < 10);  // there are only 9 values
	    
	    position = the_position;
	    in = source;
//	    refresh();
	}
};

/***** fmed *****/
class FMedSource : public PipeSource
{
protected:
    inline Image *recalc() 
	{
	    return in->get_current()->fmed(current);
	}
    
public:
    FMedSource(ImageSource *source) : PipeSource(source)
	{
	    in = source;
//	    refresh();
	}
};

/***** minimum *****/
class MinimumSource : public PipeSource
{
protected:
    inline Image *recalc() 
	{
	    return in->get_current()->minimum(current);
	}
    
public:
    MinimumSource(ImageSource *source) : PipeSource(source)
	{
	    in = source;
//	    refresh();
	}
};

/***** maximum *****/
class MaximumSource : public PipeSource
{
protected:
    inline Image *recalc() 
	{
	    return in->get_current()->maximum(current);
	}
    
public:
    MaximumSource(ImageSource *source) : PipeSource(source)
	{
	    in = source;
//	    refresh();
	}
};

/***** sobel (thresholded) *****/
class SobelEdgeSource : public PipeSource
{
protected:
    inline Image *recalc() 
	{
	    return in->get_current()->sobel(threshold, current);
	}
    
public:
    unsigned int threshold;
    
    SobelEdgeSource(ImageSource *source, unsigned int thresh = 20) 
	: PipeSource (source) 
	{
	    threshold = thresh;
	    in = source;
//	    refresh();
	}
};

/***** sobel (unthresholded) *****/
class SobelSource : public PipeSource
{
protected:
    inline Image *recalc() 
	{
	    return in->get_current()->sobel(current);
	}
    
public:
    SobelSource(ImageSource *source) : PipeSource(source)
	{
	    in = source;
//	    refresh();
	}
};

/***** threshold *****/
class ThresholdSource : public PipeSource
{
protected:
    inline Image *recalc() 
	{
	    return in->get_current()->threshold(threshold, current, &no_marked);
	}
    
public:
    unsigned char threshold;
    unsigned int no_marked;
    
    ThresholdSource(ImageSource *source, unsigned char thresh) : PipeSource(source) 
	{
	    threshold = thresh;
	    in = source;
//	    refresh();
	}
};

/***** sub_image *****/
class SubImageSource : public PipeSource
{
protected:
    unsigned int xmin;
    unsigned int xmax;
    unsigned int ymin;
    unsigned int ymax;
    
    inline Image *recalc() 
	{
	    return in->get_current()->extract_subimage(xmin, xmax, ymin, ymax);
	}
    
public:
    SubImageSource(ImageSource *the_source,
		   unsigned int the_xmin, unsigned int the_xmax,
		   unsigned int the_ymin, unsigned int the_ymax)
	: PipeSource(the_source) 
	{
	    xmin = the_xmin;
	    xmax = the_xmax;
	    ymin = the_ymin;
	    ymax = the_ymax;
	    in = the_source;
//	    refresh();
	}
};

#ifdef _SVID_SOURCE
// the following uses SVID 48-bit random numbers...

/***** NoisySource *****/

class NoisySource : public PipeSource
{
protected:
    inline Image *recalc()
	{
	    Image *ret_img = in->get_current()
		->add_gaussian_noise(sd, noise, current);
	    total_noise += noise;
	    return ret_img;
	}
    
public:
    realno sd;
    realno noise;
    realno total_noise;
    
    NoisySource(ImageSource *source, int sdev) : PipeSource(source)
	{
	    sd = sdev;
	    in = source;
	    total_noise = 0;
	}
};

#endif // ifdef _SVID_SOURCE

/*****  difference image (abs source1-source2) *****/
class DifferenceSource : public PipeSource  /// (nts) was an ImageSource
{
protected:
    ImageSource *source1;
    ImageSource *source2;
    
    inline Image *recalc()
	{
	    return source1->get_current()->difference
		(source2->get_current(), current);
	}
public: 
    inline virtual Image *refresh()
	{ 
	    if ((source1->get_current() == NULL) ||
		(source2->get_current() == NULL)) 
		return NULL;
	    else
		return current = recalc();
	}
    
    inline virtual Image *get_next()
	{
//  	    // frame_id++;
//  	    if ((source1->get_next() == NULL) || (source2->get_next() == NULL))
//  		return NULL;
//  	    else
	    frame_count++;
	    return get_current();  //  automatically follow in's new frame_id
	}
    
    inline virtual Image *get_current()
	{
	    // match frame id to parents' frame id
	    if ((current != NULL) &&                   // we have a valid image
		(current->get_frame_id() ==
		 MAX(source1->get_source_frame_id(),
		     source2->get_source_frame_id()))) // and are up to date ?
	    {
		// then do not spend any CPU time on re-calculating
		return current;
	    }

	    source1->get_current();           // recursive update as necessary
	    source2->get_current();           // recursive update as necessary
	    current = recalc();
	    current->set_frame_id(MAX(source1->get_source_frame_id(),
				      source2->get_source_frame_id()));
	    current->set_frame_time_in_ms(MAX(source1->get_source_frame_time_in_ms(),
					      source2->get_source_frame_time_in_ms()));
	    return current;
	}
    
    DifferenceSource(ImageSource *s1, ImageSource *s2)
	: PipeSource(s1) ///, PipeSource(s2)
	{
	    in = source1 = s1;
	    source2 = s2;
	}
};

/***** thresholded difference *****/
class ThresholdedDifferenceSource : public PipeSource /// (nts) was ImageSource
{
protected:
    ImageSource *source1;
    ImageSource *source2;
    
public:
    realno threshold;
    
protected:
    inline Image *recalc()
	{
	    return source1->get_current()
		->difference(source2->get_current(), threshold, current);
	}
    
public: 
    inline Image *refresh()
	{ 
	    if ((source1->get_current() == NULL) ||
		(source2->get_current() == NULL)) 
		return NULL;
	    else
		return current = recalc();
	}
    
    inline virtual Image *get_next()
	{
//  	    // frame_id++;
//  	    if ((source1->get_next() == NULL) || (source2->get_next() == NULL))
//  		return NULL; 
//  	    else
	    frame_count++;
	    return get_current();  //  automatically follow in's new frame_id
	}
    
    
    inline Image *get_current()
	{
	    // match frame id to parents' frame id
	    if ((current != NULL) &&                   // we have a valid image
		(current->get_frame_id() ==
		 MAX(source1->get_source_frame_id(),
		     source2->get_source_frame_id()))) // and are up to date ?
	    {
		// then do not spend any CPU time on re-calculating
		return current;
	    }

	    source1->get_current();           // recursive update as necessary
	    source2->get_current();           // recursive update as necessary
	    current = recalc();
	    current->set_frame_id(MAX(source1->get_source_frame_id(),
				      source2->get_source_frame_id()));
	    current->set_frame_time_in_ms(MAX(source1->get_source_frame_time_in_ms(),
					      source2->get_source_frame_time_in_ms()));
	    return current;
	}
    
    ThresholdedDifferenceSource(ImageSource *s1, ImageSource *s2,
				realno thresh = -1)
	: PipeSource(s1) ///, PipeSource(s2)
	{
	    in = source1 = s1;
	    source2 = s2;
	    threshold = thresh;
	}
};


/******************/
class SimpleDiffSource : public PipeSource /// (nts) was ImageSource
{
protected:
    ImageSource *source1;
    ImageSource *source2;
public:
    realno threshold;
    unsigned int no_marked;
    
protected:
    inline Image *recalc()
	{
	    return source1->get_current()->
		simple_difference(source2->get_current(), threshold,
				  &no_marked, current);
	}
    
public:
    inline Image *refresh()
	{
	    if ((source1->get_current() == NULL) ||
		(source2->get_current() == NULL))
		return NULL;
	    else
		return current = recalc();
	}
    inline virtual Image *get_next()
	{
//  	    // frame_id++;
//  	    if ((source1->get_next() == NULL) || (source2->get_next() == NULL))
//  		return NULL;
//  	    else
	    frame_count++;
	    return get_current();  //  automatically follow in's new frame_id
	}
    
    inline Image *get_current()
	{
	    // match frame id to parents' frame id
	    if ((current != NULL) &&                   // we have a valid image
		(current->get_frame_id() ==
		 MAX(source1->get_source_frame_id(),
		     source2->get_source_frame_id()))) // and are up to date ?
	    {
		// then do not spend any CPU time on re-calculating
		return current;
	    }

	    source1->get_current();           // recursive update as necessary
	    source2->get_current();           // recursive update as necessary
	    current = recalc();
	    current->set_frame_id(MAX(source1->get_source_frame_id(),
				      source2->get_source_frame_id()));
	    current->set_frame_time_in_ms(MAX(source1->get_source_frame_time_in_ms(),
					      source2->get_source_frame_time_in_ms()));
	    return current;
	}
    
    SimpleDiffSource(ImageSource *s1, ImageSource *s2, realno thresh = -1)
	: PipeSource(s1) ///, PipeSource(s2)
	{
	    in = source1 = s1;
	    source2 = s2;
	    threshold = thresh;
	}
};

/******************/

class BlendSource : public PipeSource /// (nts) was ImageSource
{
protected:
    ImageSource *source1;
    ImageSource *source2;
    int a;
    int b;
    
    inline Image *recalc()
	{
	    return source1->get_current()->image_blend
		(source2->get_current(), a, b, current);
	}
    
public:
    inline Image *refresh()
	{
	    if ((source1->get_current() == NULL) ||
		(source2->get_current() == NULL))
		return NULL;
	    else
		return current = recalc();
	}
    
    inline virtual Image *get_next()
	{
//  	    // frame_id++;
//  	    if ((source1->get_next() == NULL) || (source2->get_next() == NULL))
//  		return NULL;
//  	    else
	    frame_count++;
	    return get_current();  //  automatically follow in's new frame_id
	}
    
    inline Image *get_current()
	{
	    // match frame id to parents' frame id
	    if ((current != NULL) &&                   // we have a valid image
		(current->get_frame_id() ==
		 MAX(source1->get_source_frame_id(),
		     source2->get_source_frame_id()))) // and are up to date ?
	    {
		// then do not spend any CPU time on re-calculating
		return current;
	    }

	    source1->get_current();           // recursive update as necessary
	    source2->get_current();           // recursive update as necessary
	    current = recalc();
	    current->set_frame_id(MAX(source1->get_source_frame_id(),
				      source2->get_source_frame_id()));
	    current->set_frame_time_in_ms(MAX(source1->get_source_frame_time_in_ms(),
					      source2->get_source_frame_time_in_ms()));
	    return current;
	}
    
    BlendSource(ImageSource *s1, ImageSource *s2,
		int param_a = 1, int param_b = 1)
	: PipeSource(s1) ///, PipeSource(s2)
	{
	    in = source1 = s1;
	    source2 = s2;
	    a = param_a;
	    b = param_b;
	}
};
/******************/


class VFlipSource : public PipeSource
{
protected:
    inline Image *recalc()
	{
	    return in->get_current()->flip_vertically(current);
	}
    
public:
    VFlipSource(ImageSource *source) : PipeSource(source)
	{
	    in = source;
//	    refresh();
	}
};

/******************/

// SkipSource
// sample an input ImageSource or PipeSource
// pass on frames {start_frame + {0, n, 2n, ...}}

class SkipSource : public PipeSource /// (nts) was ImageSource
{
    
protected:
    
    ImageSource *src;
    PipeSource *in;
    unsigned int skip;
    
    inline Image *recalc()
	{
	    if (src != NULL)  // we are derived from an ImageSource
		return src->get_current();  // ignore our PipeSource
	    else
		return in->get_current();
	}
    
public:
    
    inline virtual Image *get_next()
	{
  	    if (src != NULL)
	    {
		// we are derived from an ImageSource
		unsigned int count;
		for (count = 0; count < skip; count++)
		    current = src->get_next(); // ignoring our PipeSource

		current->set_frame_id(get_source_frame_id());
		current->set_frame_time_in_ms(get_source_frame_time_in_ms());
	    }
  	    else
	    {
		unsigned int count;
		for (count = 0; count < skip; count++)
		    current = in->get_next();

		current->set_frame_id(in->get_source_frame_id());
		current->set_frame_time_in_ms(in->get_source_frame_time_in_ms());
	    }
	    frame_count++;
	    return current;
	}
    
    inline virtual void clean_up()
	{
	    current = NULL;
	    in = NULL;
	}
    
    virtual unsigned int get_xdim() const
	{
	    if (src != NULL)  // we are derived from an ImageSource
		return src->get_xdim(); // ignore our PipeSource
	    else
		return in->get_xdim();
	}
    
    virtual unsigned int get_ydim() const
	{
	    if (src != NULL)  // we are derived from an ImageSource
		return src->get_ydim(); // ignore our PipeSource
	    else
		return in->get_ydim();
	}
    
    SkipSource(unsigned int n, ImageSource *source) : PipeSource(source)
	{
	    src = source;
	    in = new PipeSource(src, true);
	    skip = n;
	}
    
    SkipSource(unsigned int n, PipeSource *source) : PipeSource(source)
	{
	    src = NULL;  // we are not derived from an ImageSource
	    in = source;
	    skip = n;
	}
    
    ~SkipSource()
	{
	    current = NULL;
	    delete in;
	}
    
};

// RepeatSource
// temporally re-sample an input ImageSource
// pass on each frame n times


class RepeatSource : public PipeSource /// (nts) was ImageSource
{
    
protected:
    
    ImageSource *in;
    frame_id_t repeat;
    frame_id_t count;
    
public:
    
    RepeatSource(frame_id_t the_repeat, ImageSource *source) : PipeSource(source)
	{
	    in = source;
	    repeat = the_repeat;
	    count = 0;
	}
    
    inline virtual Image *get_next() 
	{ 
	    if (frame_count == 0)
	    {
		// first image: get it from the source
		current = in->get_current();   // NB no check here (could be NULL)
	    }
	    else
		if (frame_count % repeat == 0)
		{
		    // we have shown the image `repeat' times: get next image from the source
		    current = in->get_next();   // NB no check here (could be NULL)
		}
	    
	    // adjust frame id according to current
	    current->set_frame_id(in->get_source_frame_id());
	    current->set_frame_time_in_ms(in->get_source_frame_time_in_ms());

	    frame_count = (frame_count + 1) % MAX_FRAME_ID;
	    
	    return current;
	}
    
//     inline void clean_up()
// 	{
// 	    current = NULL;
// 	    in = NULL;
// 	}
    
    ~RepeatSource()
	{
	    current = NULL;
	    delete in;
	}
};

/******************/

// commented out by nts --- Feb 2001
//  // SingleImageSource
//  // first call to get_next returns an image
//  // subsequent calls return NULL
//
//  class SingleImageSource : public ImageSource
//  {
//  public:
//    
//      SingleImageSource(Image *frame) : ImageSource(frame, 0)
//  	{
//  	    // nothing
//  	}
//    
//      inline virtual Image *get_next()
//  	{
//	    frame_count++;
//  	    if (frame_number++ == 0)
//  		return current;
//  	    return NULL;
//  	}
//  };


/******************/

class ConcatSource : public PipeSource /// (nts) was ImageSource
{
protected:
    
    ImageSource **in_array;
    unsigned int no_sources;
    unsigned int current_source;
    
public:
    
    ConcatSource(ImageSource *s1, ImageSource *s2) : PipeSource(s1)
	{ 
	    in_array = new ImageSource*[2]; 
	    in_array[0] = s1;
	    in_array[1] = s2;
	    no_sources = 2;
	    current_source = 0;
	    // this has been done already:  in = in_array[0];
	}
    
    ConcatSource(ImageSource **the_in_array,
		 unsigned int the_no_sources) : PipeSource(the_in_array[0])
	{
	    in_array = the_in_array;
	    no_sources = the_no_sources;
	    current_source = 0;
	    // this has been done already:  in = in_array[0];
	}
    
    inline virtual Image *get_next()
	{
	    if (current_source >= no_sources)
		return NULL;

	    Image *curr_img = in_array[current_source]->get_next();

	    if ((curr_img == NULL) && (current_source + 1 < no_sources))
	    {
		// proceed to next image source
		current_source++;
		in = in_array[current_source];
		curr_img = in_array[current_source]->get_next();
	    }
	    
	    if (curr_img != NULL)
	    {
		if (current == NULL)
		    current = curr_img->virtual_copy();
		
		if (curr_img->get_image_type() != current->get_image_type())
		{
		    cerror << "ConcatSource: cannot concat different image types "
			   << endl;
		    exit(1);
		}
		if (current->get_data() != curr_img->get_data())
		{
		    current->set_data(curr_img->get_data());
		    // FIXME: possible problem with ``own_data = true;'' in set_data() ???
		}
		current->set_frame_id(in->get_source_frame_id());
		current->set_frame_time_in_ms(in->get_source_frame_time_in_ms());
	    }
	    
	    frame_count++;
	    return current;
	}
    
//      Image *get_current()
//  	{
//  	    // FIXME: might not work.
//  	    in->get_current();                 // recursive update as necessary
//  	    return current;           // NB no checking for in's frame id here!
//  	}
    
    inline void clean_up()
	{
	    in_array = NULL;
	    current = NULL;
	}
    
    ~ConcatSource()
	{
	    for (unsigned int i = 0; i < no_sources; i++)
		delete in_array[i];
	    delete [] in_array;
	}
    
};

/******************/

class GammaCorrectSource: public PipeSource
{
public:
    
    GammaCorrectMap gamma_map;
    
protected:
    inline Image *recalc() 
	{
	    return in->get_current()->map_intensities(gamma_map,current);
	}
    
public:
    GammaCorrectSource(ImageSource *source, realno gamma_value)
	: PipeSource(source), gamma_map(gamma_value)
	{
	    in = source;
//	    refresh();
	}
};



////   Colour Filtering to Grey8Image --- PipeSource i/face  ////  nts Feb 2001

class ColourFilterSource: public PipeSource
{
    
protected:
    ColourFilteringMethod CF_method;
    
    inline Image *recalc()
	{
	    return in->get_current()->ColourFilter(current,CF_method);
	}
    
public: 
    inline Image *refresh()
	{ 
	    if (get_current() == NULL)
		return NULL;
	    else
		return current = recalc();
	}
    inline virtual Image *get_next()
	{ 
//  	    // frame_id++;
//  	    if (in->get_next() == NULL)
//  		return NULL;
//  	    else
	    frame_count++;
	    return get_current();  //  automatically follow in's new frame_id
	}
    
    ColourFilterSource (ImageSource *source,
			ColourFilteringMethod method = CF_METHOD_Y_709)
	: PipeSource(source)
	{
	    in = source;
	    CF_method = method;
	}
};

#ifdef HSV

////   HSV32Source: generate `HSV32Image's --- PipeSource i/face  ////  nts Apr 2001

class HSV32Source: public PipeSource
{
    
protected:
    inline Image *recalc()
	{
	    return in->get_current()->to_HSV32(current);
	}
    
public: 
    inline Image *refresh()
	{ 
	    if (get_current() == NULL)
		return NULL;
	    else
		return current = recalc();
	}
    
    inline virtual Image *get_next()
	{ 
//  	    // frame_id++;
//  	    if (in->get_next() == NULL)
//  		return NULL;
//  	    else
	    frame_count++;
	    return get_current();  //  automatically follow in's new frame_id
	}
    
    HSV32Source (ImageSource *source) : PipeSource(source)
	{
	    in = source;
	}
};

#endif  // ifdef HSV

} // namespace ReadingPeopleTracker

#endif
