///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  MultiBackgroundSource.h                                                 //
//                                                                           //
//  A background generation class which lets you incorporate static objects  //
//     into the background and remove them when they start moving again.     //
//     Written with cars (ie unidentified, large `Region's) in mind          //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Mon Sep 03 10:02:52 BST 2001                                 //
//  Revision  : 0.1 of Fri Sep  7 16:05:59 BST 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
//  Changes:                                                                 //
//   nts: rev 1.0: initial working revision    The Future                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

////
////  NB do not include this file directly.  We are included by PipeSource.h
////

#ifndef __MULTI_BACKGROUND_SOURCE_H__
#define __MULTI_BACKGROUND_SOURCE_H__

#include "Region.h"
#include "RegionSet.h"
#include "PipeSource.h"

#ifndef NO_DISPLAY
#ifdef DEBUG
#include "os_specific_things.h"  // for sleep()
#endif   // ifdef DEBUG
#endif   // #ifndef NO_DISPLAY

namespace ReadingPeopleTracker
{

static const char* MultiBackgroundSource_Revision = "@(#) MultiBackgroundSource.h, rev 0.1 of Fri Sep  7 16:05:59 BST 2001, Author Nils T Siebel, Copyright (c) 2001 The University of Reading";

//
//   Background generation class which lets you incorporate static objects
//     into the background and remove them when they start moving again.
//     Written with cars (ie unidentified, large `Region's) in mind
//
class MultiBackgroundSource: public PipeSource
{
    
protected:
    Image *EmptyBackground;    // without any objects, initialised at instantiation
    Image *CurrentBackground;  // background with all currently static objects
    Image *MotionMask;
    
    RegionSet *static_regions;        // temporarily static objects included in background
    
    bool dirty;                // tag whether to recalc(): FIXME: ONLY IF NON-MEDIAN!!!
    
    PipeSource *BackgroundSource;  // original (fi median) background source
    
    Image *recalc()
	{ 
	    cdebug <<  " MultiBackgroundSource: recalc() ";
//  	    if (! dirty)
//  	    {
//  		cdebug << " *** (cached)" << endl;
//  		return CurrentBackground;
//  	    }
//  	    else
	    cdebug << endl;
	    
	    // get "normal" background
	    CurrentBackground = BackgroundSource -> get_current()
		-> copy(CurrentBackground);
//  	    cdebug << endl << " BackgroundSource -> get_current() called " << endl;
	    
	    // now incorporate objects (regions)
	    for (ListNode<Region> *curr = static_regions->first; curr != NULL;
		 curr = curr->next)
	    {
		register Region *reg = curr->dat;
		
		if (reg->region_img != NULL)
		{
		    // incorporate static region image into background
		    cdebug <<  " MultiBackgroundSource::recalc(): incorporate region: "
			   << endl << *reg
			   << endl;
		    CurrentBackground -> paste_subimage
			(reg->xlo, reg->xhi, reg->ylo, reg->yhi,
			 reg->region_img);
		}
		else
		{
		    // no region image to incorporate :-(
		    cdebug <<  " MultiBackgroundSource::recalc(): Warning: "
			   << "No region image available for Region. " << endl;
		}
	    }
	    return CurrentBackground;
	}
    
public:
    
    MultiBackgroundSource(PipeSource *background_source,   // "normal" background
			  Image *empty_background = NULL)  // default: 1st frame
	: PipeSource(background_source)
	{
	    in = BackgroundSource = background_source;
	    
	    if (empty_background == NULL)
	    {
		// no empty background image given, default to first video image
		cdebug <<  " MultiBackgroundSource::MultiBackgroundSource(): Warning: "
		       << " No empty background image given, defaulting to first video image "
		       << endl;
		
		empty_background = BackgroundSource->get_current();
	    }
	    
	    assert(empty_background != NULL);
	    assert(empty_background->get_width() == BackgroundSource->get_current()->get_width());
	    
	    EmptyBackground = empty_background->copy();
	    
	    static_regions = new RegionSet;   // to store temporarily static objects
	    
	    // set valid `current' image, allocating memory for CurrentBackground
	    current = CurrentBackground = empty_background->copy();
	    MotionMask = NULL;
	}
    
    void set_motion_mask(Image *Img)  // intercept and draw our objects...
	{
	    MotionMask = Img->copy(MotionMask);
	    
	    // FIXME: have to invert :-(
	    for (int x = 0; x < MotionMask->get_width(); x++)
		for (int y = 0; y < MotionMask->get_height(); y++)
		{
		    register unsigned char c = *MotionMask->get_pixel(x,y);
		    *MotionMask->get_pixel(x,y) = ~c;
		}
	    
	    
	    // mask out our regions from update...
	    for (ListNode<Region> *curr = static_regions->first; (curr != NULL); curr = curr->next)
	    {
		Region *reg = curr->dat;
		int width = reg->xhi - reg->xlo;
		
		for (int y = reg->ylo; y <= reg->yhi; y++)
		    memset(MotionMask->get_pixel(reg->xlo,y),CLEAR_MARK,width);

		cdebug << " MultiBackgroundSource::set_motion_mask(): included "
		       << width << " by " << reg->yhi-reg->ylo << " mask " << endl;
		
	    }
//  	    MotionMask->set_title("Multi Motion Mask");
//  	    MotionMask->display();
	    BackgroundSource->set_motion_mask(MotionMask);
	}
    
    
    void add_region(Region *static_region)
	{
	    cdebug <<  " MultiBackgroundSource: add_region(): " << endl
		   << *static_region;
	    static_regions->add(static_region);
	    dirty = true;
	}
    
    // moving region (started moving again) is identified by origin and removed from static_regions
    void remove_region(Region *moving_region)
	{
	    Region *static_region;
	    
	    cdebug << " MultiBackgroundSource::remove_region() " << endl;
	    ListNode<Region> *curr;
	    
	    bool removed_region = false;
	    
	    for (curr = static_regions->first; curr != NULL; curr = curr->next)
	    {
		static_region = curr->dat;
		
		if (static_region->origin != moving_region->old_origin)  // identify by old_origin
		    continue;
		
		// remove moving region from static list
		static_regions->destroy(static_region);
		
		dirty = true;
		removed_region = true;
		
		break;
	    }
	    
	    assert(removed_region == true);  // otherwise something has gone wrong
	    
//  // simply copy over old so median update can be enabled
//  	    EmptyBackground -> get_subimage
//  		(moving_region->xlo, moving_region->xhi, moving_region->ylo, moving_region->yhi,
//  		 moving_region->region_img);
	}
    
    void force_recalc()
	{
	    recalc();
	}
    
};

} // namespace ReadingPeopleTracker

#endif
