///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  MovieStore.cc                                                            //
//                                                                           //
//  This class stores a sequence of images as a movie                        //
//                                                                           //
//  Currently, only numbered JPEG images can be written.                     //
//  This code is written with full colour images in mind (24/32-bit).        //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Mon Oct 16 10:30:39 GMT 2000                                 //
//  Revision  : 1.0 of Fri Oct 27 10:33:15 GMT 2000                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <ctype.h>  // for isdigit()

#include "MovieStore.h"
#include "Image.h"
#include "text_output.h"

namespace ReadingPeopleTracker
{

static const char *MovieStore_Revision = "@(#) MovieStore.cc, rev 1.0 of Fri Oct 27 10:33:15 GMT 2000, Author Nils T Siebel, Copyright (c) 2000 The University of Reading";

// For image<nnn>.jpeg give 1st name, for instance "image007.jpeg"
MovieStore::MovieStore (MovieStoreFormat movieformat,
			char *moviefilename)  
{
    signed int pos;
    format = movieformat;
    
    // set up variables...
    switch (format)
    {
    case MOVIESTORE_JPEG:
    {
	individual_files = true;
	break;
    }
    
    case MOVIESTORE_PNM:
    {
	individual_files = true;
	break;
    }

    default:
	cerror << " MovieStore: Unknown file format ("
	       << format << " requested.  Exiting..." << endl;
	
	exit(1);
    }
    
    strncpy(filename, moviefilename, 256+16+64-1);
    filename[256+16+64-1] = '\0';
    
    // file names...
    if (individual_files)
    {
	for (pos = strlen(filename)-1; pos >= 0; pos--)
	    if (isdigit(filename[pos]))
		break;
	
	if (pos < 0)
	{
	    cerror << "MovieStore: When storing individual images, "
		   << "please give first filename, like image007.jpeg" << endl;
	    exit(1);
	}
	
	strncpy(filename_ext, &filename[pos+1], 63);
	filename[pos+1] = 0;  // for atol
	
	for ( ; pos >= 0; pos--)
	    if (! (isdigit(filename[pos])))
		break;
	
	if (! (filename_num_digits = strlen(&filename[pos+1])))
	{
	    cerror << "MovieStore: When storing individual images, "
		   << "please give first filename, like image017.jpeg" << endl;
	    exit(1);
	}
	
	strncpy(filename_base, filename,MIN(pos+1,255));
	filename_base[MIN(pos+1,255)] = 0;
	
	start_frame = current_frame = atol(&filename[pos+1]);
    }
    else
    {
	outfile = fopen(filename,"wb");   // `b' may be important on non-UNIX
	if (outfile == NULL)   // `b' may be important on non-UNIX
	{
	    cerror << "MovieStore: Could not open outfile " << filename << " " << endl;
	    exit(1);
	}
    }
    
    num_frames = 0;
}


bool MovieStore::add_a_frame(Image *frame)
{
    if (frame == NULL)
	return true;
    
    if (individual_files)
    {
	// open file...
	sprintf(filename, "%s%0*i%s", filename_base,
		filename_num_digits, current_frame,
		filename_ext);
	
	outfile = fopen(filename,"wb");  // `b' may be important on non-UNIX
	
	if (outfile == NULL)
	{
	    cerror << "MovieStore: Could not open numbered outfile "
		   << filename << " for writing. " << endl;
	    exit(1);
	}
	
	if (format == MOVIESTORE_JPEG)   // set up JPEG compressor and compress image
	{
	    if (! (num_frames))  // first call?
	    {
		jpeg_compression_info.err = jpeg_std_error(&jpeg_error_manager);
		jpeg_create_compress(&jpeg_compression_info);
	    }
	    
	    jpeg_stdio_dest(&jpeg_compression_info, outfile);
	    
	    jpeg_compression_info.image_width  = xdim = frame->get_width();
	    jpeg_compression_info.image_height = ydim = frame->get_height();
	    depth = frame->get_bytes_per_pixel();    // expect 4 for 24-bit colour (RGB32)
	    jpeg_compression_info.input_components = MIN(depth,3);   // give 3 for 24-bit
	    
	    // FIXME: should use switch() here:
	    jpeg_compression_info.in_color_space = JCS_RGB;          
	    jpeg_compression_info.data_precision = 8;
	    
	    jpeg_set_defaults(&jpeg_compression_info);
	    
	    // set quality to 70 percent, that should be fine:
	    jpeg_set_quality (&jpeg_compression_info, 70, true);
	    
	    // libjpeg cannot handle 32-bit stored images so we might have to convert:
	    if (depth == 4)     // (if image is to be flipped, we could also copy)
	    {
		// copy image to temp space, storing RGBRGBRGB... not RGBaRGBa...
		if ((xdim * ydim * 3) > sizeof(image_buffer))   // avoid overflow
		{
		    cerror << "MovieStore: Cannot handle (convert) frames this size." << endl;
		    exit(1);
		}
		
		
		register int row;
		register int offset;  // == row * xdim * 3
		register int col;
		
		register RGB32pixel *pixel_ptr;  // see def'n of RGB32pixel in RGB32Image.h
		int colwidth = xdim*3;
		
		if (Image::image_addressing_mode == IA_TOP_TO_BOTTOM)
		{
		    // no need to flip the image while converting to 32-bit
		    for (row = 0; row < ydim; row++)
		    {
			// (NB get_pixel depends on Image::image_storage_mode)
			pixel_ptr = (RGB32pixel *) frame->get_pixel(0,row);
			offset = row * colwidth;
			
			for (col = 0; col < colwidth; pixel_ptr++)
			{
			    // see definition of RGB32pixel in RGB32Image.h
			    image_buffer[offset+col++] = pixel_ptr -> red;    // R
			    image_buffer[offset+col++] = pixel_ptr -> green;  // G
			    image_buffer[offset+col++] = pixel_ptr -> blue;   // B
			}
			jpeg_row_pointer[row] = (JSAMPROW) &(image_buffer[offset]);
		    }  
		}
		else
		{
		    // we need to flip the image (y lines) while converting to 32-bit
		    for (row = 0; row < ydim; row++)
		    {
			// (NB get_pixel depends on Image::image_storage_mode)
			pixel_ptr = (RGB32pixel *) frame->get_pixel(0,ydim-row-1);
			offset = row * colwidth;
			
			for (col = 0; col < colwidth; pixel_ptr++)
			{
			    // see definition of RGB32pixel in RGB32Image.h
			    image_buffer[offset+col++] = pixel_ptr -> red;    // R
			    image_buffer[offset+col++] = pixel_ptr -> green;  // G
			    image_buffer[offset+col++] = pixel_ptr -> blue;   // B
			}
			jpeg_row_pointer[row] = (JSAMPROW) &(image_buffer[offset]);
		    }
		}
		
		// now compress the converted frame from image_buffer
		
		// start actual compression, will be written to outfile
		jpeg_start_compress(&jpeg_compression_info, true);
		jpeg_write_scanlines(&jpeg_compression_info, jpeg_row_pointer, ydim);
	    }
	    else
	    {	
		// possibly greylevel image?
		//
		// JPEG image data should be written in top-to-bottom scanline order
		// see if we need to flip the image (y lines) while encoding...
		if (Image::image_addressing_mode == IA_TOP_TO_BOTTOM)
		{
		    // no need to flip the image
		    while (jpeg_compression_info.next_scanline < ydim)
		    {
			// (NB get_pixel depends on Image::image_storage_mode)
			jpeg_row_pointer[0] =
			    frame->get_pixel(0,(ydim-jpeg_compression_info.next_scanline-1));

			// only compress one row at a time (because of image_storage_mode)
			jpeg_write_scanlines(&jpeg_compression_info, jpeg_row_pointer, 1);
		    }
		}
		else
		{
		    // flip image (y lines) while compressing
		    while (jpeg_compression_info.next_scanline < ydim)
		    {
			// (NB get_pixel depends on Image::image_storage_mode)
			jpeg_row_pointer[0] =
			    frame->get_pixel(0,jpeg_compression_info.next_scanline);
			
			// only compress one row at a time (because of image_storage_mode)
			jpeg_write_scanlines(&jpeg_compression_info, jpeg_row_pointer, 1);
		    }
		}
	    }
	    
	    
	    // finish compression, flushing output buffer
	    jpeg_finish_compress(&jpeg_compression_info);
	    
	    fclose (outfile);
	}  // format == MOVIESTORE_JPEG

	else
	    if (format == MOVIESTORE_PNM)
	    {
		// close file because Image::save_pnm opens it itself
		fclose (outfile);

		// use Image::save_pnm method to write the file
		frame->save_pnm(filename);
	    }    
	else
	{
	    cerror << " MovieStore::add_a_frame: Unknown file format ("
		   << format << " Exiting..." << endl;
	    exit(1);
	}
    }
    else
    {
	cerror << "MovieStore::add_a_frame for non-individual files not implemented." << endl;
	fclose (outfile);
	exit(1);
    }
    
    current_frame++;
    num_frames++;
    
    return true;
}

MovieStore::~MovieStore()
{
    jpeg_destroy_compress(&jpeg_compression_info);
    // (should be closed already)  fclose(outfile);
}

} // namespace ReadingPeopleTracker
