/*
 * ImageSource.h
 * base class for image sources
 *
 * author: A.M.Baumberg (amb)
 * creation date: 8/1/93
 */

#ifndef __IMAGE_SOURCE_H__
#define __IMAGE_SOURCE_H__

#include <cstdio>
#include <cstring>      // for strncpy()
#include <ctime>

#include "Image.h"
#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

class ImageSource
{
protected:
 
    ImageSourceType source_type;
    
    Image *current;               // current frame (Image)

    frame_id_t frame_count;       // simple count for get_next() calls

    static int max_skip_non_existing;
    
    char imgsrc_window_title[64];
   
    static bool skip_non_existing;
    bool have_valid_header;
    
    
public:
    inline ImageSourceType get_source_type() const
	{
	    return source_type;   
	}
    
    // NB the following methods must be virtual
    inline virtual Image *get_current()
	{
            if (current == NULL)   // happens in the very first frame
		current = get_next();
	    return current;
	}

    inline virtual frame_id_t get_source_frame_id()
	{
	    return get_frame_id();
	}

    inline virtual frame_time_t get_source_frame_time_in_ms()
	{
	    return get_frame_time_in_ms();
	}

    inline frame_time_t get_frame_count() const
	{
	    return frame_count;
	}

    // NB set_current should be used with caution
    void set_current(Image *new_current)
    {
        current = new_current;
    }

    inline frame_id_t get_frame_id()
	{
            return get_current()->get_frame_id();
	}
    
    inline frame_time_t get_frame_time_in_ms()
	{
            return get_current()->get_frame_time_in_ms();
	}
    
    ImageSource(Image *the_current = NULL)
	{
	    current = the_current;
	    imgsrc_window_title[0] = 0;
	    have_valid_header = false;
	    frame_count = 0;

	    source_type = IST_UNKNOWN;  // overwrite in derived class's constructor!
	}
    
    virtual ~ImageSource()
	{
	    if (current != NULL)
		delete current;
	}
    
    virtual long display(long wndow = NULLWIN);
    
    inline void set_title(char *title)
	{
#ifndef NO_DISPLAY
	    if (current != NULL)
	    {
		frame_id_t saved_frame_id = current->get_frame_id();  // save
		current->set_title(title);
		current->set_frame_id(saved_frame_id);  // restore
	    }
	    else
		strncpy(imgsrc_window_title,title,63);  // remember for next display()
#endif // ifndef NO_DISPLAY
	}
    
    inline void set_title(unsigned char *title)
	{
	    set_title((char *) title); // cast to keep our compiler happy...
	}
    
    inline bool get_have_valid_header()
	{
	    return have_valid_header;
	}


    //
    // virtual methods, please redefine for all derived ImageSource classes
    //

    // NB derived get_next() must define valid frame_time_in_ms and frame_id for each Image
    virtual Image *get_next() = 0;
 
    // These three must return valid values after the constructor is called without
    //   allocating the current image!
    virtual unsigned int get_xdim() const = 0;

    virtual unsigned int get_ydim() const = 0;

    virtual ImageType get_image_type() const = 0;
    

};

} // namespace ReadingPeopleTracker

#endif
