///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  HSV32Image.h                                                             //
//                                                                           //
//  HSV image class derived from RGB32Image class                            //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Thu Apr 19 11:59:19 BST 2001                                 //
//  Revision  : 0.0 of Thu Apr 19 11:59:19 BST 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef __HSV32_IMAGE_H__
#define __HSV32_IMAGE_H__

#include "RGB32Image.h"
#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

// write out an HSV32pixel as (h,s,v) triplet
ostream &operator << (ostream &target, HSV32pixel &ColourPixel);

class HSV32Image : public RGB32Image
{
protected:
    
public:
    
    HSV32Image(unsigned int the_width, unsigned int the_height,
	       frame_id_t the_frame_id = 0, frame_time_t the_frame_time_in_ms = 0,
	       unsigned char *the_data = NULL)
	: RGB32Image(the_width, the_height,
		     the_frame_id, the_frame_time_in_ms,
		     the_data)
	{
	    image_type  = HSV32;

	    // nothing else needed which RGB32Image::RGB32Image does not do
	}
    
    // conversion of RGB colour pixels to HSV using algorithm by D Travis.
    // 2nd version uses lookup but falls back to the 1st version if no lookup
    // table has been set up by the function setup_HSV32_to_RGB32_lookup_table().
    static HSV32pixel to_HSV32 (const RGB32pixel *RGBpix);
    
    static HSV32pixel to_HSV32_using_lookup (const RGB32pixel *RGBpix);
    
//      virtual ImageType imgtype() const
//  	{
//  	    return HSV32;
//  	}
    
    Image *copy (Image *r = NULL);
    
    inline Image *copy_type(int w = 0, int h = 0) 
	{ 
	    if (w == 0)
		w = width;
	    if (h == 0)
		h = height;
	    return new HSV32Image(w, h);
	}
    
    inline Image *virtual_copy(Image *res = NULL) const
	{
	    if (res == NULL)
		return new HSV32Image(width, height, frame_id, frame_time_in_ms, data);
	    else 
	    {
		// just copy over our variables
		memcpy(res, this, sizeof(*this));
		res->own_data = false;
		return res;
	    }
	}
    
// just treat as RGB for now.
//    long display(long glw = NULLWIN);
//    RGB32Image *rgb_read_image(RGB32Image *res);
//    void save_pnm(char *filename);
    
    ////  "ColourFilter" to grey --- adapted from RGB32Image class by nts Apr 2001
    Image *ColourFilter (Image *res = NULL,
			 ColourFilteringMethod method
			 = CF_METHOD_Y_709);
    
    ////  "ColourDistance" to grey --- added by nts May 2001
    Image *ColourDistance (Image *res = NULL,
			   ColourDistanceMethod method
			   = CD_METHOD_V_WEIGHTED_HS);
    
    Image *difference(Image *image2, Image *res = NULL);
    Image *difference(Grey8Image *image2, Image *res = NULL);
    Image *threshold(unsigned int threshold, Image *res = NULL,
		     unsigned int *no_marked = NULL);
    Image *resample(int xstep, int ystep, Image *resampled = NULL);
    Image *mask(Image *mask, Image *res);
    
    void clear(HSV32pixel fill_colour);
    void clear_border();
    
    void draw_rectangle(int xmin, int xmax, int ymin, int ymax, int val);
    
    // set up 64 MB lookup table for to_HSV32_using_lookup(RGB32pixel *) :
    void setup_HSV32_to_RGB32_lookup_table();
    
    static inline unsigned char colour_distance (
	const RGB32pixel RGBpixel1, const RGB32pixel RGBpixel2)
	{
	    HSV32pixel HSVpixel1 = to_HSV32(&RGBpixel1);
	    HSV32pixel HSVpixel2 = to_HSV32(&RGBpixel2);
	    
	    register unsigned int H_diff;
	    
	    if (HSVpixel1.H < HSVpixel2.H)
		H_diff = HSVpixel2.H - HSVpixel1.H;
	    else 
		H_diff = HSVpixel1.H - HSVpixel2.H;
	    
	    if (H_diff > 127)  // must wrap-around to get angular difference
		H_diff = 255 - H_diff;
	    // NB H_diff is in [0; 127] now
	    
	    return (H_diff *
		    ABS(HSVpixel1.S - HSVpixel2.S) *
		    ABS(HSVpixel1.V - HSVpixel2.V)) / 32768;
	}

private:
    // private variables and helpers
    static const char *RGB32_to_HSV32_lookup_table_filename;
    static HSV32pixel *RGB32_to_HSV32_lookup_table;
    
    int compar_char(const void *c1, const void *c2);
    unsigned int to_uint(const HSV32pixel colour);    
    
};

} // namespace ReadingPeopleTracker

#endif
