#
#  generic definitions and variables used in all `Makefile's in our code.
#
#  here you will find compiler settings and options, libraries etc
#

# Flag to avoid including this file more than once.

ifndef GENERIC_MAKE
GENERIC_MAKE = 1

#
# some general settings: compiler, variables, helpers etc
#
# important: set VPATH to parent directory as we compile in a subdirectory eg, Linux2
VPATH = ..

SHELL = /bin/sh

ifndef CPLUS
CPLUS = g++
endif

CPLUSPLUS = $(CPLUS)

# Determine the Operating System (OS); under Linux, the following will give "Linux" and "Linux2"
BASIC_OS =  $(shell echo `uname -s`)
OS = $(shell echo `uname -s``uname -r|cut -c1-1`)

# Determine gcc major version (first number output by "gcc --version"): 3 or 4
GCC_MAJOR_VERSION =  $(shell echo `gcc --version | tr '. ' '\012\012' | egrep '3|4' | head -1`)

#
# the link ~/advisor should point to /advisor/nils or ...
#   --- where our individual copies of The Code can be found
#
ACTUAL_HOME = $(shell echo `tcsh -cf 'cd ~/advisor; pwd'`)
SRC_HOME = ${ACTUAL_HOME}/source
BIN_HOME = ${ACTUAL_HOME}/bin

#
# path to Ygl installation, if used (see below)
#
YGL_LIBDIR = ${ACTUAL_HOME}/Ygl

#
# path to XML library, i.e. Xerces-C (see below)
#
# # # XML_LIBDIR = /opt/xml4c
XML_LIBDIR = ${ACTUAL_HOME}/xerces-c

#
# path to Motif-compatible library
#
# MOTIF_LIBDIR = /usr/X11R6/LessTif/Motif2.0
MOTIF_LIBDIR = ${ACTUAL_HOME}/openmotif

#
# Whether to display images.  Enable "NO_DISPLAY = 1" for double speed but no display output
#
NO_DISPLAY = 0
#NO_DISPLAY = 1

# whether to use single precision float type for maths, instead of double (off for now)
USE_FLOAT = 0
# USE_FLOAT = 1

#
# Compiler flags to generate debug output (-DDEBUG) or not.
# NB: do define NDEBUG (-DNDEBUG) for releases to disable assertions!
#

# development/debugging flags:
CDEBUG = -DDEBUG -g -O -march=pentiumpro

# profiling flags:
# CDEBUG = -DNDEBUG -g -pg -O3 -march=pentiumpro

# release flags:
# CDEBUG = -DNDEBUG -g -O3 -march=pentiumpro -fomit-frame-pointer


#
# These compiler options determing how many warnings are given
#

# a large range of warnings --- standard, strongly recommended
DEFINES = -Wimplicit -Wreturn-type -Wformat -Wparentheses -Wchar-subscripts -Wswitch -Wpointer-arith -Wconversion -Wcast-align -Wuninitialized
#
# additional warnings (not always useful but recommended before releases)
# DEFINES += -Winline -Wshadow -Wcast-qual -Wunused -Wno-non-virtual-dtor
# DEFINES += -pedantic -Winline -Wshadow -Wcast-qual -Wall -Wunused -Wno-non-virtual-dtor

# important: switch on RTTI (just in case it is disabled by default)
DEFINES += -frtti

# whether to use single precision maths (float type) instead of double precision
ifeq ($(USE_FLOAT),1)
DEFINES += -DUSE_FLOAT=1
endif

# check whether to disable all graphical output...
ifeq ($(NO_DISPLAY),1)
# yes: don't display anything.  However, still use GL image addressing etc
DEFINES += -DNO_DISPLAY -DUSE_GL
endif

# endianness (MSB || LSB first):
ifeq ($(BASIC_OS),Linux)
DEFINES += -DLSB_FIRST
else
DEFINES += -DMSB_FIRST
endif

# whether our compiler has min, max functions (ISO C99 defines them in stl_algobase)
DEFINES += -DHAVE_MIN_MAX

# whether our compiler has the ISO C99 <sstream> header
DEFINES += -DHAVE_SSTREAM

#
# Graphics libraries: for image display and graphical user interface
#

# Under Linux:
# There is a 'GL' library (YGL) that emulates the SGI's GL using X11. This
# library is quite complete, but does not have the functions to display b-splines
# (which e.g. VOGL has).  If 'GL' and 'VOGL' are used at the same time, some
# problems of multiple defined functions appear while linking.  As 'YGL' is
# more complete than 'VOGL' in some aspects (devices, events, ...), this
# can be solved using 'YGL', and instead of display curves, display 
# polygons (flag DISPLAY_POLY).
ifeq ($(NO_DISPLAY),0)
ifeq ($(BASIC_OS),Linux)
DEFINES += -DUSE_GL
GL_INCLUDES = -I$(YGL_LIBDIR)/gl -I$(YGL_LIBDIR)
GL_LIBS = -L$(YGL_LIBDIR) -lYgl -Wl,-rpath -Wl,$(YGL_LIBDIR)
endif
endif

#
# Motif-compatible libraries (e.g. openmotif, lesstif)
#
MOTIF_INCLUDES = -I$(MOTIF_LIBDIR)/include -I/usr/X11R6/include -I/usr/include/X11
MOTIF_LIBS = -L$(MOTIF_LIBDIR)/lib -L/usr/X11R6/lib -L/usr/lib/X11R6 -lXm -lXt -lX11 -lXext -lXmu -lXpm

#
# Language level defines
#
# defining __STRICT_ANSI__ disables all non-ANSI C compiler extensions
# defining _ISOC99_SOURCE _ISOC9X_SOURCE enables ISO/IEC 9899:1999(E) ("ISO C99")
#          as amended by ISO/IEC 9899:1999/COR1:2001(E) and should always be used
#          it gives us, among other things, math inlines like fmin(...) and fmax(...)
# defining _BSD_SOURCE adds BSD 4.3 functions, like usleep(...) and strcasecmp(...)
# defining _SVID_SOURCE gives System V style 48-bit random number generator
#          functions void srand48(long int seed_val), double drand48(void) which
#          are not always needed and will only be used if you define _SVID_SOURCE
ifeq ($(BASIC_OS),Linux)
DEFINES += -DDISPLAY_POLY
# NB enable _BSD_SOURCE on Linux systems during development to get timer functions (displaying...)
DEFINES += -D__STRICT_ANSI__ -D_ISOC99_SOURCE=1 -D_ISOC9X_SOURCE=1 -D_BSD_SOURCE=1 -DHAVE_POSIX_PIPES
# DEFINES += -D__STRICT_ANSI__ -D_ISOC99_SOURCE=1 -D_ISOC9X_SOURCE=1
endif 

#
#  thread handling: we now use the ANSI/IEEE POSIX 1003.1c - 1995 compliant pthread library:
#
ifeq ($(BASIC_OS),Linux)
  DEFINES += -D_REENTRANT
  SYS_LIBS += -lpthread
endif

###############   System specific definitions.

#
#  Math libraries: choose either NAG or BLAS/LAPACK
#

#
# --- this is for NAG.  we use mark 15 at the moment (2002)
#
# MATH_LIBS = -L/usr/lib/nag \
# 	  -lf01aaf -lf02abf -lf02agf -lf01qcf -lf01qef -lf02wef -lf04jgf \
# 	  -Wl,-rpath -Wl,/usr/lib/nag \
# 	  -lblas -lg2c -lm

#
# --- this is for BLAS/LAPACK, we use version 3.0 from http://www.netlib.org/lapack/
#
ifeq ($(GCC_MAJOR_VERSION),3)
  # use this first option for gcc version 3.x
  MATH_LIBS = -llapack -lblas -lm -lg2c
else
  # use this second option for gcc version 4.x
  MATH_LIBS = -llapack -lblas -lm -lgfortran
endif

DEFINES += -DUSE_LAPACK


#
#  XML library: we use Xerces-C, available from http://xml.apache.org/xerces-c/
#
#  In the past we used XML4C from http://www.alphaworks.ibm.com/ but it is discontinued
# XML_LIBS = -lxerces-c -licu-uc -licudata -Wl,-rpath -Wl,$(XML_LIBDIR)/lib	  
#
XML_LIBS = -L$(XML_LIBDIR)/lib -lxerces-c -Wl,-rpath -Wl,$(XML_LIBDIR)/lib	  
XML_INCLUDES = -I$(XML_LIBDIR)/include -I$(XML_LIBDIR)/include/xercesc

#
# IEEE1394 (aka FireWire or i.Link) libraries 
#
IEEE1394_LIBS =  -lavc1394 -lraw1394 -lrom1394 -ldv
IEEE1394_INCLUDES = -I/usr/include/libavc1394 -I/usr/include/libraw1394 -I/usr/include/libdv

#
# OS specific defines which are queried within The Code
#
ifeq ($(BASIC_OS),Linux)
# GNU g++ has machine-dependent optimizations for functions defined in
# string.h, most probably using inline assembler code; these might be quite
# expensive since the code size can increase significantly.  These
# optimizations are not used unless the symbol __USE_STRING_INLINES is defined
# before including the <cstring> header.  We enable these since we optimize
# for speed (not size).
DEFINES += -D__USE_STRING_INLINES
DEFINES += -DLINUX
endif

#
# generic variables for all runs of the compiler
#

GENERIC_INCLUDES =  -I.. \
	-I$(SRC_HOME)/data \
	-I$(SRC_HOME)/imgsrc \
	-I$(SRC_HOME)/tracking \
	-I$(SRC_HOME)/matrix \
	-I$(SRC_HOME)/PCA \
	-I$(SRC_HOME)/utils \
	-I$(SRC_HOME)/interface \
	-I$(SRC_HOME)/xml \
	$(XML_INCLUDES) \
	$(IEEE1394_INCLUDES) \
	$(MOTIF_INCLUDES) \
	$(GL_INCLUDES)


GENERIC_LDFLAGS = -L../$(OS) \
	  -L$(SRC_HOME)/data/$(OS) \
	  -L$(SRC_HOME)/imgsrc/$(OS) \
	  -L$(SRC_HOME)/tracking/$(OS) \
	  -L$(SRC_HOME)/matrix/$(OS) \
	  -L$(SRC_HOME)/PCA/$(OS) \
	  -L$(SRC_HOME)/utils/$(OS) \
	  -L$(SRC_HOME)/interface/$(OS) \
	  -L$(SRC_HOME)/xml/$(OS) \
	  $(XML_LIBS) \
	  $(MOTIF_LIBS) \
	  $(MATH_LIBS) \
	  $(SYS_LIBS) \
	  $(XML_LIBS) \
	  $(GL_LIBS)  \
	  $(IEEE1394_LIBS)

#
# Finally, define the generic compiler command with the options defined above
#
C++ = $(CPLUSPLUS) $(DEFINES) $(GENERIC_INCLUDES) $(GL_INCLUDES)

#
# do this for profiling on a GNU system (very easy):    nts May 2001
#
#   1 - compile with "-g -pg -a" flags (-pg is most important)
#   2 - run program (give -l last_frame_no because program HAS to exit nicely)
#       this creates statistical data file gmon.out upon exiting
#   3 - run gprof on the executable (reads and uses created gmon.out file)
#       this creates the profiling data on stdout (so say "> filename")
#   4 - finished.  view file in favourite viewer/editor, use kprof or similar
#

endif
