/*
 * Region.h
 *
 * a Region is a rectangular subimage and 
 * in general will consist of one connected blob
 * 
 * class created 2/3/93
 * author: Adam Baumberg
 * 
 */

#ifndef __REGION_H__
#define __REGION_H__

#include <string.h>      // for memmove(), memset()

#include "Observation.h"
#include "BoundaryPoints.h"
#include "tracker_defines_types_and_helpers.h"

namespace ReadingPeopleTracker
{

class RegionSet;
class Profile;
class RegionTracker;
class Image;
class PointVector;

class Region : public Observation
{
private:
    RegionTracker *tracker;

    static const unsigned int start_list[8];
    static const unsigned int end_list[8];
    static const Point2 directions[8];

public:
    
    // value of origin in previous frame (for movement estimation)
    Point2 old_origin;

    // shape of Region
    Image *region_img;
    
    // additional region data from XML region description and whether it is available
    xml_blob_data_t xml_blob_data;
    bool xml_blob_data_available;

    // direction of motion of blob 
    Point2 direction;
    
    // status of temporal background integration (using MultiBackgroundSource)
    bool incorporated_into_background;

    // misc
    realno inv_grad;
    BoundaryPoints *region_boundary;
    
public:
    
    friend class RegionSet;
    friend class RegionTracker;
    
    Region(int x1, int x2, int y1, int y2, Image *img = NULL)
	{ 
	    xlo = x1; xhi = x2; ylo = y1; yhi = y2;
	    width =  xhi - xlo + 1;
	    height = yhi - ylo + 1;
	    origin.x = (xhi + xlo) / 2.0;
	    origin.y = (yhi + ylo) / 2.0;
	    old_origin = origin;
	    region_img = img;
	    region_boundary = NULL;
	    direction = Point2(0,0);
	    memset((void *) &xml_blob_data, 0, sizeof(xml_blob_data));  // initialise with 0s
	    xml_blob_data_available = false;
	    incorporated_into_background = false;
	    inv_grad = 0;	    
	}
    Region() : Observation()
	{
	    old_origin.x = old_origin.y = 0;
	    region_img = NULL;
	    region_boundary = NULL;
	    memset((void *) &xml_blob_data, 0, sizeof(xml_blob_data));  // initialise with 0s
	    xml_blob_data_available = false;
	    direction.x = direction.y = 0;
	    inv_grad = 0;
	    incorporated_into_background = false;
	}
    ~Region();

    Region &operator= (const Region &original);

    // merge two regions
    void merge(Region *reg2);
    
    void draw_points();
    void draw_box();
    void draw_box_in_image(Image *canvas,
			   unsigned int line_width = 2);
    
    void smooth_boundary(realno sd = 1.0, int win_size = 8) 
	{
	    region_boundary->gauss_smooth(sd, win_size); 
	}
    
    void get_best_line() 
	{
	    region_boundary->find_best_line(inv_grad, origin.x, origin.y);
	}
    
    void trace();
    
    void find_ends() 
	{ 
	    region_boundary->find_ends(inv_grad, origin.x, origin.y);
	}
    
    bool to_spline(PointVector *res)
	{ 
	    return region_boundary->convert_to_spline(res);
	}
    
    void recenter()
	{ 
	    region_boundary->recenter(origin.x, origin.y);
	}
     
    Profile *to_profile();
    
private:
    // private helper
    static int combine(void *pix1, void *pix2);
    static int and_pixels (void *pix1, void *pix2);
    static int or_pixels (void *pix1, void *pix2);

};

istream &operator>> (istream &strmin, Region &region);
ostream &operator<< (ostream &strmout, const Region &region);

} // namespace ReadingPeopleTracker

#endif
