/*
 * Profile.h
 * 
 * a record class for storing the Profile data
 * of an object
 *
 * AMB 27/4/93
 */

#ifndef __PROFILE_H__
#define __PROFILE_H__

#include "Observation.h"
#include "BoundaryPoints.h"
#include "EnvParameter.h"
#include "KalmanFilterOneD.h"
#include "KalmanFilterTwoD.h"

namespace ReadingPeopleTracker
{

class Image;
class ColourBlob;
// class ActiveShapeTracker;
class SplineWeights;

#define spline point_data()


class Profile : public PointVector, public Observation
{
    friend class ProfileSet;
    friend std::istream &operator>> (std::istream &in_strm, Profile &profile);
    friend std::ostream &operator<< (std::ostream &out_strm, const Profile &profile);

private:
    static const realno LARGE;
  
//     ActiveShapeTracker *tracker;

    // inverse of the gradient of line of best fit
    realno inv_grad;
    
    // used in tracking module
    Point2 &direction;

    // simple flag for ProfileSequence only (used to use the removed ref_no)
    int PCA_flag;

    SplineWeights *spline_weights;

public:    

    static unsigned int NO_CONTROL_POINTS;

////////////// tracking data

    // tracking status
    bool filters_initialised;
    
    realno a_x, a_y;			// alignment parameters
    unsigned int no_shape_parameters;		// equivalent to model depth!
    StaticKalmanTwoD *a_filter;         // alignment (scale and rotation) filter
    FilterOneD **filters;		// shape filters
    
    FilterTwoD *pos_filter;       // origin filter
    
    StaticKalmanOneD *s_filter;   // scale filter or NULL
    StaticKalmanOneD *th_filter;  // rotation filter or NULL
    
    realno track_error;		// the error on last active fit
    realno fitness;             // track fitness (between 0 and 1, 1 == perfect fit)
    
    ColourBlob *colour_info;    // array for tracking coloured edges
    int no_cblobs;              // number of ColourBlobs in the above array
    
// constructor / destructors
    
public:
    Profile();
    Profile(realno ox, realno oy, realno wdth, realno hgt);
    Profile(Profile &original);
    Profile &operator= (const Profile &original);
    Profile(Point2 &p_o, realno &p_w, realno &p_h, realno &p_grad,
	    PointVector &pnts, Point2 &direction);
    ~Profile();

// data access functions
    
    realno *get_shape(int i);
    realno get_data(int i); 
    void set_data(int i, realno val); 
    int get_data_size();
//     void set_tracker(ActiveShapeTracker *the_tracker)
// 	{
// 	    tracker = the_tracker;
// 	}
    
    
// utility functions
    
    // update Observation variables like xlo, xhi, width etc
    void update_size_variables();

// commented out: use update_variables() and then access Observation::xlo etc
//    void get_box(int &ixmin, int &ixmax, int &iymin, int &iymax);
//     void get_y_bounds(realno &ymin, realno &ymax);
//    realno get_size();  // returns 0.5 * (height + width)
   
    void draw(int col1, int col2, int col3);
    void draw_filled();
    void draw();
    
    void normalise(bool flag = true);
    // scale shape
    void scale(realno );
    // scale_coords -- scale up coordinates
    void scale_coords(realno );
    
    void unnormalise()
	{
	    normalise(false);
	}
    
    void map(const realno &ax, const realno &ay, Profile *result = NULL);
    
    void align_to(Profile *mean, Profile *result,
		  realno &s, realno &th, realno *weights = NULL);
    
    void recenter(realno *weights = NULL);
    
    BoundaryPoints *to_points(BoundaryPoints *res = NULL);
    BoundaryPoints *sample_normals(BoundaryPoints *res = NULL);
    void get_normals(Profile &res);
    
    // draw at given origin
    void display_shape(realno ox, realno oy);
    
    // draw into an Image structure
    void draw_in_image(Image *canvas, Point2 origin, bool filled = true);
    void draw_in_image(Image *canvas, bool filled = true)
	{
	    draw_in_image(canvas, origin, filled);
	}
    
    // reflect about y-axis
    Profile *reflect(Profile *result = NULL);
    
    void points_to_spline();
    
    // things for tracked profiles
    void map_to_model_frame()
	{
	    realno det = a_x * a_x + a_y * a_y;
	    this->map(a_x / det, -a_y / det);
	}
    
    void interpolate(NagVector &coeff, NagVector &res);

    // get enclosing box
    // return a box that encloses the current shape 
    // taking into account the positional uncertainty
    void get_enclosing_box(int &xmin, int &xmax, int &ymin, int &ymax);

    void update_shape_filters();
    
    static inline void putback_string(istream &in_strm, char *str1)
	{
	    while ((*str1) != 0)
		in_strm.putback(*str1++);
	}
    
    // static variables for drawing/ output routines
    
    static bool draw_boxes;	// draw control points ?
    static bool draw_arrow;	// draw direction arrow ?
    static bool draw_in_colour;	// draw using colour ?
    static bool draw_spline;	// draw boundary with a curve ?
    static bool draw_rectangle;	// draw with a rectangle ?
    static bool draw_fill;	// draw filled 2D shape ?
    static bool draw_dashed;	// draw with dashed lines ?
    static bool draw_label;     // draw (text) label for track ?
    
    static bool output_points;	// output control point data ?
    
    static int draw_linewidth;	   // line width for drawing people
    static realno draw_box_size;   // size for control point boxes

private:
    
    // private helpers
    void setup_spline_weights();
    
};

ostream &operator<< (ostream &out_strm, const Profile &profile);
istream &operator>> (istream &in_strm, Profile &profile);  

} // namespace ReadingPeopleTracker

#endif
