///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Observation.h                                                            //
//                                                                           //
//  Base storage class for an observation: a Region, Profile etc             //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Tue Oct 30 14:37:42 GMT 2001                                 //
//  Revision  : 0.0 of Tue Oct 30 14:37:42 GMT 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef __OBSERVATION_H__
#define __OBSERVATION_H__

#include "Point2.h"
#include "tracker_defines_types_and_helpers.h"
#include <cassert>

namespace ReadingPeopleTracker
{

typedef enum { MEASUREMENT , PREDICTION , SPLITTING , OTHER } ObservationSourceType;

class Observation
{
public:

    // source of information: measurement or prediction
    ObservationSourceType source;
    
    bool is_visible;         // whether Observation can be seen in the current frame
    frame_time_t time_first_detected;  // time (in ms) Observation was first detected
    frame_time_t time_last_detected;   // time (in ms) Observation was last detected
    frame_id_t frame_first_detected;   // frame number Observation was first detected
    frame_id_t frame_last_detected;    // frame number Observation was last detected

    // bounding box
    int xlo, xhi, ylo, yhi;   // NB: numbers could be < 0 so do not use unsigned int
    
    // origin (NB: could be off-image, coordinates could be < 0)
    Point2 origin;
    
    // width and height
    realno width, height;
    
    // constructor: sets reasonable default values
    Observation(ObservationSourceType data_source = OTHER)
	{
	    source = data_source;
	    is_visible = true;
	    frame_first_detected = 0;
	    frame_last_detected = 0;
	    xlo = xhi = ylo = yhi = 0;
	    origin.x = origin.y = 0;
	    width = height = 1;
	}

    // destructor: virtual to make sure derived destructors are called
    virtual ~Observation() { };

    Observation &operator=(const Observation &original);


    // check class invariants in this class
    void check_class_invariants()
	{
	    // data consistency

	    assert (xlo < xhi);
	    assert (ylo < yhi);
	    
	    assert (abs (xhi - width - xlo) <= 1);
	    assert (abs (yhi - height - ylo) <= 1);
	    
	    assert (time_first_detected <= time_last_detected);

	    // Data for this one not available!  (In some parent parent parent class...)
	    // This assertion is actually checked elsewhere (some ...Tracker class(es))
	    //
	    // assert ((is_visible == false) ^
	    //         (frame_last_detected == current_frame_id for relevant Camera));

	    // If the following assertion fails handling of this situation needs
	    // to be implemented elsewhere
	    // This could go wrong but should not unless the program runs a few
	    // years or so and the frame numbers get wrapped around
	    assert (frame_first_detected <= frame_last_detected);

	    // range checking

	    // the following 2 are not necessarily implied by assertions given above
	    assert (width > 0);
	    assert (height > 0);
	    
#if (FRAME_ID_T_MAX - MAX_FRAME_ID) > 0
	    // this makes sense only if we do not use the full range of frame_id_t
	    assert (frame_first_detected <= MAX_FRAME_ID);
	    assert (frame_last_detected <= MAX_FRAME_ID);
#endif
	}

};

} // namespace ReadingPeopleTracker

#endif
