///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  HumanFeatures.cc                                                         //
//                                                                           //
//  Some definitions and structures for human features such as head etc      //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Tue Apr 17 15:57:52 BST 2001                                 //
//  Revision  : 0.1 of Thu Apr 19 12:20:49 BST 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "HumanFeatures.h"
#include "RGB32Image.h"       // for zooming in video image
#include "Grey8Image.h"       // for zooming in motion image
#include "HumanFeatureTracker.h"

namespace ReadingPeopleTracker
{

static const char *HumanFeatures_Revision = "@(#) HumanFeatures.cc, rev 0.1 of Thu Apr 19 12:20:49 BST 2001, Author Nils T Siebel, Copyright (c) 2001 The University of Reading";

// definition and initialisation of static member variables
bool HumanFeatures::draw_head_search_area = true;
bool HumanFeatures::draw_head_bbox = true;
bool HumanFeatures::draw_head_centre = true;
bool HumanFeatures::draw_shoulder_width = true;


HeadInfo &HeadInfo::operator= (HeadInfo &original)
{
    // copy over normal variables
    x = original.x;
    y = original.y;
    x_abs = original.x_abs;
    y_abs = original.y_abs;
    width = original.width;
    height = original.height;
    
    // instantiate new VerticalHistogram in order to copy *original.head_histogram
    head_histogram = new VerticalHistogram;
    *head_histogram = *original.head_histogram;

    return *this;
}

HumanFeatures &HumanFeatures::operator= (HumanFeatures &original)
{
    Observation::operator=(original);

    if (original.head != NULL)
    {
	// instantiate new HeadInfo in order to copy over *original.head
	head = new HeadInfo;
	*head = *original.head;  // using the operator defined above
    }
    else
	head = NULL;
    
    // (the other variables are static)

    return *this;
}

// draw all detected HumanFeatures into image
void HumanFeatures::draw()
{
#ifndef NO_DISPLAY
    draw_head();  // that's all we have for now...
#endif   // #ifndef NO_DISPLAY
}


void HumanFeatures::draw_head()
{
#ifndef NO_DISPLAY
    if (head == NULL)
	return;
    
    int zoom = Image::display_zoom->value;   // because recti and circ don't use zoom
    VerticalHistogram *histogram = head->head_histogram;
    
    linewidth(2);
    
    if (draw_head_search_area)
    {
#ifdef USE_GL
	// bbox of search area
	if (getplanes() == 8)   // CMap (grey?) image  (fi motion image)
	    color(30);
	else
	    if (getplanes() == 24)      // RGB mode  (fi video image)
		RGBcolor(127,32,32);
#endif

	recti(zoom*(histogram->xlo),
	      zoom*(histogram->ylo),
	      zoom*(histogram->xhi),
	      zoom*(histogram->yhi));
    }
    
    if (draw_head_bbox)
    {
#ifdef USE_GL
	// head bbox
	if (getplanes() == 8)   // CMap (grey?) image  (fi motion image)
	    color(50);
	else
	    if (getplanes() == 24)      // RGB mode  (fi video image)
		RGBcolor(255,10,127);
#endif
	
	recti(zoom*(head->x_abs - head->width / 2),
	      zoom*(head->y_abs - head->height / 2),
	      zoom*(head->x_abs + head->width / 2),
	      zoom*(head->y_abs + head->height / 2));
    }
    
    if (draw_head_centre)
    {
#ifdef USE_GL
	// head centre: mark with a circle
	if (getplanes() == 8)   // CMap (grey?) image  (fi motion image)
	    color(50);
	else
	    if (getplanes() == 24)      // RGB mode  (fi video image)
		RGBcolor(255,10,127);
#endif

	circ(zoom*(head->x_abs),
	     zoom*(head->y_abs), zoom + 1);
    }
    
    if (draw_shoulder_width)
    {
#ifdef USE_GL
	// assumed shoulder width (from head search histogram)
	if (getplanes() == 8)   // CMap (grey?) image  (fi motion image)
	    color(50);
	else
	    if (getplanes() == 24)      // RGB mode  (fi video image)
		RGBcolor(255,10,127);
#endif

	move2i(zoom*(histogram->xlo + histogram->leftmost),
	       zoom*(histogram->ylo + head->y - head->height / 2 - 1));
	draw2i(zoom*(histogram->xlo + histogram->leftmost),
	       zoom*(histogram->ylo + 1));
	
	move2i(zoom*(histogram->xlo + histogram->rightmost),
	       zoom*(histogram->ylo + head->y - head->height / 2 - 1));
	draw2i(zoom*(histogram->xlo + histogram->rightmost),
	       zoom*(histogram->ylo + 1));
    }	
#endif   // #ifndef NO_DISPLAY
}

// write out HeadInfo structure
ostream &operator << (ostream &target, const HeadInfo head)
{
    target << "Head Info: head at (" << head.x << "," << head.y << ") relative, ("
	   << head.head_histogram->xlo + head.x << ","
	   << head.head_histogram->ylo+head.y << ") absolute, size "
	   << head.width << "x" << head.height << " pixels " << endl;
    return target;
}

ostream &operator << (ostream &target, const HumanFeatures features)
{
    target << "Features: " << features.head;
    
    // more features to be written out here...
    // target << "Features: " << features.body;  \ETC

    return target;
}

// FIXME: Dummy istream operator.  we need it for listimplement(HumanFeatures)
//        but until we actually use it I will leave it empty --- nts Apr 2001.
istream &operator >> (istream &source, const HumanFeatures features)
{
    return source;
};

} // namespace ReadingPeopleTracker
